# coding: utf-8
from __future__ import print_function

import collections
import copy
import itertools
import logging

import tornado.gen
import tornado.httpclient

from . import application
from . import rpc
from . import settings
from . import ticker
from . import utils
from . import _unistat_callback

from infra.netmon.agent.idl import common_pb2 as common


class RequestSizeLimiter(object):

    def __init__(self, maxlen):
        self._limits = tuple(min(x, maxlen) for x in (1, 2, 3, maxlen))
        self._current_limit_idx = len(self._limits) - 1

    def increase_limit(self):
        if self._current_limit_idx + 1 < len(self._limits):
            self._current_limit_idx += 1

    def reduce_limit(self):
        if self._current_limit_idx > 0:
            self._current_limit_idx -= 1

    def get_limit(self):
        return self._limits[self._current_limit_idx]


class SenderService(application.AppMixin, application.Service):
    """Simply send reports into netmon."""

    def __init__(self, report_ttl_sec=240, maxlen=5):
        self._report_ttl_usec = report_ttl_sec * utils.US

        self._queues = {}
        self._request_limiters = {}

        self._queues[common.REGULAR_PROBE] = collections.deque(maxlen=maxlen)
        self._request_limiters[common.REGULAR_PROBE] = RequestSizeLimiter(maxlen)

        if settings.current().noc_sla_urls:
            self._queues[common.NOC_SLA_PROBE] = collections.deque(maxlen=maxlen)
            self._request_limiters[common.NOC_SLA_PROBE] = RequestSizeLimiter(maxlen)

        self._loop = ticker.LoopingCall(
            "sender",
            self._try_send_all,
            min(10, max(settings.current().check_interval)),
            round_by_interval=True
        )

    def _get_backends_by_type(self, probe_type):
        if probe_type == common.NOC_SLA_PROBE:
            return self._app[application.BackendMaintainerService].get_noc_sla_backends()
        else:
            return self._app[application.BackendMaintainerService].get_prod_backends()

    def _remove_stale_probes(self, queue):
        now = utils.timestamp()
        while queue and now - queue[0][0].Generated > self._report_ttl_usec:
            queue.popleft()

    @tornado.gen.coroutine
    def _try_send(self, probe_type, active_queue):
        self._remove_stale_probes(active_queue)
        limit = min(self._request_limiters[probe_type].get_limit(),
                    len(active_queue))
        if not limit:
            return

        # In order to keep report queue ordered by TProbeReport.Generated field,
        # we divide it into 3 parts. After send_reports() the parts contain:
        #
        # saved_queue  - previous state of active_queue
        # report_list  - reports we passed to send_reports()
        # active_queue - reports enqueued while we waited for send_reports()
        #
        # After send_reports() we restore active_queue by prepending the parts:
        # On success: active_queue = saved_queue + active_queue
        # On error:   active_queue = saved_queue + report_list + active_queue

        saved_queue = copy.copy(active_queue)
        active_queue.clear()
        report_list = [saved_queue.pop() for idx in xrange(limit)]

        send_reports_wrapper = utils.suppress_http_errors(self._app[rpc.RpcClient].send_reports)
        response_codes = yield {
            url: send_reports_wrapper(
                list(itertools.chain.from_iterable(report_list)),
                url
            )
            for url in self._get_backends_by_type(probe_type)
        }
        logging.debug("Got response codes %s", response_codes)
        success_count = sum(code == 200 for code in response_codes.itervalues())
        if success_count > 0:
            self._request_limiters[probe_type].increase_limit()
        else:
            logging.warning("Reducing report count in requests for probe type %s",
                            probe_type)
            self._request_limiters[probe_type].reduce_limit()
            _unistat_callback.push_signal(_unistat_callback.ReportsThrottled, 1.0)
            slots = active_queue.maxlen - len(active_queue)
            active_queue.extendleft(report_list[:slots])

        while saved_queue and len(active_queue) < active_queue.maxlen:
            active_queue.appendleft(saved_queue.pop())

    @tornado.gen.coroutine
    def _try_send_all(self):
        for probe_type, queue in self._queues.iteritems():
            self._try_send(probe_type, queue)

    def enqueue(self, report_list):
        """Add elements into send queue. If there are too many elements in the queue,
        oldest one will be removed.
        """
        for report_type, queue in self._queues.iteritems():
            # NETMON-580: all reports from the same enqueue() call should be
            # sent together, so we append the whole list.
            reports = [x for x in report_list if x.Type == report_type]
            if reports:
                queue.append(reports)

    @tornado.gen.coroutine
    def cancel(self):
        yield self._loop.cancel()
