# coding: utf-8
# pylint: disable=invalid-name
from __future__ import print_function

import os
import collections
import socket

import yaml
import pathlib2

import _netmon

from . import utils

from infra.netmon.agent.idl import common_pb2

identity = lambda x: x

to_timeval = lambda x: int(x * 1000)
from_timeval = lambda x: x / 1000.0

# upper() and lower() for backward compatibility
to_protocol = lambda x: tuple(utils.STRING_TO_PROTOCOL[p.upper()] for p in x)
from_protocol = lambda x: tuple(utils.PROTOCOL_TO_STRING[p].lower() for p in x)

to_networks = lambda x: None if "any" in x else tuple(x)
from_networks = lambda x: ("any",) if x is None else tuple(x)

to_url_list = lambda x: tuple(x.split(',')) if x else ()
from_url_list = lambda x: ','.join(x)

Mapper = collections.namedtuple("Mapper", ("name", "alias", "load", "dump"))
MapperShortcut = lambda name: Mapper(name, name, identity, identity)


def to_permutations(x):
    result = {}
    for first, second in x:
        for values in (result.setdefault(first, set()), result.setdefault(second, set())):
            values.add(first)
            values.add(second)
    return result


def from_permutations(x):
    return sorted({
        (min(first, second), max(first, second))
        for first, values in x.iteritems() for second in values
        if first != second
    })


class Settings(object):

    _instance = None

    MAPPERS = (
        MapperShortcut("debug"),
        MapperShortcut("freeze_topology"),

        MapperShortcut("user"),
        MapperShortcut("group"),

        MapperShortcut("stats_port"),
        Mapper("var_dir", "var_dir", pathlib2.Path, str),
        Mapper("pid_path", "pid_path", pathlib2.Path, str),
        Mapper("log_path", "log_path", pathlib2.Path, str),

        MapperShortcut("hostname"),
        MapperShortcut("netmon_url"),
        Mapper("noc_sla_urls", "noc_sla_url", to_url_list, from_url_list),
        Mapper("noc_sla_fallback_urls", "noc_sla_fallback_url", to_url_list, from_url_list),
        MapperShortcut("provisioning_url"),

        MapperShortcut("yp_sd_url"),
        MapperShortcut("yp_sd_cluster_name"),
        MapperShortcut("yp_sd_endpoint_set_id"),
        MapperShortcut("yp_sd_request_interval"),

        Mapper("echo_port_range", "echo_port_range",
               lambda x: tuple(xrange(*x)),
               lambda x: (x[0], x[-1] + 1) if x else None),
        Mapper("tcp_port_range", "tcp_port_range",
               lambda x: tuple(xrange(*x)),
               lambda x: (x[0], x[-1] + 1) if x else None),
        Mapper("check_interval", "check_interval", tuple, identity),
        MapperShortcut("packet_count"),
        MapperShortcut("packet_ttl"),
        Mapper("packet_timeout", "packet_timeout", to_timeval, from_timeval),
        Mapper("packet_delay", "packet_delay", to_timeval, from_timeval),
        Mapper("probe_start_delay", "probe_start_delay", to_timeval, from_timeval),

        MapperShortcut("diagnostic_packet_count"),
        MapperShortcut("diagnostic_packet_size"),
        MapperShortcut("diagnostic_packet_ttl"),
        Mapper("diagnostic_packet_delay", "diagnostic_packet_delay", to_timeval, from_timeval),

        MapperShortcut("max_targets"),
        MapperShortcut("max_scheduled_targets"),
        MapperShortcut("p2p_threshold"),

        Mapper("protocols", "protocols", to_protocol, from_protocol),
        Mapper("networks", "networks", to_networks, from_networks),
        Mapper("groups", "groups", tuple, identity),
        Mapper("vlans", "vlans", to_permutations, from_permutations),
        Mapper("vrfs", "vrfs", to_permutations, from_permutations),
        MapperShortcut("automatic_targets"),

        MapperShortcut("topology_ttl"),
        MapperShortcut("group_ttl"),

        Mapper("per_switch_target_count", "switch_targets", identity, identity),
        Mapper("per_queue_target_count", "queue_targets", identity, identity),
        Mapper("per_intra_datacenter_target_count", "inside_dc_targets", identity, identity),
        Mapper("per_inter_datacenter_target_count", "between_dc_targets", identity, identity),
        MapperShortcut("max_probability"),
        MapperShortcut("minimal_weight"),
        Mapper("traffic_classes", "traffic_class", tuple, identity),
        MapperShortcut("link_poller_src_ip"),
        MapperShortcut("link_poller_fb_src_ip"),
        MapperShortcut('link_poller_port'),

        MapperShortcut("use_topology_ips"),
        MapperShortcut("ignore_ipv4_dns_fails"),
        MapperShortcut("dns_resolve_ip4"),

        MapperShortcut("unistat_pusher"),

        MapperShortcut("uniform_probes"),
        MapperShortcut("link_poller"),
        MapperShortcut("allow_virtual"),

        MapperShortcut("use_skynet"),
    )

    CPP_MAPPERS = (
        MapperShortcut("check_full_tos"),
        MapperShortcut("fix_reply_tos"),
        MapperShortcut("udp_socket_count"),
        MapperShortcut("packet_size"),
        MapperShortcut("link_poller_packet_size"),
        MapperShortcut("udp_multiport_probes"),
        MapperShortcut("use_hw_timestamps")
    )

    def __init__(self):
        self._cpp_settings = _netmon.Settings()

        self._hostname = None
        self._config_path = None
        self._cli_kwargs = None

        self.debug = False
        self.freeze_topology = False

        self.user = "nobody"
        self.group = "nogroup"

        self.stats_port = None
        self.var_dir = pathlib2.Path(os.path.abspath(os.getcwd()))
        self.pid_path = pathlib2.Path("netmon-agent.pid")
        self.log_path = None

        self.netmon_url = "https://netmon.yandex-team.ru"
        self.noc_sla_urls = tuple()
        self.noc_sla_fallback_urls = tuple()
        self.provisioning_url = self.netmon_url

        self.yp_sd_url = None
        self.yp_sd_cluster_name = None
        self.yp_sd_endpoint_set_id = None
        self.yp_sd_request_interval = 60

        self.echo_port_range = tuple(xrange(11016, 11025))
        self.tcp_port_range = tuple(xrange(11016, 11025))
        self.check_interval = (30, 60)

        self.packet_count = 10
        self.packet_ttl = -1
        self.packet_timeout = 1000
        self.packet_delay = 100
        self.probe_start_delay = 1000

        self.diagnostic_packet_count = 500
        self.diagnostic_packet_size = 1500
        self.diagnostic_packet_ttl = -1
        self.diagnostic_packet_delay = 10

        self.max_targets = 128
        self.max_scheduled_targets = 512
        self.p2p_threshold = 32

        self.protocols = (common_pb2.ICMP, common_pb2.UDP)

        # value of None disables network checks
        self.networks = (utils.BACKBONE, utils.FASTBONE)

        self.groups = ()
        self.vlans = {}
        self.vrfs = {}
        self.automatic_targets = False

        self.topology_ttl = 3600
        self.group_ttl = 3600

        # inside switch
        self.per_switch_target_count = 16
        # inside queue
        self.per_queue_target_count = 8
        # inside dc
        self.per_intra_datacenter_target_count = 256
        # between dc
        self.per_inter_datacenter_target_count = 1024
        self.max_probability = 1.0
        self.minimal_weight = 0.1
        self.traffic_classes = (0,)
        self.link_poller_src_ip = None
        self.link_poller_fb_src_ip = None
        self.link_poller_port = 319  # PTP

        self.use_topology_ips = False
        self.ignore_ipv4_dns_fails = False
        self.dns_resolve_ip4 = False

        self.unistat_pusher = True

        self.uniform_probes = False
        self.link_poller = False
        self.allow_virtual = False

        self.use_skynet = True

    def __getattr__(self, name):
        """ Called when self.name is not a regular attribute """
        return getattr(self._cpp_settings, name)

    @property
    def hostname(self):
        # TODO: hostname may change
        if self._hostname is None:
            try:
                self._hostname = utils.detect_hostname()
            except socket.gaierror:
                # sometimes DNS don't work properly, let's use something
                self._hostname = socket.gethostname()
        return self._hostname

    @hostname.setter
    def hostname(self, value):
        self._hostname = value

    def _settings_obj_to_dict(self, obj, mappers):
        return {
            mapper.alias: (
                mapper.dump(getattr(obj, mapper.name))
                if getattr(obj, mapper.name, None) is not None
                else None
            )
            for mapper in mappers
        }

    def to_dict(self):
        result = self._settings_obj_to_dict(self, self.MAPPERS)
        result.update(self._settings_obj_to_dict(self._cpp_settings, self.CPP_MAPPERS))
        return result

    def _update_settings_obj_from_dict(self, obj, mappers, d):
        for mapper in mappers:
            value = d.get(mapper.alias, None)
            if value is not None:
                setattr(obj, mapper.name, mapper.load(value))

    def from_dict(self, d):
        self._update_settings_obj_from_dict(self, self.MAPPERS, d)
        self._update_settings_obj_from_dict(self._cpp_settings, self.CPP_MAPPERS, d)

    def from_cli(self, kwargs):
        self._config_path = kwargs.pop("config_path", None)
        self._cli_kwargs = kwargs
        self.load()

    def load(self):
        if self._cli_kwargs is not None:
            self.from_dict(self._cli_kwargs)

        # config should be applied after cli because cli have all default values
        if self._config_path is not None:
            with open(self._config_path, "rb") as stream:
                root = yaml.safe_load(stream)
                self.from_dict(root["subsections"]["netmon"]["subsections"]["agent"].get("config") or {})

    @classmethod
    def current(cls):
        if cls._instance is None:
            cls._instance = cls()
        return cls._instance


# shortcut
current = Settings.current
