# coding: utf-8
from __future__ import absolute_import

import sys
import collections
import logging
import random

import tornado.ioloop
import tornado.gen
import tornado.concurrent
import tornado.httpclient

from . import utils, settings, exceptions

# how much of the original interval we add with each failed iteration
INTERVAL_INCREASE_ON_FAIL = 0.25
# the actual value will be somewhere between 0.25-0.1 and 0.25+0.1
INTERVAL_INCREASE_MAX_JITTER = 0.1
# use this delay as minimal ceil in case of repeating error
INTERVAL_ON_ERROR = 600


class Interval(object):

    def __init__(self, interval):
        if isinstance(interval, (tuple, list)):
            self.start, self.stop = interval
        else:
            self.start = interval
            self.stop = interval

    def get(self):
        if self.start == self.stop:
            return self.start
        else:
            return random.randint(self.start, self.stop)


class Ticker(object):
    """Base class for things that do some stuff periodically"""

    def __init__(self, interval, start_delay=0, round_by_interval=False):
        self._loop = tornado.ioloop.IOLoop.current()
        self._interval = Interval(interval)
        self._round_by_interval = round_by_interval

        self._running = False
        self._canceled = False
        self._start_ts = None
        self._current_timeout = None
        self._waiters = collections.deque()
        self._errors_count = 0

        self._spin_deadline = None
        self._schedule_iteration(start_delay)

    def _component_name(self):
        return type(self).__name__.lower()

    def _schedule_iteration(self, delay):
        # don't need to schedule canceled or started ticker
        if self._canceled or self._running:
            return

        # sleep for time what left on this iteration
        self._current_timeout = self._loop.call_later(max(delay, 0), self._on_tick_start)

    def _on_tick_start(self):
        """Time is up, let's execute current tick."""
        self._start_ts = self._loop.time()
        self._current_timeout = None

        if not self._canceled:
            self._running = True
            self._spin_deadline = None

            try:
                future = tornado.gen.maybe_future(self.tick())
            except Exception:
                # tick may not return a future and can throw exceptions
                future = tornado.concurrent.Future()
                future.set_exc_info(sys.exc_info())

            self._loop.add_future(future, self._on_tick_done)

    def _on_tick_error(self, future):
        """Catch exceptions in tick and redirect them to logger."""
        exc = future.exception()
        if isinstance(future, tornado.concurrent.Future):
            exc_info = future.exc_info()
        else:
            exc_info = (type(exc),) + future.exception_info()

        interval_increase = INTERVAL_INCREASE_ON_FAIL
        interval_increase += random.uniform(-INTERVAL_INCREASE_MAX_JITTER, INTERVAL_INCREASE_MAX_JITTER)
        actual_interval = (2.0 * (1 + interval_increase)) ** (1 + self._errors_count)
        max_interval = max(self._interval.stop, INTERVAL_ON_ERROR)
        if actual_interval > max_interval:
            self._errors_count = 0
        else:
            self._errors_count += 1
        deadline = self._loop.time() + max(min(actual_interval, max_interval), 1)

        if isinstance(exc, exceptions.RetryError):
            logging.debug(
                "Retrying %s", self._component_name(), exc_info=exc_info)
        elif isinstance(exc, tornado.httpclient.HTTPError):
            logging.warning(
                "HTTP error happend at %s: %s",
                self._component_name(), str(exc), exc_info=exc_info)
        else:
            logging.error(
                "Unhandled error from %s: %s",
                self._component_name(), str(exc), exc_info=exc_info)

        return deadline

    def _on_tick_done(self, future):
        """Schedule execution of next tick."""
        self._running = False

        # use another deadline in case of error
        deadline = None
        exc = future.exception()
        if exc is None:
            self._errors_count = 0
            # check custom deadline in tick method return value
            res = future.result()
            if isinstance(res, int) and res > 0:
                if (res > self._start_ts):
                    # absolute time point
                    deadline = res
                else:
                    # timeout
                    deadline = res + self._start_ts
        else:
            deadline = self._on_tick_error(future)

        # print statistics
        session_time = self._loop.time() - self._start_ts
        logging.debug("Execution time of %s is %.3f seconds", self._component_name(), session_time)

        # compute delay until next run
        if deadline is None:
            if self._round_by_interval:
                deadline = utils.round_by_interval(
                    self._interval.stop, self._start_ts, settings.current().hostname)
            else:
                deadline = self._start_ts + self._interval.get()
        delay = max(0, deadline - self._loop.time())

        # maybe some one else (or maybe ticker itself) want to run us ASAP
        if self._spin_deadline is not None:
            spin_delay = self._spin_deadline - self._loop.time()
            delay = min(delay, spin_delay)

        if self._current_timeout is None:
            # sleep for time what left on this iteration
            self._schedule_iteration(delay)

        self._spin_deadline = None

        # dispatch waiters
        if exc is None or not isinstance(exc, exceptions.RetryError):
            while self._waiters:
                tornado.concurrent.chain_future(future, self._waiters.popleft())

    def _remove_timeout(self):
        if self._current_timeout is not None:
            self._loop.remove_timeout(self._current_timeout)
            self._current_timeout = None

    @tornado.gen.coroutine
    def cancel(self):
        """Cancel all subsequent ticks."""
        if not self._canceled:
            self._canceled = True
            self._remove_timeout()

            if self._running:
                logging.info("Waiting for %s", self._component_name())
                yield self.wait()

            yield tornado.gen.maybe_future(self.on_cancel())

            waiters, self._waiters = self._waiters, None
            while waiters:
                waiters.popleft().set_result(None)

    def spin(self, spin_delay=0.01):
        """Do next tick not later than spin_delay"""
        if self._canceled:
            return

        spin_deadline = self._loop.time() + spin_delay
        if self._spin_deadline is not None and self._spin_deadline < spin_deadline:
            # iteration already scheduled earlier
            return

        self._spin_deadline = spin_deadline
        self._remove_timeout()
        self._schedule_iteration(spin_delay)

    def wait(self):
        """Wait until current (or next, if no current exists) tick will be finished."""
        future = tornado.concurrent.Future()
        if self._waiters is not None:
            self._waiters.append(future)
        else:
            future.set_result(None)
        return future

    def tick(self):
        """Abstract method that should do some useful job."""
        raise NotImplementedError(self)

    def on_cancel(self):
        """Called when this ticker is canceled."""
        pass


class LoopingCall(Ticker):
    """
    Allows running a function at specifying interval
    but also allows specifying next run time via the usual tick method return value
    """

    def __init__(self, name, function, interval, start_delay=0, round_by_interval=False):
        self._name = name
        self._callback = function
        super(LoopingCall, self).__init__(interval, start_delay, round_by_interval)

    def _component_name(self):
        return self._name

    def tick(self):
        return self._callback()
