# coding: utf-8
from __future__ import print_function

import datetime
import logging
import collections

import click
import yaml
import tornado.gen
import tornado.httpclient
import tornado.locks
import tornado.process

import _netmon

from . import rpc
from . import utils
from . import application
from .settings import Settings


@tornado.gen.coroutine
def _spawn_process(cmd, timeout=30):
    process = tornado.process.Subprocess(cmd)

    try:
        yield tornado.gen.with_timeout(
            # 5 seconds for it to load the process and w/e
            datetime.timedelta(seconds=timeout + 5),
            process.wait_for_exit(raise_error=False)
        )
    except tornado.gen.TimeoutError:
        utils.kill_process(process.pid)
        raise

    if process.returncode != 0:
        raise Exception("Skynet returned non-zero code {}".format(process.returncode))


class TopologyResource(collections.namedtuple("TopologyResource", ("url", "skynet_id"))):

    @tornado.gen.coroutine
    def load_via_skynet(self, path, timeout=120):
        if not Settings.current().use_skynet:
            raise RuntimeError("Skynet usage is disabled")

        yield _spawn_process([
            "/skynet/tools/sky", "get", "-d", str(path.parent), "-u", "-t", str(timeout), self.skynet_id
        ], timeout=timeout)

        if not path.exists():
            raise Exception("File {} not found".format(path.name))

    @tornado.gen.coroutine
    def load_via_http(self, path, timeout=120):
        tmp_path = path.parent.joinpath("{}.tmp".format(path.name))
        with open(str(tmp_path), "wb") as stream:
            yield tornado.httpclient.AsyncHTTPClient().fetch(
                self.url,
                method="GET",
                connect_timeout=5,
                request_timeout=timeout,
                streaming_callback=stream.write
            )

        tmp_path.rename(path)
        if Settings.current().use_skynet:
            try:
                yield _spawn_process([
                    "/skynet/tools/sky", "share", "-d", str(path.parent), path.name
                ])
            except Exception as exc:
                logging.warning("Can't reshare %s: %s", path.name, exc)


@tornado.gen.coroutine
def _get_latest_resource(app):
    reply = yield app[rpc.RpcClient].provisioning_info()

    logging.info("Latests resource skynet id is %s", reply.TopologyInfo.SkynetId)

    raise tornado.gen.Return(TopologyResource(
        reply.TopologyInfo.PlainMDSLink,
        reply.TopologyInfo.SkynetId
    ))


@tornado.gen.coroutine
def _download_topology(path, app):
    resource = yield _get_latest_resource(app)

    with utils.do_backup(path):
        if Settings.current().use_skynet:
            try:
                yield resource.load_via_skynet(path)
            except Exception:
                logging.exception('Unable to download hosts data from copier, try to download it from MDS:')
            else:
                return

        yield resource.load_via_http(path)


def get_path():
    return Settings.current().var_dir.joinpath("topology.msgpack.gz")


class TopologyIterable:
    class _Interface:
        pass

    def __init__(self, path, hostname):
        self._tree = _netmon.TopologyTree(str(path), hostname)
        self._items = []

    def __iter__(self):
        return self

    def next(self):
        while len(self._items) == 0 and self._tree.next():
            for iface in self._tree.interfaces():

                # The problem is that NPyBind implements move semantic
                # for setters/getters by default,
                # which is not compatible with our case
                #
                # Underlying cpp implementation returns Interface()
                # objects which contain references to internal memory
                # of single mutable msgpack object hidden deep inside.
                #
                # So let's just internalize object right here
                # since potential consumers (e.g. TargetTree)
                # can also store references to Interface() fields.

                copy = self._Interface()

                copy.name = str(iface.name)
                copy.host = str(iface.host)
                copy.switch = str(iface.switch)
                copy.queue = str(iface.queue)
                copy.datacenter = str(iface.datacenter)
                copy.network_type = str(iface.network_type) if iface.network_type else None
                copy.vrf = str(iface.vrf) if iface.vrf else None
                copy.vlan = iface.vlan
                copy.is_virtual = iface.is_virtual
                copy.ipv4_address = str(iface.ipv4_address) if iface.ipv4_address else None
                copy.ipv6_address = str(iface.ipv6_address) if iface.ipv6_address else None

                self._items.append(copy)

        if self._items:
            return self._items.pop(0)

        raise StopIteration()

    def local_interfaces(self):
        return self._tree.local_interfaces()

    def interfaces(self):
        self._items = []
        self._tree.reset()
        return self


topology_update_lock = tornado.locks.Lock()


@tornado.gen.coroutine
def tree(app):
    path = get_path()
    current_settings = Settings.current()
    hostname = current_settings.hostname

    with (yield topology_update_lock.acquire()):
        if utils.should_file_be_updated(current_settings.topology_ttl, path, hostname) and not current_settings.freeze_topology:
            try:
                yield _download_topology(path, app)
            except Exception:
                logging.exception('Unable to download hosts data from MDS, let\'s use old one:')

    raise tornado.gen.Return(TopologyIterable(str(path), hostname))


class HostInfoCommand(object):

    def __init__(self, settings, fqdn=None, dc=None, queue=None, switch=None, vlan=None, vrf=None, limit=10):
        self._app = application.Application()
        self._app.register(rpc.RpcClient())
        self._fqdn = fqdn
        self._dc = dc
        self._queue = queue
        self._switch = switch
        self._vlan = vlan
        self._vrf = vrf
        self._limit = limit

        if all(x is None for x in (self._fqdn, self._dc, self._queue, self._switch, self._vlan, self._vrf)):
            self._fqdn = settings.hostname

    def _is_needed(self, iface):
        return (
            (
                self._fqdn is not None
                and (iface.name == self._fqdn or iface.host == self._fqdn)
            )
            or (self._dc is not None and iface.datacenter == self._dc)
            or (self._queue is not None and iface.queue == self._queue)
            or (self._switch is not None and iface.switch == self._switch)
            or (self._vlan is not None and iface.vlan is not None and str(iface.vlan) == self._vlan)
            or (self._vrf is not None and iface.vrf is not None and iface.vrf == self._vrf)
            or False
        )

    def start(self):
        topology_tree = self._app.run_sync(lambda: tree(self._app))
        for iface in topology_tree.interfaces():
            if self._is_needed(iface):
                click.echo(yaml.safe_dump({
                    "name": iface.name,
                    "host": iface.host,
                    "switch": iface.switch,
                    "queue": iface.queue,
                    "datacenter": iface.datacenter,
                    "network_type": iface.network_type,
                    "vlan": iface.vlan,
                    "vrf": iface.vrf,
                    "is_virtual": iface.is_virtual,
                }))
