# coding: utf-8
"""This module should implement different logic that will convert hosts
from topology into proper target set.
"""
from __future__ import print_function

from . import interfaces
from . import selector
from . import settings
from . import utils

from infra.netmon.agent.idl import common_pb2 as common


class Transformer(object):

    @staticmethod
    def _interface_is_allowed(iface):
        return not interfaces.is_mtn_vlan(iface.vlan)

    @staticmethod
    def _are_interfaces_compatible(source, target):
        if target.network_type != source.network_type:
            return False
        elif source.vrf is not None and target.vrf is not None:
            return source.vrf == target.vrf
        elif source.vlan is not None and target.vlan is not None:
            return source.vlan == target.vlan
        else:
            # assume by default that there are no connectivity between hosts
            return False

    @staticmethod
    def _create_selectors(indexes):
        result = []
        for index in indexes:
            if index.is_valid():
                result.append(
                    selector.LeveledSelector(index)
                    if settings.current().p2p_threshold <= len(index)
                    else selector.PeerToPeerSelector(index)
                )
        return result

    def transform(self, tree):
        raise NotImplementedError()


class GroupAwareTransfomer(Transformer):
    """Accept list of host groups, for each group create separate selector."""

    def __init__(self, host_groups):
        self._host_groups = host_groups

    def transform(self, topology):
        trees = [
            (selector.TargetTree(source_iface), source_iface, group)
            for source_iface in topology.local_interfaces()
            if self._interface_is_allowed(source_iface)
            for group in self._host_groups
        ]

        for target in topology.interfaces():
            for tree, source, group in trees:
                if (
                    target.host in group and
                    target.network_type == source.network_type and
                    self._interface_is_allowed(target)
                ):
                    tree.push(target)

            # Don't starve the main thread.
            # This fixes https://st.yandex-team.ru/NETMON-458#5e74cbca2aa7ff61ee3cc9d0
            utils.yield_thread()

        for tree, _, _ in trees:
            tree.finalize()

        return self._create_selectors([tree for tree, _, _ in trees])


class NetworkAwareTransfomer(Transformer):
    """Select targets from same VRF or VLAN."""

    class _Iterable:
        def __init__(self, parent, iterable, source):
            self._parent = parent
            self._iter = iterable
            self._source = source

        def __iter__(self):
            return self

        def next(self):
            while True:
                target = self._iter.next()

                if (
                    self._parent._are_interfaces_compatible(self._source, target) and
                    self._parent._interface_is_allowed(target) and
                    target.is_virtual == self._source.is_virtual
                ):
                    return target

    def transform(self, tree):
        return self._create_selectors([
            selector.TargetTree(
                source_iface,
                self._Iterable(self, tree.interfaces(), source_iface)
            )
            for source_iface in tree.local_interfaces()
            if self._interface_is_allowed(source_iface)
        ])


class VlanAwareTransfomer(Transformer):
    """Select targets from specified VRF or VLAN."""

    class _Iterable:
        def __init__(self, parent, iterable, source, vlans, vrfs):
            self._parent = parent
            self._iter = iterable
            self._source = source
            self._vlans = vlans
            self._vrfs = vrfs

        def __iter__(self):
            return self

        def next(self):
            if self._vlans is None and self._vrfs is None:
                raise StopIteration

            while True:
                target = self._iter.next()

                match_vlan = (
                    self._vlans is not None and
                    target.vlan is not None and
                    target.vlan in self._vlans
                )

                match_vrf = (
                    self._vrfs is not None and
                    target.vrf is not None and
                    target.vrf in self._vrfs
                )

                if (
                    target.network_type == self._source.network_type and
                    self._parent._interface_is_allowed(target) and
                    (match_vlan or match_vrf)
                ):
                    return target

    def __init__(self, vlans=None, vrfs=None):
        self._vlans = vlans or {}
        self._vrfs = vrfs or {}

    def transform(self, tree):
        return self._create_selectors([
            selector.TargetTree(
                source_iface,
                self._Iterable(
                    self,
                    tree.interfaces(),
                    source_iface,
                    self._vlans,
                    self._vrfs
                )
            )
            for source_iface in tree.local_interfaces()
            if self._interface_is_allowed(source_iface)
        ])


def check_dscp_compatibility(network_type, tc):
    # Some infrastructure kludge, filter out useless probes.
    # Valid combinations at the moment:
    # Fastbone:
    #   - CS1 - RTC
    #   - CS2 - YT
    # Backbone:
    #   - CS3 - RTC
    #   - CS4 - YT

    if tc == 0:  # no priority or CS0
        return True

    if network_type == utils.FASTBONE:
        if tc == common.CS1 or tc == common.CS2:
            return True

    if network_type == utils.BACKBONE:
        if tc == common.CS3 or tc == common.CS4:
            return True

    return False
