#include "netmon.h"
#include "dns_cache.h"

#include <infra/netmon/agent/pollers/echo/controller.h>
#include <infra/netmon/agent/pollers/tcp/controller.h>
#include <infra/netmon/agent/pollers/udp/controller.h>
#include <infra/netmon/agent/pollers/icmp/controller.h>
#include <infra/netmon/agent/pollers/link/controller.h>
#include <infra/netmon/agent/resolver/controller.h>

#include <library/cpp/pybind/v2.h>

#include <util/network/sock.h>
#include <util/generic/cast.h>

auto ExportHistogram() {
    using T = NNetmon::THistogram;
    return NPyBind::TPyClass<T, NPyBind::TPyClassConfigTraits<true>, long&, int&> ("Histogram")
        .Def("get_total_count", &T::GetTotalCount)
        .Def("record_value", &T::RecordValue)
        .Def("get_min", &T::GetMin)
        .Def("get_max", &T::GetMax)
        .Def("get_mean", &T::GetMean)
        .Def("get_std_deviation", &T::GetStdDeviation)
        .Def("get_value_at_percentile", &T::GetValueAtPercentile)
        .Def("get_count_at_value", &T::GetCountAtValue)
        .Complete();
}

DEFINE_TRANSFORMERS(ExportHistogram);

i64 MergeHistogram(NNetmon::THistogram* lhs, NNetmon::THistogram* rhs) {
    // it's free function because of pybind limitation
    return lhs->Add(*rhs);
}

auto ExportAddress() {
    using T = NNetmon::TAddress;
    return NPyBind::TPyClass<T, NPyBind::TPyClassConfigTraits<true>, int&, const TString&, int&> ("Address")
        .Def("family", &T::Family)
        .Def("hostname", &T::Hostname)
        .Def("port", &T::Port)
        .Complete();
}

DEFINE_TRANSFORMERS(ExportAddress);

namespace NPyBind {
    template<>
    bool FromPyObject<TDuration>(PyObject *obj, TDuration& res) {
        ui64 x;
        if (!FromPyObject<ui64>(obj, x)) {
            return false;
        } else {
            res = TDuration::MilliSeconds(x);
            return true;
        }
    }

    template<>
    bool FromPyObject<NAddr::IRemoteAddrRef>(PyObject *obj, NAddr::IRemoteAddrRef& res) {
        const NNetmon::TAddress* ptr;
        if (!FromPyObject(obj, ptr)) {
            return false;
        }

        switch (ptr->Family) {
            case AF_INET6: {
                const TSockAddrInet6 sa(ptr->Hostname.c_str(), IntegerCast<TIpPort>(ptr->Port));
                res.Reset(MakeAtomicShared<NAddr::TOpaqueAddr>(sa.SockAddr()));
                return true;
            };
            case AF_INET: {
                const TSockAddrInet sa(ptr->Hostname.c_str(), IntegerCast<TIpPort>(ptr->Port));
                res.Reset(MakeAtomicShared<NAddr::TOpaqueAddr>(sa.SockAddr()));
                return true;
            }
            default: {
                return false;
            }
        }
    }
}

namespace NAddr {
    PyObject *BuildPyObject(const IRemoteAddr& val) {
        const sockaddr* a = val.Addr();
        char buf[INET6_ADDRSTRLEN + 1];
        PyObject* addr = nullptr;

        switch (a->sa_family) {
            case AF_INET: {
                const TIpAddress sa(*reinterpret_cast<const sockaddr_in*>(a));

                NPyBind::TPyObjectPtr res(PyTuple_New(2), true);

                PyTuple_SET_ITEM(res.Get(), 0, NPyBind::BuildPyObject(IpToString(sa.Host(), buf, sizeof(buf))));
                PyTuple_SET_ITEM(res.Get(), 1, NPyBind::BuildPyObject(sa.Port()));

                addr = res.RefGet();
                break;
            }
            case AF_INET6: {
                const sockaddr_in6* sa = reinterpret_cast<const sockaddr_in6*>(a);

                if (!inet_ntop(AF_INET6, reinterpret_cast<const void*>(&sa->sin6_addr.s6_addr), buf, sizeof(buf))) {
                    ythrow TSystemError() << "inet_ntop() failed";
                }

                NPyBind::TPyObjectPtr res(PyTuple_New(4), true);

                PyTuple_SET_ITEM(res.Get(), 0, NPyBind::BuildPyObject(buf));
                PyTuple_SET_ITEM(res.Get(), 1, NPyBind::BuildPyObject(InetToHost(sa->sin6_port)));
                PyTuple_SET_ITEM(res.Get(), 2, NPyBind::BuildPyObject(InetToHost(sa->sin6_flowinfo)));
                PyTuple_SET_ITEM(res.Get(), 3, NPyBind::BuildPyObject(sa->sin6_scope_id));

                addr = res.RefGet();
                break;
            }
            default: {
                ythrow yexception() << TStringBuf("unknown protocol");
            }
        }

        NPyBind::TPyObjectPtr res(PyTuple_New(2), true);
        PyTuple_SET_ITEM(res.Get(), 0, NPyBind::BuildPyObject(a->sa_family));
        PyTuple_SET_ITEM(res.Get(), 1, addr);
        return res.RefGet();
    }

    PyObject *BuildPyObject(const IRemoteAddrRef& val) {
        return BuildPyObject(*val);
    }
}

auto ExportDnsCache() {
    using T = NNetmon::TDnsCache;
    return NPyBind::TPyClass<T, NPyBind::TPyClassConfigTraits<true>> ("DnsCache")
        .Def("get", &T::Get)
        .Def("set", &T::Set)
        .Def("set_with_ttl", &T::SetWithTTL)
        .Def("cleanup", &T::Cleanup)
        .Complete();
}

PyObject* BuildPyObject(const TDuration& val) {
    return NPyBind::BuildPyObject(val.MilliSeconds());
}

auto ExportProbeConfig() {
    using T = NNetmon::TProbeConfig;
    return NPyBind::TPyClass<T, NPyBind::TPyClassConfigTraits<true>, unsigned int&, const NAddr::IRemoteAddrRef&, const NAddr::IRemoteAddrRef&, i32&, unsigned int&> ("ProbeConfig")
        .Def("type", &T::Type)
        .Def("timeout", &T::Timeout)
        .Def("delay", &T::Delay)
        .Def("start_delay", &T::StartDelay)
        .Def("packet_count", &T::PacketCount)
        .Def("packet_size", &T::PacketSize)
        .Def("packet_ttl", &T::TimeToLive)
        .Def("histogram", &T::Histogram)
        .Complete();
}

DEFINE_TRANSFORMERS(ExportProbeConfig);

auto ExportProbeReport() {
    using T = NNetmon::TProbeReport;
    return NPyBind::TPyClass<T, NPyBind::TPyClassConfigTraits<false>> ("ProbeReport")
        .DefReadonly("type", &T::Type)
        .DefReadonly("protocol", &T::Protocol)
        .DefReadonly("failed", &T::Failed)
        .DefReadonly("error", &T::Error)
        .DefReadonly("source_addr", &T::SourceAddr)
        .DefReadonly("target_addr", &T::TargetAddr)
        .DefReadonly("family", &T::Family)
        .DefReadonly("received", &T::Received)
        .DefReadonly("congested", &T::Congested)
        .DefReadonly("corrupted", &T::Corrupted)
        .DefReadonly("lost", &T::Lost)
        .DefReadonly("tos_changed", &T::TosChanged)
        .DefReadonly("average", &T::Average)
        .DefReadonly("histogram", &T::Histogram)
        .DefReadonly("truncated", &T::Truncated)
        .DefReadonly("offender", &T::Offender)
        .DefReadonly("generated", &T::Generated)
        .DefReadonly("type_of_service", &T::TypeOfService)
        .Complete();
}

namespace NNetmon {
    DEFINE_CONVERTERS(ExportHistogram);
    DEFINE_CONVERTERS(ExportProbeReport);

    template <>
    PyObject* ConvertToPython<TVector<TProbeReport>>(const TVector<TProbeReport>& val) {
        return ::NPyBind::BuildPyObject(val);
    }

    template <>
    PyObject* ConvertToPython<TMaybe<TVector<NAddr::IRemoteAddrRef>>>(const TMaybe<TVector<NAddr::IRemoteAddrRef>>& val) {
        return ::NPyBind::BuildPyObject(val);
    }
}

auto ExportResolverController() {
    using T = NNetmon::TResolverController;
    return NPyBind::TPyClass<T, NPyBind::TPyClassConfigTraits<true>, PyObject*&> ("ResolverController")
        .Def("resolve", &T::Resolve)
        .Def("start", &T::Start)
        .Def("stop", &T::Stop)
        .Complete();
}

auto ExportEchoController() {
    using T = NNetmon::TEchoController;
    return NPyBind::TPyClass<T, NPyBind::TPyClassConfigTraits<true>, PyObject*&> ("EchoController")
        .Def("sync_addresses", &T::SyncAddresses)
        .Def("start", &T::Start)
        .Def("stop", &T::Stop)
        .Complete();
}

auto ExportUdpPollerController() {
    using T = NNetmon::TUdpPollerController;
    return NPyBind::TPyClass<T, NPyBind::TPyClassConfigTraits<true>, PyObject*&> ("UdpPollerController")
        .Def("sync_addresses", &T::SyncAddresses)
        .Def("schedule_checks", &T::ScheduleChecks)
        .Def("start", &T::Start)
        .Def("stop", &T::Stop)
        .Complete();
}

auto ExportIcmpPollerController() {
    using T = NNetmon::TIcmpPollerController;
    return NPyBind::TPyClass<T, NPyBind::TPyClassConfigTraits<true>, PyObject*&> ("IcmpPollerController")
        .Def("sync_addresses", &T::SyncAddresses)
        .Def("schedule_checks", &T::ScheduleChecks)
        .Def("start", &T::Start)
        .Def("stop", &T::Stop)
        .Complete();
}

auto ExportTcpPollerController() {
    using T = NNetmon::TTcpPollerController;
    return NPyBind::TPyClass<T, NPyBind::TPyClassConfigTraits<true>, PyObject*&> ("TcpPollerController")
        .Def("sync_addresses", &T::SyncAddresses)
        .Def("schedule_checks", &T::ScheduleChecks)
        .Def("start", &T::Start)
        .Def("stop", &T::Stop)
        .Complete();
}

auto ExportLinkPollerController() {
    using T = NNetmon::TLinkPollerController;
    return NPyBind::TPyClass<T, NPyBind::TPyClassConfigTraits<true>, PyObject*&> ("LinkPollerController")
        .Def("sync_addresses", &T::SyncAddresses)
        .Def("report_status", &T::ReportStatus)
        .Def("start", &T::Start)
        .Def("stop", &T::Stop)
        .Complete();
}

void DoInitNetmonFunctions() {
    DefFunc("merge_histogram", &MergeHistogram);
}

void DoInitNetmonTypes() {
    ExportDnsCache();

    ExportHistogram();
    ExportAddress();
    ExportProbeConfig();
    ExportProbeReport();

    ExportResolverController();
    ExportEchoController();
    ExportUdpPollerController();
    ExportIcmpPollerController();
    ExportTcpPollerController();
    ExportLinkPollerController();
}
