#include "topology.h"

#include <library/cpp/msgpack/strbuf_adaptor.h>
#include <library/cpp/msgpack/string_adaptor.h>
#include <library/cpp/pybind/v2.h>

#include <contrib/libs/msgpack/include/msgpack.hpp>

#include <util/datetime/cputimer.h>
#include <util/memory/segmented_string_pool.h>
#include <util/generic/hash_set.h>
#include <util/stream/file.h>
#include <util/stream/zlib.h>

namespace {
    const std::size_t CHUNK_SIZE = 64 * 1024;

    // don't block tornado ioloop in TTopologyTree constructor for too long
    const TDuration TOPOLOGY_INIT_LIMIT = TDuration::Seconds(25);

    template<class TContainer, class TValue>
    class TMsgpackIterator {
    public:
        TMsgpackIterator(const TContainer& object)
            : Object(object)
        {
        }

        inline TValue* begin() const {
            return Object.ptr;
        }

        inline TValue* end() const {
            return Object.ptr + Object.size;
        }

    private:
        TContainer Object;
    };

    using TMapIterator = TMsgpackIterator<msgpack::object_map, msgpack::object_kv>;
    using TListIterator = TMsgpackIterator<msgpack::object_array, msgpack::object>;
}

struct TInterface {
    TStringBuf InterfaceName;
    TStringBuf HostName;

    TStringBuf DatacenterName;
    TStringBuf QueueName;
    TStringBuf SwitchName;

    TMaybe<TStringBuf> NetworkType;
    TMaybe<TStringBuf> VRF;
    TMaybe<int> VLAN;

    bool IsVirtual = false;

    TMaybe<TStringBuf> IPv4Address;
    TMaybe<TStringBuf> IPv6Address;
};

auto ExportInterface() {
    using T = TInterface;
    return NPyBind::TPyClass<T, NPyBind::TPyClassConfigTraits<false>> ("Interface")
        .DefReadonly("name", &T::InterfaceName)
        .DefReadonly("host", &T::HostName)
        .DefReadonly("switch", &T::SwitchName)
        .DefReadonly("queue", &T::QueueName)
        .DefReadonly("datacenter", &T::DatacenterName)
        .DefReadonly("network_type", &T::NetworkType)
        .DefReadonly("vlan", &T::VLAN)
        .DefReadonly("vrf", &T::VRF)
        .DefReadonly("is_virtual", &T::IsVirtual)
        .DefReadonly("ipv4_address", &T::IPv4Address)
        .DefReadonly("ipv6_address", &T::IPv6Address)
        .Complete();
}

DEFINE_CONVERTERS(ExportInterface);

class TTopologyTree : public TNonCopyable {
public:
    class TTopologyReader {
        TFile File_;
        TFileInput FileInput_;
        TBufferedZLibDecompress Stream_;
        msgpack::unpacker Unpacker_;
        msgpack::unpacked Object_;

    public:

        using TRef = THolder<TTopologyReader>;

        inline TTopologyReader(const TString& path)
            : File_(path, EOpenModeFlag::OpenExisting |
                    EOpenModeFlag::RdOnly | EOpenModeFlag::Seq),
              FileInput_(File_),
              Stream_(&FileInput_)
        {}

        static TRef Make(const TString& path) {
            return MakeHolder<TTopologyReader>(path);
        }

        bool Next() {
            while (true) {
                if (Unpacker_.next(Object_)) {
                    return true;
                }

                Unpacker_.reserve_buffer(CHUNK_SIZE);

                auto bytes_read = Stream_.Read(Unpacker_.buffer(), CHUNK_SIZE);
                if (!bytes_read) {
                    return false;
                }

                Unpacker_.buffer_consumed(bytes_read);
            }
        }

        msgpack::unpacked& Object() {
            return Object_;
        }
    };

    inline TTopologyTree(const TString& path, const TString& localName)
        : Path_(path),
          Reader_(TTopologyReader::Make(Path_))
    {
        TSimpleTimer timer;

        // First pass for local interfaces
        while (Reader_->Next()) {
            msgpack::object obj(Reader_->Object().get());
            if (obj.type == msgpack::type::MAP) {
                ParseHost(obj, localName, LocalInterfaces_);

                if (LocalInterfaces_.size()) {
                    break;
                }
            }

            if (timer.Get() > TOPOLOGY_INIT_LIMIT) {
                ythrow yexception() << "topology initialization took too long";
            }
        }

        // Re-open file for iterative scan
        LocalReader_ = std::move(Reader_);
        Reader_ = TTopologyReader::Make(Path_);
    }

    inline const TVector<TInterface>& Interfaces() const {
        return Interfaces_;
    }

    inline const TVector<TInterface>& LocalInterfaces() const {
        return LocalInterfaces_;
    }

    inline bool Next() {
        Interfaces_.clear();

        while (Reader_->Next()) {
            msgpack::object obj(Reader_->Object().get());
            if (obj.type == msgpack::type::MAP) {
                ParseHost(obj, "", Interfaces_);

                return true;
            }
        }

        return false;
    }

    void Reset() {
        Reader_ = TTopologyReader::Make(Path_);
    }

private:

    static inline bool IsUnknown(const TStringBuf& value) {
        return value.empty() || value == TStringBuf("unknown");
    }

    void ParseHost(msgpack::object& ptr, const TString& localName, TVector<TInterface> &result) {
        TMaybe<TStringBuf> hostName;
        TMaybe<TStringBuf> datacenterName;
        TMaybe<TStringBuf> queueName;
        TMaybe<TStringBuf> inventoryNumber;

        result.clear();

        for (auto& item : TMapIterator(ptr.via.map)) {
            if (item.val.is_nil()) {
                continue;
            }

            TStringBuf key = item.key.as<TStringBuf>();
            if (key == TStringBuf("fqdn")) {
                hostName = item.val.as<TStringBuf>();
            } else if (key == TStringBuf("dc")) {
                datacenterName = item.val.as<TStringBuf>();
            } else if (key == TStringBuf("queue")) {
                queueName = item.val.as<TStringBuf>();
            } else if (key == TStringBuf("invnum")) {
                const auto value(item.val.as<TStringBuf>());
                if (!IsUnknown(value)) {
                    inventoryNumber = value;
                }
            } else if (key == TStringBuf("interfaces") && item.val.type == msgpack::type::ARRAY) {
                for (auto& iface : TListIterator(item.val.via.array)) {
                    if (iface.type == msgpack::type::MAP) {
                        ParseIface(iface, result);
                    }
                }
            }
            // Don't parse host owners to save CPU
        }

        if (!hostName.Defined() || !datacenterName.Defined() || !queueName.Defined()) {
            return;
        }

        if (localName && hostName != localName) {
            result.clear();
            return;
        }

        for (auto& iface : result) {
            iface.HostName = *hostName;
            iface.QueueName = *queueName;
            iface.DatacenterName = *datacenterName;
            iface.IsVirtual = !inventoryNumber.Defined();
        }
    }

    inline void ParseIface(msgpack::object& ptr, TVector<TInterface>& interfaces) {
        TMaybe<TStringBuf> hostName;
        TMaybe<TStringBuf> switchName;
        TMaybe<TStringBuf> networkType;
        TMaybe<TStringBuf> vrf;
        TMaybe<int> vlan;

        TMaybe<TStringBuf> ipv4Address;
        TMaybe<TStringBuf> ipv6Address;

        for (auto& item : TMapIterator(ptr.via.map)) {
            TStringBuf key = item.key.as<TStringBuf>();
            if (key == TStringBuf("fqdn")) {
                hostName = item.val.as<TStringBuf>();
            } else if (key == TStringBuf("switch")) {
                switchName = item.val.as<TStringBuf>();
            } else if (key == TStringBuf("network_type")) {
                const auto value(item.val.as<TStringBuf>());
                if (!IsUnknown(value)) {
                    networkType = value;
                }
            } else if (key == TStringBuf("vrf")) {
                const auto value(item.val.as<TStringBuf>());
                if (!IsUnknown(value)) {
                    vrf = value;
                }
            } else if (key == TStringBuf("vlan")) {
                vlan = item.val.as<int>();
            } else if (key == TStringBuf("ipv4addr")) {
                const auto value(item.val.as<TStringBuf>());
                if (!IsUnknown(value)) {
                    ipv4Address = value;
                }
            } else if (key == TStringBuf("ipv6addr")) {
                const auto value(item.val.as<TStringBuf>());
                if (!IsUnknown(value)) {
                    ipv6Address = value;
                }
            }
        }

        if (!hostName.Defined() || !switchName.Defined()) {
            return;
        }

        interfaces.emplace_back();
        TInterface& iface(interfaces.back());
        iface.InterfaceName = *hostName;
        iface.SwitchName = *switchName;
        if (networkType.Defined()) {
            iface.NetworkType.ConstructInPlace(*networkType);
        }
        if (vrf.Defined()) {
            iface.VRF.ConstructInPlace(*vrf);
        }
        if (vlan.Defined()) {
            iface.VLAN = vlan;
        }
        if (ipv4Address.Defined()) {
            iface.IPv4Address.ConstructInPlace(*ipv4Address);
        }
        if (ipv6Address.Defined()) {
            iface.IPv6Address.ConstructInPlace(*ipv6Address);
        }
    }

    const TString Path_;

    TTopologyReader::TRef Reader_;
    TTopologyReader::TRef LocalReader_;

    /* Following vectors store references to objects
       from corresponding readers above */

    TVector<TInterface> Interfaces_;
    TVector<TInterface> LocalInterfaces_;
};

auto ExportTopologyTree() {
    using T = TTopologyTree;
    return NPyBind::TPyClass<T, NPyBind::TPyClassConfigTraits<true>, const TString&, const TString&> ("TopologyTree")
        .Def("interfaces", &T::Interfaces)
        .Def("local_interfaces", &T::LocalInterfaces)
        .Def("next", &T::Next)
        .Def("reset", &T::Reset)
        .Complete();
}

void DoInitTopologyTypes() {
    ExportInterface();
    ExportTopologyTree();
}
