#include "utils.h"

#include <infra/netmon/agent/common/metrics.h>
#include <infra/netmon/agent/common/logger.h>
#include <infra/netmon/agent/common/settings.h>

#include <library/cpp/pybind/v2.h>
#include <library/cpp/logger/global/global.h>

#include <util/generic/xrange.h>
#include <util/stream/mem.h>
#include <util/network/interface.h>

#include <array>

#include <sys/capability.h>
#include <sys/prctl.h>

#include <grp.h>

namespace {
    struct TCapsDeleter {
        static inline void Destroy(cap_t ptr) noexcept {
            if (ptr != nullptr) {
                cap_free(ptr);
            }
        }
    };

    using TCapsHolder = THolder<std::remove_pointer<cap_t>::type, TCapsDeleter>;

    void DropCapabilities() {
        TCapsHolder caps(cap_from_text("cap_net_raw+eip"));
        if (cap_set_proc(caps.Get()) < 0) {
            ythrow TSystemError() << "cap_set_proc() failed";
        }

    #ifndef PR_CAP_AMBIENT
    #   define PR_CAP_AMBIENT       47
    #endif
    #ifndef PR_CAP_AMBIENT_RAISE
    #   define PR_CAP_AMBIENT_RAISE 2
    #endif

        // ignore errors on Linux < 4.3
        (void)prctl(PR_CAP_AMBIENT, PR_CAP_AMBIENT_RAISE, CAP_NET_RAW, 0, 0);
    }

    bool DropPrivileges(const TString& user, unsigned int uid, unsigned int gid) {
        if (geteuid() == 0) {
            if (setgid(gid) < 0) {
                ythrow TSystemError() << "setgid(" << gid << ") failed";
            }

            if (initgroups(user.data(), gid) < 0) {
                ythrow TSystemError() << "initgroups(" << user << ", " << gid << ") failed";
            }

            if (prctl(PR_SET_KEEPCAPS, 1, 0, 0, 0) < 0) {
                ythrow TSystemError() << TStringBuf("prctl(PR_SET_KEEPCAPS, 1) failed");
            }

            if (setuid(uid) < 0) {
                ythrow TSystemError() << "setuid(" << uid << ") failed";
            }

            DropCapabilities();
        }
        return true;
    }

    bool InitMetrics() {
        NNetmon::InitUnistat();
        return true;
    }

    TString GetMetrics() {
        return TUnistat::Instance().CreateJsonDump(10, false);
    }

    bool InitGlobalLog(PyObject* logger) {
        DoInitGlobalLog(MakeHolder<TPythonLogBackend>(logger));
        return true;
    }

    bool StartUnistatPusher() {
        Py_BEGIN_ALLOW_THREADS
        NNetmon::StartUnistatPusher();
        Py_END_ALLOW_THREADS
        return true;
    }

    bool StopUnistatPusher() {
        Py_BEGIN_ALLOW_THREADS
        NNetmon::StopUnistatPusher();
        Py_END_ALLOW_THREADS
        return true;
    }
}

void DoInitUtilsFunctions() {
    DefFunc("drop_privileges", &DropPrivileges);
    DefFunc("init_metrics", &InitMetrics);
    DefFunc("get_metrics", &GetMetrics);
    DefFunc("init_global_log", &InitGlobalLog);
    DefFunc("start_unistat_pusher", &StartUnistatPusher);
    DefFunc("stop_unistat_pusher", &StopUnistatPusher);
}

auto ExportSettings() {
    using T = NNetmon::TSettings;
    return NPyBind::TPyClass<T, NPyBind::TPyClassConfigTraits<true>>("Settings")
        .AsProperty("check_full_tos", &T::GetCheckFullTypeOfService, &T::SetCheckFullTypeOfService)
        .AsProperty("fix_reply_tos", &T::GetFixReplyTypeOfService, &T::SetFixReplyTypeOfService)
        .AsProperty("link_poller_packet_size", &T::GetLinkPollerPacketSize, &T::SetLinkPollerPacketSize)
        .AsProperty("packet_size", &T::GetPacketSize, &T::SetPacketSize)
        .AsProperty("udp_multiport_probes", &T::GetUdpUseMultiPortProbes, &T::SetUdpUseMultiPortProbes)
        .AsProperty("udp_socket_count", &T::GetUdpSocketCount, &T::SetUdpSocketCount)
        .AsProperty("use_hw_timestamps", &T::GetUseHwTimestamps, &T::SetUseHwTimestamps)
        .Complete();
}

void DoInitUtilsTypes() {
    ExportSettings();
}
