#include "controller.h"

#include <library/cpp/logger/thread.h>

#include <util/system/thread.h>

namespace NNetmon {
    namespace {
        class TServiceStopCall : public TCall {
        public:
            TServiceStopCall(TRpcExecutor& executor, IService* service)
                : Executor(executor)
                , Service(service)
            {
            }

        protected:
            void Dispatch(TCont*) noexcept override {
                if (Service) {
                    Service->Stop();
                }

                // don't call stop here because of infinite loop
                Executor.Abort();
            }

        private:
            TRpcExecutor& Executor;
            IService* Service;
        };
    }

    class TBaseController::TImpl : public ISimpleThread {
    public:
        TImpl(TBaseController* parent, const TStringBuf& name, TLog& logger)
            : Name(name)
            , Logger(logger)
            , Executor(
#ifdef _asan_enabled_
                4 *
#endif
                50 * 1024
              )
            , RpcExecutor(&Executor)
            , Parent(parent)
            , Service(nullptr)
        {
        }

        ~TImpl()
        {
            // stop controller loop from another thread
            RpcExecutor.Schedule(MakeIntrusive<TServiceStopCall>(RpcExecutor, Service));
            Join();

            if (Service) {
                Parent->DestroyService();
                Service = nullptr;
            }
        }

        void CreateService() {
            if (!Service) {
                Service = Parent->CreateService(Logger, Executor);
            }
        }

        void* ThreadProc() override {
            TThread::SetCurrentThreadName(Name.data());
            Logger << TLOG_DEBUG << "Starting RPC...";
            RpcExecutor.Start();
            Logger << TLOG_DEBUG << "Starting executor...";
            Executor.Execute();
            Logger << TLOG_DEBUG << "Executor exited...";
            RpcExecutor.Stop();
            Logger << TLOG_DEBUG << "RPC stopped...";
            return this;
        }

        const TRpcExecutor& GetRpcExecutor() const noexcept {
            return RpcExecutor;
        }

    private:
        TStringBuf Name;
        TLog& Logger;
        TContExecutor Executor;
        TRpcExecutor RpcExecutor;
        TBaseController* Parent;
        IService* Service;
    };

    TBaseController::TBaseController(const TStringBuf& name, PyObject* logger)
        : PythonLogBackend(logger)
        , Logger(MakeHolder<TThreadedLogBackend>(&PythonLogBackend))
        , Impl(MakeHolder<TImpl>(this, name, Logger))
    {
    }

    TBaseController::~TBaseController()
    {
        Logger << TLOG_DEBUG << "Destroying controller...";

        Py_BEGIN_ALLOW_THREADS
        Impl.Destroy();
        Logger.ReleaseBackend();
        Py_END_ALLOW_THREADS
    }

    void TBaseController::Start() {
        Y_VERIFY(Impl);

        Logger << TLOG_DEBUG << "Starting controller...";

        Py_BEGIN_ALLOW_THREADS
        Impl->CreateService();
        Impl->Start();
        Py_END_ALLOW_THREADS
    }

    void TBaseController::Stop() {
        Y_VERIFY(Impl);

        Logger << TLOG_DEBUG << "Stopping controller...";

        Py_BEGIN_ALLOW_THREADS
        Impl.Destroy();
        Logger.ReleaseBackend();
        Py_END_ALLOW_THREADS
    }

    void TBaseController::Execute(const TCall::TRef& call) const noexcept {
        Y_VERIFY(Impl);
        Impl->GetRpcExecutor().Schedule(call);
    }
}
