#pragma once

#include "logger.h"
#include "rpc.h"
#include "probe.h"
#include "utils.h"

#include <library/cpp/logger/log.h>

namespace NNetmon {
    class IService {
    public:
        virtual ~IService() = default;

        virtual void Stop() noexcept = 0;
    };

    class TBaseController : public TNonCopyable {
    public:
        TBaseController(const TStringBuf& name, PyObject* logger);
        virtual ~TBaseController();

        void Start();
        void Stop();

        void Execute(const TCall::TRef& call) const noexcept;

    protected:
        virtual IService* CreateService(TLog& logger, TContExecutor& executor) = 0;
        virtual void DestroyService() noexcept = 0;

    private:
        TPythonLogBackend PythonLogBackend;
        TLog Logger;

        class TImpl;
        THolder<TImpl> Impl;
    };

    template <class T>
    class TBaseSyncAddressesCall : public TPythonCall {
    public:
        TBaseSyncAddressesCall(PyObject* callback, T& service, const TVector<const TAddress*>& addressList)
            : TPythonCall(callback)
            , Service(service)
        {
            AddressList.reserve(addressList.size());
            for (const auto& address : addressList) {
                AddressList.emplace_back(*address);
            }
        }

    protected:
        void Dispatch(TCont*) override {
            // FIXME: proper error handling
            TVector<TNetworkAddress> addresses;
            for (const auto& x : AddressList) {
                addresses.emplace_back(x.Hostname, x.Port);
            }
            Service.SyncAddresses(addresses);
        }

    private:
        T& Service;
        TVector<TAddress> AddressList;
    };

    template <class T>
    class TBaseScheduleChecksCall : public TPythonCall {
    public:
        TBaseScheduleChecksCall(PyObject* callback, T& service, const TVector<const TProbeConfig*>& configList)
            : TPythonCall(callback)
            , Service(service)
        {
            ConfigList.reserve(configList.size());
            for (const auto& config : configList) {
                ConfigList.emplace_back(*config);
            }
        }

    protected:
        void Dispatch(TCont* cont) override {
            Service.ScheduleChecks(cont, ConfigList, ReportList);
        }

        PyObject* ToPython() override {
            return ConvertToPython(ReportList);
        }

    private:
        T& Service;
        TVector<TProbeConfig> ConfigList;
        TVector<TProbeReport> ReportList;
    };
}
