#include "metrics.h"

#include <util/generic/xrange.h>

#include <cmath>

namespace NNetmon {
    namespace {
        const i64 BUCKET_COUNT = 100; // should be divisible by 2

        class TDefaultWeights {
        public:
            TDefaultWeights() {
                Values.resize(BUCKET_COUNT);
                Values[0] = 0.0;
                for (const auto idx : xrange(BUCKET_COUNT - 1)) {
                    // 60s is our ceil
                    Values[idx + 1] = std::pow(1.2, static_cast<i64>(idx));
                }
            }

            TVector<double> Values;
        };
    }

    const TVector<double>& GetBucketWeights() {
        return Singleton<TDefaultWeights>()->Values;
    }

    void InitUnistat() {
        auto& t = TUnistat::Instance();

        auto priority = NUnistat::TPriority(10);
        const auto& intervals = GetBucketWeights();

        t.DrillHistogramHole("probe_receive_lag", "hgram", priority, intervals);

        t.DrillHistogramHole("echo_dispatch_delay", "hgram", priority, intervals);
        t.DrillHistogramHole("echo_flight_delay", "hgram", priority, intervals);
        t.DrillHistogramHole("echo_queue_delay", "hgram", priority, intervals);
        t.DrillFloatHole("echo_received_probes", "summ", priority);
        t.DrillFloatHole("echo_sent_probes", "summ", priority);
        t.DrillFloatHole("echo_truncated_probes", "summ", priority);
        t.DrillFloatHole("echo_expired_probes", "summ", priority);

        t.DrillFloatHole("packet_sent", "summ", priority);
        t.DrillFloatHole("packet_received", "summ", priority);
        t.DrillFloatHole("packet_lost", "summ", priority);
        t.DrillHistogramHole("packet_round_trip_delay", "hgram", priority, intervals);
    }

    void StartUnistatPusher() {
        auto initializer = [](TUnistat& t) {
            auto priority = NUnistat::TPriority(10);
            const auto& intervals = GetBucketWeights();

            t.DrillHistogramHole("link_rtt_hw", "hgram", priority, intervals);
            t.DrillHistogramHole("link_rtt_sys", "hgram", priority, intervals);
            t.DrillHistogramHole("link_rtt_user", "hgram", priority, intervals);
            t.DrillHistogramHole("link_send_interval", "hgram", priority,
                { 50.0,  60.0,  70.0,  80.0,  90.0,  95.0, 100.0,
                 105.0, 110.0, 115.0, 120.0, 130.0, 140.0, 150.0,
                 160.0, 170.0, 180.0, 190.0, 200.0, 250.0, 300.0,
                 400.0, 500.0 });

            t.DrillHistogramHole("clock_skew_sys", "hgram", priority, intervals);

            t.DrillHistogramHole("target_delay", "hgram", priority, intervals);

            t.DrillFloatHole("reports_throttled", "summ", priority);

            t.DrillFloatHole("schedule_outdated_ips", "summ", priority);

            t.DrillFloatHole("probe_overflows", "summ", priority);

            t.DrillFloatHole("packet_congested", "summ", priority);
            t.DrillFloatHole("packet_corrupted", "summ", priority);
            t.DrillFloatHole("packet_martian", "summ", priority);

            t.DrillFloatHole("dns_cache_hit", "summ", priority);
            t.DrillFloatHole("dns_cache_miss", "summ", priority);

            t.DrillFloatHole("yp_sd_fails", "summ", priority);
            t.DrillFloatHole("dns_fails", "summ", priority);

            t.DrillFloatHole("agent_starts", "summ", priority);
        };
        TUnistatPusher::TOptions options;
        options.ParseTags("itype=netmonagent;ctype=production;tier=none");
        TUnistatPusher::Instance().Start(initializer, options);
    }

    void StopUnistatPusher() {
        TUnistatPusher::Instance().Stop();
    }
}
