#pragma once

#include <library/cpp/histogram/hdr/histogram.h>
#include <library/cpp/logger/global/global.h>

#include <util/network/address.h>

namespace NNetmon {
    namespace {
        const TDuration MAX_PACKET_DELAY = TDuration::Seconds(10);
        const double CLOCK_SKEW_WARN_THRESHOLD = 50000; // 50 ms
    }

    enum class EProtocol : ui16 {
        ICMP = 1,
        UDP = 2,
        TCP = 3,
        LINK_POLLER = 4
    };

    class THistogram {
    public:
        THistogram(i64 highestTrackableValue, i32 numberOfSignificantValueDigits)
            : Histogram_(MakeAtomicShared<NHdr::THistogram>(highestTrackableValue, numberOfSignificantValueDigits))
        {
        }

        THistogram(const THistogram& other)
            : Histogram_(other.Histogram_)
        {
        }

        i64 Add(const THistogram& other) {
            return Histogram_->Add(*other.Histogram_);
        }

        inline i64 GetTotalCount() const {
            return Histogram_->GetTotalCount();
        }

        inline bool RecordValue(i64 value) {
            return Histogram_->RecordValue(value);
        }

        inline i64 GetMin() const {
            return Histogram_->GetMin();
        }

        inline i64 GetMax() const {
            return Histogram_->GetMax();
        }

        inline double GetMean() const {
            return Histogram_->GetMean();
        }

        inline double GetStdDeviation() const {
            return Histogram_->GetStdDeviation();
        }

        i64 GetValueAtPercentile(double percentile) const {
            return Histogram_->GetValueAtPercentile(percentile);
        }

        i64 GetCountAtValue(i64 value) const {
            return Histogram_->GetCountAtValue(value);
        }

    private:
        TAtomicSharedPtr<NHdr::THistogram> Histogram_;
    };

    struct TProbeConfig {
        TProbeConfig(unsigned int type,
                     const NAddr::IRemoteAddrRef& source,
                     const NAddr::IRemoteAddrRef& target,
                     i32 tos,
                     unsigned int packetCount)
            : Type(type)
            , SourceAddr(source)
            , TargetAddr(target)
            , TypeOfService(tos)
            , Timeout(MAX_PACKET_DELAY)
            , Delay(TDuration::MilliSeconds(100))
            , StartDelay(TDuration::MilliSeconds(1000))
            , PacketCount(packetCount)
            , PacketSize(8000)
            , TimeToLive(-1)
            , Histogram(false)
        {
        }

        unsigned int Type;
        NAddr::IRemoteAddrRef SourceAddr;
        NAddr::IRemoteAddrRef TargetAddr;
        i32 TypeOfService;
        TDuration Timeout;
        TDuration Delay;
        TDuration StartDelay;
        unsigned int PacketCount;
        std::size_t PacketSize;
        i32 TimeToLive;
        bool Histogram;
    };

    struct TProbeReport {
        TProbeReport(unsigned int type,
                     EProtocol protocol,
                     const NAddr::IRemoteAddrRef& source,
                     const NAddr::IRemoteAddrRef& target,
                     i32 tos,
                     ui16 received,
                     ui16 congested,
                     ui16 corrupted,
                     ui16 lost,
                     ui16 tosChanged,
                     double average,
                     const THistogram* histogram,
                     bool truncated,
                     const TMaybe<NAddr::TOpaqueAddr>& offender)
            : Type(type)
            , Protocol(static_cast<ui16>(protocol))
            , SourceAddr(source)
            , TargetAddr(target)
            , TypeOfService(tos)
            , Family(TargetAddr->Addr()->sa_family)
            , Failed(false)
            , Received(received)
            , Congested(congested)
            , Corrupted(corrupted)
            , Lost(lost)
            , TosChanged(tosChanged)
            , Average(average)
            , Truncated(truncated)
            , Offender(offender)
            , Generated(TInstant::Now().MicroSeconds())
        {
            if (histogram) {
                Histogram.ConstructInPlace(*histogram);
            }
        }

        TProbeReport(unsigned int type,
                     EProtocol protocol,
                     const NAddr::IRemoteAddrRef& source,
                     const NAddr::IRemoteAddrRef& target,
                     i32 tos,
                     const TString& error)
            : Type(type)
            , Protocol(static_cast<ui16>(protocol))
            , SourceAddr(source)
            , TargetAddr(target)
            , TypeOfService(tos)
            , Family(TargetAddr->Addr()->sa_family)
            , Failed(true)
            , Error(error)
            , Generated(TInstant::Now().MicroSeconds())
        {
        }

        const unsigned int Type = 0;
        const ui16 Protocol = 0;
        const NAddr::IRemoteAddrRef SourceAddr;
        const NAddr::IRemoteAddrRef TargetAddr;
        const i32 TypeOfService = 0;
        const ui16 Family = 0;
        const bool Failed = false;
        const TString Error;
        const ui16 SourcePort = 0;
        const ui16 Received = 0;
        const ui16 Congested = 0;
        const ui16 Corrupted = 0;
        const ui16 Lost = 0;
        const ui16 TosChanged = 0;
        const double Average = 0.0;
        TMaybe<THistogram> Histogram;
        const bool Truncated = false;
        const TMaybe<NAddr::TOpaqueAddr> Offender;
        const ui64 Generated = 0;
    };

    struct TProbeStatistics {
        bool Failed_;
        TString Error_;
        ui16 LastSeqno_;

        ui16 ReceivedCounter_;
        ui16 LostCounter_;
        ui16 TosChangedCounter_;
        ui16 CongestedCounter_;
        ui16 CorruptedCounter_;
        double Average_;
        double MaxClockSkewSys_;
        THolder<THistogram> Histogram_;
        bool Truncated_;
        TMaybe<NAddr::TOpaqueAddr> Offender_;

        TProbeStatistics(bool histogram)
            : Failed_(false)
            , LastSeqno_(0)
            , ReceivedCounter_(0)
            , LostCounter_(0)
            , TosChangedCounter_(0)
            , CongestedCounter_(0)
            , CorruptedCounter_(0)
            , Average_(0)
            , MaxClockSkewSys_(0)
            , Truncated_(false)
        {
            if (histogram) {
                Histogram_.Reset(MakeHolder<THistogram>(static_cast<i64>(MAX_PACKET_DELAY.MicroSeconds()), 3));
            }
        }

        void Report(const TProbeConfig& config, EProtocol protocol, TMaybe<TProbeReport>& report) noexcept {
            if (MaxClockSkewSys_ > CLOCK_SKEW_WARN_THRESHOLD) {
                WARNING_LOG << "Clock skew with " << NAddr::PrintHost(*config.TargetAddr) << " is too big: " << MaxClockSkewSys_ / 1000 << " ms" << Endl;
            }
            if (Failed_) {
                report.ConstructInPlace(
                    config.Type,
                    protocol,
                    config.SourceAddr,
                    config.TargetAddr,
                    config.TypeOfService,
                    Error_
                );
            } else {
                report.ConstructInPlace(
                    config.Type,
                    protocol,
                    config.SourceAddr,
                    config.TargetAddr,
                    config.TypeOfService,
                    ReceivedCounter_,
                    CongestedCounter_,
                    CorruptedCounter_,
                    LostCounter_,
                    TosChangedCounter_,
                    Average_,
                    Histogram_.Get(),
                    Truncated_,
                    Offender_
                );
            }
        }
    };
}
