#include "rpc.h"

#include <library/cpp/coroutine/util/pipeevent.h>

#include <util/thread/lfqueue.h>

namespace NNetmon {
    namespace {
        class TGILGuard : public TNonCopyable {
        public:
            TGILGuard()
                : State(PyGILState_Ensure())
            {
            }

            ~TGILGuard() {
                PyGILState_Release(State);
            }

        private:
            PyGILState_STATE State;
        };
    }

    // call should live no longer than given object
    void TCall::Execute(TCont* cont) noexcept {
        Y_VERIFY(OnReady_);
        Cont_ = cont;

        try {
            Dispatch(cont);
        } catch (...) {
            Error_ = CurrentExceptionMessage();
        }

        Cont_ = nullptr;

        RunCallbacks();
    }

    void TCall::Cancel() noexcept {
        Y_VERIFY(OnReady_);
        Error_ = "cancelled";
        RunCallbacks();
    }

    TPythonCall::~TPythonCall() {
        if (Callback_) {
            TGILGuard guard;
            Callback_.Reset();
        }
    }

    void TPythonCall::RunCallbacks() noexcept {
        Y_VERIFY(Callback_);

        {
            TGILGuard guard;
            NPyBind::TPyObjectPtr none(Py_None, true);
            if (Error().empty()) {
                NPyBind::TPyObjectPtr result(ToPython(), true);
                Y_VERIFY(PyObject_CallFunctionObjArgs(Callback_.Get(), result.Get(), none.RefGet(), NULL) != nullptr);
            } else {
                NPyBind::TPyObjectPtr result(NPyBind::BuildPyObject(Error()), true);
                Y_VERIFY(PyObject_CallFunctionObjArgs(Callback_.Get(), none.RefGet(), result.Get(), NULL) != nullptr);
            }
            Callback_.Reset();
        }

        // call to OnReady should be the last because after that instance will be destroyed
        TCall::RunCallbacks();
    }

    class TRpcExecutor::TImpl {
    public:
        inline TImpl(TContExecutor* executor)
            : Executor_(executor)
            , Cont_(nullptr)
            , Running_(false)
        {
        }

        ~TImpl()
        {
            Stop();
        }

        inline void Start() {
            Y_VERIFY(!Cont_);
            Cont_ = Executor_->Create<TRpcExecutor::TImpl, &TRpcExecutor::TImpl::Run>(this, "rpc_service");
        }

        void Schedule(const TCall::TRef& call) const noexcept {
            // this will increase ref counts to given call
            IncomingQueue_.Enqueue(call);
            // yes, make this simpler. Write shouldn't block for too long.
            IncomingEvent_.Signal();
        }

        void Abort() {
            if (Cont_) {
                Running_ = false;
                Cont_->Cancel();
            }
        }

    private:
        void OnReady(TCall* ptr) {
            // FIXME: check for memory leak
            TCall::TRef call(ptr);
            ptr->UnRef();
            ptr->Unlink();
        }

        inline void Run(TCont* cont) noexcept {
            Cont_ = cont;
            DoRun();
            Cont_ = nullptr;
        }

        inline void DoRun() noexcept {
            Y_VERIFY(Cont_ != nullptr && !Running_);
            Running_ = true;

            while (!Cont_->Cancelled()) {
                try {
                    IncomingEvent_.Wait(Cont_);
                } catch(const TSystemError& error) {
                    if (error.Status() == ECANCELED) {
                        break;
                    } else {
                        // this shouldn't happened
                        throw;
                    }
                }

                while (Running_) {
                    TCall::TRef call;
                    if (!IncomingQueue_.Dequeue(&call)) {
                        break;
                    }

                    call->Ref();
                    call->SetReadyCallback<TImpl, &TImpl::OnReady>(this);
                    Calls_.PushBack(call.Get());

                    Executor_->Create<TCall, &TCall::Execute>(call.Get(), "rpc_call");
                }
            }

            // cancel enqueued requests
            while (!IncomingQueue_.IsEmpty()) {
                TCall::TRef call;
                IncomingQueue_.Dequeue(&call);
                call->Cancel();
            }

            // wait for active jobs
            while (!Calls_.Empty()) {
                Cont_->Join(Calls_.Front()->Cont());
            }

            Running_ = false;
        }

        inline void Stop() noexcept {
            Abort();

            if (Cont_) {
                while (Cont_) {
                    Executor_->Running()->Yield();
                }
            }
        }

        TContExecutor* Executor_;
        TCont* Cont_;

        mutable TLockFreeQueue<TCall::TRef> IncomingQueue_;
        mutable TPipeEvent IncomingEvent_;

        TCall::TListType Calls_;

        bool Running_;
    };

    TRpcExecutor::TRpcExecutor(TContExecutor* executor)
    {
        Impl.Reset(MakeHolder<TImpl>(executor));
    }

    TRpcExecutor::~TRpcExecutor()
    {
    }

    void TRpcExecutor::Start() {
        Y_VERIFY(Impl);
        Impl->Start();
    }

    void TRpcExecutor::Abort() noexcept {
        Y_VERIFY(Impl);
        Impl->Abort();
    }

    void TRpcExecutor::Stop() noexcept {
        Y_VERIFY(Impl);
        Impl.Destroy();
    }

    void TRpcExecutor::Schedule(const TCall::TRef& call) const noexcept {
        Y_VERIFY(Impl);
        Impl->Schedule(call);
    }
}
