#pragma once

#include <library/cpp/coroutine/engine/impl.h>
#include <library/cpp/pybind/ptr.h>
#include <library/cpp/pybind/cast.h>

namespace NNetmon {
    // needed to declare implementation in *.cpp files
    template<typename T>
    PyObject* ConvertToPython(const T& val);

    class TCall : public TIntrusiveListItem<TCall>, public TAtomicRefCount<TCall> {
    public:
        using TRef = TIntrusivePtr<TCall>;
        using TListType = TIntrusiveList<TCall>;

        virtual ~TCall() = default;

        template <typename T, void (T::*M)(TCall*)>
        inline void SetReadyCallback(T* obj) noexcept {
            OnReady_ = [obj, this] () {
                (obj->*M)(this);
            };
        }

        // call should live no longer than the given object
        void Execute(TCont* cont) noexcept;
        void Cancel() noexcept;

        inline const TString& Error() noexcept {
            return Error_;
        }

        virtual void RunCallbacks() noexcept {
            OnReady_();
        }

        inline TCont* Cont() noexcept {
            return Cont_;
        }

    protected:
        // this method should be implemented by children
        virtual void Dispatch(TCont* cont) = 0;

    private:
        std::function<void()> OnReady_;
        TString Error_;
        TCont* Cont_ = nullptr;
    };

    class TPythonCall : public TCall {
    public:
        inline TPythonCall(PyObject* callback)
            : TCall()
            // we are the owner since now
            , Callback_(callback, true)
        {
            if (!PyCallable_Check(Callback_.Get())) {
                ythrow yexception() << "object is not callable";
            }
        }

        ~TPythonCall();

        void RunCallbacks() noexcept override;

    protected:
        virtual PyObject* ToPython() {
            Py_RETURN_NONE;
        }

    private:
        NPyBind::TPyObjectPtr Callback_;
    };

    class TRpcExecutor : public TNonCopyable {
    public:
        TRpcExecutor(TContExecutor* executor);
        ~TRpcExecutor();

        void Start();
        void Abort() noexcept;
        void Stop() noexcept;
        void Schedule(const TCall::TRef& call) const noexcept;

    private:
        class TImpl;
        THolder<TImpl> Impl;
    };
}
