#pragma once

#include "rpc.h"

#include <library/cpp/logger/log.h>
#include <library/cpp/logger/null.h>
#include <library/cpp/containers/intrusive_hash/intrhash.h>

#include <util/generic/hash_set.h>
#include <util/system/backtrace.h>
#include <util/digest/multi.h>
#include <util/network/address.h>

#include <linux/if_packet.h>
#include <linux/if_ether.h>

namespace NNetmon {
    namespace {
        /* Keep sll_protocol and link poller socket protocol the same! */
        const int LL_SOCK_PROTO = ETH_P_ALL;
    }

    struct TAddress {
        TAddress()
        {
        }

        TAddress(int family, const TString& hostname, int port)
            : Family(family)
            , Hostname(hostname)
            , Port(port)
        {
        }

        int Family;
        TString Hostname;
        int Port;
    };

    class TLlAddress : public NAddr::IRemoteAddr {
    public:
        TLlAddress(int ifindex, const ui8 addr[ETH_ALEN])
        {
            ::memset(&Addr_, 0, sizeof(sockaddr_ll));
            Addr_.sll_family = AF_PACKET;
            Addr_.sll_protocol = htons(LL_SOCK_PROTO);
            Addr_.sll_ifindex = ifindex;
            Addr_.sll_halen = ETH_ALEN;
            ::memcpy(Addr_.sll_addr, addr, ETH_ALEN);
        }

        TLlAddress(const sockaddr_ll& addr) {
            ::memcpy(&Addr_, &addr, sizeof(Addr_));
        }

        socklen_t Len() const override {
            return sizeof(sockaddr_ll);
        }

        const sockaddr* Addr() const override {
            return (const sockaddr*)&Addr_;
        }

        sockaddr_ll* LlAddr() {
            return &Addr_;
        }

    private:
        sockaddr_ll Addr_;
    };

    union TSa {
        const sockaddr* Sa;
        const sockaddr_in* In;
        const sockaddr_in6* In6;
        const sockaddr_ll* Ll;

        inline TSa(const sockaddr* sa) noexcept
            : Sa(sa)
        {
        }

        std::size_t hash() const noexcept {
            switch (Sa->sa_family) {
                case AF_INET:
                    return MultiHash(
                        In->sin_addr.s_addr,
                        In->sin_port
                    );
                case AF_INET6:
                    return MultiHash(
                        ComputeHash(TStringBuf{reinterpret_cast<const char*>(&In6->sin6_addr), sizeof(in6_addr)}),
                        In6->sin6_port
                    );
                default:
                    return 0;
            }
        }

        inline bool operator==(const TSa& r) const noexcept {
            if (Sa->sa_family == r.Sa->sa_family) {
                switch (Sa->sa_family) {
                    case AF_INET:
                        return In->sin_port == r.In->sin_port && In->sin_addr.s_addr == r.In->sin_addr.s_addr;
                    case AF_INET6:
                        return In6->sin6_port == r.In6->sin6_port && !memcmp(&In6->sin6_addr, &r.In6->sin6_addr, sizeof(in6_addr));
                }
            }

            return false;
        }

        inline bool operator!=(const TSa& r) const noexcept {
            return !(*this == r);
        }
    };

    inline NAddr::IRemoteAddrRef CopyRemoteAddr(const NAddr::IRemoteAddr& addr_) noexcept {
        const TSa sa(addr_.Addr());
        switch (sa.Sa->sa_family) {
            case AF_INET: {
                return MakeAtomicShared<NAddr::TIPv4Addr>(*sa.In);
            }
            case AF_INET6: {
                return MakeAtomicShared<NAddr::TIPv6Addr>(*sa.In6);
            }
            case AF_PACKET: {
                return MakeAtomicShared<TLlAddress>(*sa.Ll);
            }
            default: {
                Y_VERIFY(false);
            }
        }
    }

    inline void SetPort(sockaddr_in* addr, ui16 port) noexcept {
        addr->sin_port = HostToInet(port);
    }

    inline void SetPort(sockaddr_in6* addr, ui16 port) noexcept {
        addr->sin6_port = HostToInet(port);
    }

    inline void SetPort(sockaddr* addr, ui16 port) noexcept {
        switch (addr->sa_family) {
            case AF_INET: {
                return SetPort(reinterpret_cast<sockaddr_in*>(addr), port);
            }
            case AF_INET6: {
                return SetPort(reinterpret_cast<sockaddr_in6*>(addr), port);
            }
            default: {
                return;
            }
        }
    }

    inline NAddr::IRemoteAddrRef SetPort(const NAddr::IRemoteAddr& source, ui16 port=0) noexcept {
        NAddr::TOpaqueAddr storage(&source);
        SetPort(storage.MutableAddr(), port);
        return CopyRemoteAddr(storage);
    }

    inline ui16 ExtractPort(const sockaddr_in* addr) noexcept {
        return InetToHost(addr->sin_port);
    }

    inline ui16 ExtractPort(const sockaddr_in6* addr) noexcept {
        return InetToHost(addr->sin6_port);
    }

    inline ui16 ExtractPort(const sockaddr* addr) noexcept {
        switch (addr->sa_family) {
            case AF_INET: {
                return ExtractPort(reinterpret_cast<const sockaddr_in*>(addr));
            }
            case AF_INET6: {
                return ExtractPort(reinterpret_cast<const sockaddr_in6*>(addr));
            }
            default: {
                return 0;
            }
        }
    }

    inline ui16 ExtractPort(const NAddr::IRemoteAddrRef& addr) noexcept {
        return ExtractPort(addr->Addr());
    }

    struct TAddrHash {
        inline size_t operator()(const TSa& x) const noexcept {
            return x.hash();
        }
        inline size_t operator()(const NAddr::IRemoteAddr& x) const noexcept {
            return TSa(x.Addr()).hash();
        }
    };

    struct TAddrEqual {
        inline bool operator()(const NAddr::IRemoteAddr& lhs, const NAddr::IRemoteAddr& rhs) const noexcept {
            return TSa(lhs.Addr()) == TSa(rhs.Addr());
        }
        inline bool operator()(const NAddr::IRemoteAddr& lhs, const TSa& rhs) const noexcept {
            return TSa(lhs.Addr()) == rhs;
        }
    };

    using TAddrSet = THashSet<NAddr::TAddrInfo, TAddrHash, TAddrEqual>;
    using TOpaqueAddrSet = THashSet<NAddr::TOpaqueAddr, TAddrHash, TAddrEqual>;

    struct TAddrIntrHashOps: public ::TCommonIntrHashOps {
        static inline bool EqualTo(const struct sockaddr* lhs, const struct sockaddr* rhs) {
            return TSa(lhs) == TSa(rhs);
        }

        static inline bool EqualTo(const NAddr::IRemoteAddr& lhs, const NAddr::IRemoteAddr& rhs) {
            return EqualTo(lhs.Addr(), rhs.Addr());
        }

        static inline bool EqualTo(const NAddr::IRemoteAddr& lhs, const sockaddr* rhs) {
            return EqualTo(lhs.Addr(), rhs);
        }

        static inline size_t Hash(const NAddr::IRemoteAddr& addr) noexcept {
            return TSa(addr.Addr()).hash();
        }

        static inline size_t Hash(const struct sockaddr* addr) noexcept {
            return TSa(addr).hash();
        }
    };

    // We assume that all agents run on machines with the same endianness
    static inline ui64 UnalignedLoad64(const void *src) {
        ui64 value;
        memcpy(&value, src, sizeof(value));
        return value;
    }

    static inline void UnalignedStore64(void *dst, ui64 value) {
        memcpy(dst, &value, sizeof(value));
    }


    class TLimitedLogger : public TNonCopyable {
    private:
        class TDevNull {
        public:
            TDevNull()
                : Logger(MakeHolder<TNullLogBackend>())
            {
            }

            template <class T>
            inline TLogElement operator<<(const T& t) const {
                return Logger << t;
            }

        private:
            mutable TLog Logger;
        };

    public:
        TLimitedLogger(TLog& logger, TDuration interval=TDuration::Seconds(1))
            : Logger(logger)
            , Interval(interval)
            , LastLogTime(TInstant::Zero())
        {
        }

        template <class T>
        inline TLogElement operator<<(const T& t) const {
            const auto now(TInstant::Now());
            if (LastLogTime + Interval < now) {
                LastLogTime = now;
                return Logger << t;
            } else {
                return Default<TDevNull>() << t;
            }
        }

    private:
        TLog& Logger;
        const TDuration Interval;
        mutable TInstant LastLogTime;
    };
}
