#include "service.h"

#include <infra/netmon/agent/common/generator.h>
#include <infra/netmon/agent/common/loop.h>
#include <infra/netmon/agent/common/utils.h>
#include <infra/netmon/agent/common/metrics.h>
#include <infra/netmon/agent/pollers/udp/handler.h>

#include <util/generic/maybe.h>

#include <array>

namespace NNetmon {
    namespace {
        const std::size_t MAX_QUEUE_SIZE = 100;
        const TDuration PACKET_TTL = TDuration::Seconds(3);
    }

    /*
     * Code based on library/cpp/coroutine/listener/listen.cpp.
     */
    class TEchoService::TImpl {
    public:
        inline TImpl(TLog& logger, TContExecutor* executor)
            : Logger_(logger)
            , Executor_(executor)
            , PacketPool_(TDefaultAllocator::Instance())
            , BufferPool_(TDefaultAllocator::Instance())
        {
        }

    private:
        class TSocketEchoService : public TIntrusiveListItem<TSocketEchoService> {
        public:
            using TListType = TIntrusiveListWithAutoDelete<TSocketEchoService, TDelete>;

            inline TSocketEchoService(TImpl* parent, NAddr::IRemoteAddrPtr addr)
                : Parent_(parent)
                , IOHandler_(Parent_->Logger_, std::move(addr))
                , Cont_(nullptr)
                , Loop_(Parent_->Executor_, this, "echo_socket")
            {
            }

            inline const NAddr::IRemoteAddr* Addr() const noexcept {
                return IOHandler_.Addr().Get();
            }

            inline void Start() {
                Loop_.Start();
            }

            inline void Run(TCont* cont) noexcept {
                Cont_ = cont;
                DoRun();
                Cont_ = nullptr;
            }

        private:
            using TMaybeIOStatus = TMaybe<TContIOStatus, NMaybe::TPolicyUndefinedFail>;

            inline void DoRun() noexcept {
                Y_VERIFY(Cont_ != nullptr);

                while (!Cont_->Cancelled()) {
                    try {
                        OneShot();
                    } catch(...) {
                        static TLimitedLogger logger(Parent_->Logger_);
                        logger << TLOG_WARNING << CurrentExceptionMessage() << Endl;
                    }
                }

                IOHandler_.Close();
            }

            inline void OneShot() {
                ui16 opFilter = 0;

                while (WriteQueue_.Size() < MAX_QUEUE_SIZE) {
                    TUdpPacket::TRef packet(TUdpPacket::Make(Parent_->PacketPool_, Parent_->BufferPool_));

                    auto ioStatus(IOHandler_.Read(*packet));
                    if (ioStatus) {
                        packet->Size() = ioStatus->Checked();
                        if (!packet->Size()) {
                            TUnistat::Instance().PushSignalUnsafe(ESignals::EchoTruncatedProbes, 1);
                        } else {
                            auto& stats = packet->Stats();
                            // TargetReceivedTime != 0 if packet was read from socket error queue.
                            // TODO: Do we need to retry sending of those packets?
                            // No retries for now for sake of simplicity (where to store retry counter is not obvious).
                            if (!stats.TargetReceivedTime) {
                                stats.TargetReceivedTime = IOHandler_.RcvdTime();
                                TUnistat::Instance().PushSignalUnsafe(ESignals::EchoReceivedProbes, 1);
                                if (packet->CongestionEncountered()) {
                                    PushSignal(EPushSignals::PacketCongested, 1);
                                }
                                if (stats.TargetReceivedTime > stats.SourceSentTime) {
                                    TUnistat::Instance().PushSignalUnsafe(
                                        ESignals::EchoFlightDelay, stats.TargetReceivedTime - stats.SourceSentTime);
                                }
                                WriteQueue_.PushBack(packet.Release());
                            }
                        }
                    } else {
                        // would block on read
                        opFilter |= CONT_POLL_READ;
                        break;
                    }
                }

                while (!WriteQueue_.Empty()) {
                    ui64 now(TInstant::Now().MicroSeconds());
                    auto packetDeadline(now - PACKET_TTL.MicroSeconds());

                    TUdpPacket::TRef packet(WriteQueue_.PopFront());
                    if (packet->Stats().TargetReceivedTime <= packetDeadline) {
                        TUnistat::Instance().PushSignalUnsafe(ESignals::EchoExpiredProbes, 1);
                        continue;
                    } else {
                        TUnistat::Instance().PushSignalUnsafe(
                            ESignals::EchoDispatchDelay, now - packet->Stats().TargetReceivedTime);
                    }

                    packet->Stats().TargetSentTime = now;
                    auto ioStatus(IOHandler_.Write(*packet));
                    if (ioStatus) {
                        auto written = ioStatus->Checked();
                        if (written != packet->Size()) {
                            TUnistat::Instance().PushSignalUnsafe(ESignals::EchoTruncatedProbes, 1);
                        }
                        auto& stats = packet->Stats();
                        stats.TargetRespTime = IOHandler_.SentTime();
                        TUnistat::Instance().PushSignalUnsafe(ESignals::EchoSentProbes, 1);
                        if (stats.TargetRespTime > stats.TargetReceivedTime) {
                            TUnistat::Instance().PushSignalUnsafe(
                                ESignals::EchoDispatchDelay, stats.TargetRespTime - stats.TargetReceivedTime);
                        }
                    } else {
                        // would block on write
                        WriteQueue_.PushFront(packet.Release());
                        opFilter |= CONT_POLL_WRITE;
                        break;
                    }
                }

                if (opFilter) {
                    IOHandler_.PollT(Cont_, opFilter, TDuration::Max());
                } else {
                    Cont_->SleepT(TDuration::Zero());
                }
            }

            const TImpl* const Parent_;
            TUdpIOHandler IOHandler_;
            TCont* Cont_;
            TUdpPacket::TListType WriteQueue_;

            TSimpleContLoop<TSocketEchoService, &TSocketEchoService::Run> Loop_;
        };

        class TServices : public TSocketEchoService::TListType {
        public:
            inline TIterator Find(const NAddr::IRemoteAddr& addr) {
                const TSa sa(addr.Addr());

                TIterator it = Begin();
                TIterator const end = End();
                while (it != end && sa != it->Addr()->Addr()) {
                    ++it;
                }

                return it;
            }
        };

    public:
        void SyncAddresses(const TVector<TNetworkAddress>& addresses) {
            TAddrSet target;
            for (auto& addr : addresses) {
                for (auto it(addr.Begin()); it != addr.End(); ++it) {
                    target.emplace(&*it);
                }
            }

            for (auto service(Services_.Begin()); service != Services_.End(); ) {
                auto it(target.find(TSa(service->Addr()->Addr())));
                if (it != target.end()) {
                    // address already exists
                    target.erase(it);
                    ++service;
                } else {
                    // collect the garbage
                    const auto* addr(service->Addr());
                    ++service;
                    Remove(*addr);
                }
            }

            bool failed = false;
            for (auto& addr : target) {
                try {
                    Register(addr);
                } catch(...) {
                    Logger_ << TLOG_WARNING << CurrentExceptionMessage() << Endl;
                    failed = true;
                }
            }

            if (failed) {
                ythrow yexception() << "can't bind to all given addresses";
            }
        }

    private:
        inline void Register(const NAddr::IRemoteAddr& addr) {
            const TSa sa(addr.Addr());

            switch (sa.Sa->sa_family) {
                case AF_INET: {
                    Services_.PushBack(new TSocketEchoService(this, MakeHolder<NAddr::TIPv4Addr>(*sa.In)));
                    break;
                }
                case AF_INET6: {
                    Services_.PushBack(new TSocketEchoService(this, MakeHolder<NAddr::TIPv6Addr>(*sa.In6)));
                    break;
                }
                default: {
                    ythrow yexception() << TStringBuf("unknown protocol");
                }
            }

            Services_.Back()->Start();
        }

        inline void Remove(const NAddr::IRemoteAddr& addr) {
            const TServices::TIterator it = Services_.Find(addr);
            if (it != Services_.End()) {
                delete &*it;
            }
        }

        TLog& Logger_;
        TContExecutor* Executor_;
        mutable TUdpPacket::TPool PacketPool_;
        mutable TOnDemandBuffer::TPool BufferPool_;
        TServices Services_;
    };

    TEchoService::TEchoService(TLog& logger, TContExecutor* executor)
        : Impl(MakeHolder<TImpl>(logger, executor))
    {
    }

    TEchoService::~TEchoService()
    {
    }

    void TEchoService::Stop() noexcept {
        Y_VERIFY(Impl);
        Impl.Destroy();
    }

    void TEchoService::SyncAddresses(const TVector<TNetworkAddress>& addresses) {
        Y_VERIFY(Impl);
        Impl->SyncAddresses(addresses);
    }
}
