#include "service.h"
#include "handler.h"

#include <infra/netmon/agent/common/generator.h>

#include <util/generic/xrange.h>

namespace NNetmon {
    class TIcmpPollerService::TImpl {
    private:
        class TProbeGenerator : public TSingleSocketProbeGenerator<TIcmpIOHandler>, public TIntrusiveHashItem<TProbeGenerator>, public TObjectFromPool<TProbeGenerator> {
        public:
            using TRef = THolder<TProbeGenerator>;
            using TMapType = TIntrusiveHashWithAutoDelete<TProbeGenerator, TOps>;

            template <typename... Args>
            static inline TRef Make(TPool& pool, Args&&... args) {
                return TRef(new (&pool) TProbeGenerator(std::forward<Args>(args)...));
            }

            inline TProbeGenerator(const TImpl* parent, const TProbeConfig& config, TLimitedLogger& logger, TContExecutor* executor, TIcmpIOHandler& handler, ui8 probeId)
                : TSingleSocketProbeGenerator(config, parent->Protocol(), logger, executor, handler, probeId)
                , Parent_(parent)
            {
            }

            bool ValidatePacket(const TIcmpPacket& received, TIcmpPacket& sent) noexcept override {
                Y_VERIFY(received.Stats().Seqno == sent.Stats().Seqno);
                sent.Stats().SourceReceivedTime = received.Stats().SourceReceivedTime;

                return true;
            }

            inline TIcmpPacket::TRef CreatePacket(ui16 seqno) noexcept override {
                TIcmpPacket::TRef packet(TIcmpPacket::Make(
                    Parent_->PacketPool_,
                    Parent_->BufferPool_,
                    *Config_.TargetAddr,
                    Config_.TypeOfService,
                    Config_.TimeToLive
                ));
                packet->Stats().ProbeId = Key_.second;
                packet->Stats().Seqno = seqno;
                packet->Stats().SourceSentTime = TInstant::Now().MicroSeconds();
                packet->Size() = Min(Config_.PacketSize, MAX_PACKET_LENGTH);
                return packet;
            }

        private:
            const TImpl* const Parent_;
        };

        class TIcmpSocket : public TIntrusiveHashItem<TIcmpSocket> {
        public:
            using TRef = THolder<TIcmpSocket>;

            struct TOps: public TAddrIntrHashOps {
                static inline const NAddr::IRemoteAddr& ExtractKey(const TIcmpSocket& obj) noexcept {
                    return *obj.IOHandler_.Addr();
                }
            };

            using TMapType = TIntrusiveHashWithAutoDelete<TIcmpSocket, TOps>;

            static TRef Make(const TImpl* parent, const NAddr::IRemoteAddrRef& addr) noexcept {
                try {
                    return MakeHolder<TIcmpSocket>(parent, addr);
                } catch(...) {
                    static TLimitedLogger logger(parent->Logger_);
                    logger << TLOG_WARNING << CurrentExceptionMessage() << Endl;
                }
                return nullptr;
            }

            TIcmpSocket(const TImpl* parent, const NAddr::IRemoteAddrRef& addr)
                : Parent_(parent)
                , IOHandler_(Parent_->Logger_, addr)
                , Cont_(nullptr)
                , Loop_(Parent_->Executor_, this, "imcp_socket")
            {
                Loop_.Start();
            }

            TBaseProbeGenerator<TIcmpIOHandler>* Attach(const TProbeConfig& config) {
                static TLimitedLogger logger(Parent_->Logger_);

                for (const auto probeId : xrange(TIcmpIOHandler::MaxProbeId())) {
                    TProbeGenerator::TRef generator(
                        TProbeGenerator::Make(Parent_->GeneratorPool_, Parent_, config, logger, Parent_->Executor_, IOHandler_, probeId)
                    );
                    if (!Generators_.Has(generator->Key())) {
                        generator->Generator().Subscribe([this] (TBaseProbeGenerator<TIcmpIOHandler>* generator) noexcept {
                            Generators_.Erase(TProbeGenerator::TOps::ExtractKey(*generator));
                        });
                        Generators_.Push(generator.Get());
                        generator->Generator().Start();
                        return generator.Release();
                    }
                }

                // only one generator with given address and probe id can exists
                return nullptr;
            }

        private:
            inline void Run(TCont* cont) noexcept {
                Cont_ = cont;

                while (!Cont_->Cancelled()) {
                    try {
                        OneShot();
                    } catch(...) {
                        static TLimitedLogger logger(Parent_->Logger_);
                        logger << TLOG_WARNING << CurrentExceptionMessage() << Endl;
                    }
                }

                // wait for running generators
                WaitForGenerators(Generators_, Cont_);

                Cont_ = nullptr;
            }

            inline void OneShot() {
                TIcmpPacket::TRef packet(TIcmpPacket::Make(Parent_->PacketPool_, Parent_->BufferPool_));

                while (!Cont_->Cancelled()) {
                    auto ioStatus(IOHandler_.Read(*packet));
                    if (ioStatus) {
                        packet->Size() = ioStatus->Checked();
                        if (!packet->Size()) {
                            ythrow yexception() << TStringBuf("no data was read");
                        } else {
                            DispatchPacket(*packet);
                            return;
                        }
                    } else {
                        // would block on read
                        IOHandler_.PollT(Cont_, CONT_POLL_READ, TDuration::Max());
                    }
                }
            }

            inline void DispatchPacket(const TIcmpPacket& packet) {
                const TProbeGenerator::TKey key{
                    MakeAtomicShared<NAddr::TOpaqueAddr>(packet.ClientAddrData()),
                    packet.Stats().ProbeId
                };
                auto it(Generators_.Find(key));
                if (it != Generators_.End()) {
                    it->Generator().OnPacketAvailable(packet);
                } else {
                    PushSignal(EPushSignals::PacketMartian, 1);
                    ythrow yexception() << TStringBuf("Unknown packet from ") << NAddr::PrintHost(*key.first)
                                        << TStringBuf(" received by ") << NAddr::PrintHost(*IOHandler_.Addr())
                                        << TStringBuf(", probe id ") << (ui32)key.second;
                }
            }

            const TImpl* const Parent_;
            TIcmpIOHandler IOHandler_;
            TCont* Cont_;

            TProbeGenerator::TMapType Generators_;

            TSimpleContLoop<TIcmpSocket, &TIcmpSocket::Run> Loop_;
        };

    public:
        inline TImpl(TLog& logger, TContExecutor* executor)
            : Logger_(logger)
            , Executor_(executor)
            , PacketPool_(TDefaultAllocator::Instance())
            , GeneratorPool_(TDefaultAllocator::Instance())
            , BufferPool_(TDefaultAllocator::Instance())
        {
        }

        TBaseProbeGenerator<TIcmpIOHandler>* Attach(const TProbeConfig& config) {
            if (config.SourceAddr->Addr()->sa_family != config.TargetAddr->Addr()->sa_family) {
                return nullptr;
            }

            auto it(Sockets_.Find(*config.SourceAddr));
            return it != Sockets_.End() ? it->Attach(config) : nullptr;
        }

        void ScheduleChecks(TCont* cont, const TVector<TProbeConfig>& configs, TVector<TProbeReport>& reports) {
            TProbeCollector<TImpl> collector(this, cont, configs);
            collector.Dump(reports);
        }

        void SyncAddresses(const TVector<TNetworkAddress>& addresses) {
            TAddrSet target;
            for (auto& addr : addresses) {
                for (auto it(addr.Begin()); it != addr.End(); ++it) {
                    target.emplace(&*it);
                }
            }

            TVector<NAddr::IRemoteAddrRef> keysToDelete;
            for (auto& sock : Sockets_) {
                const auto& key(TIcmpSocket::TOps::ExtractKey(sock));
                auto it(target.find(key));
                if (it.IsEnd()) {
                    keysToDelete.emplace_back(CopyRemoteAddr(key));
                } else {
                    target.erase(it);
                }
            }
            for (const auto& key : keysToDelete) {
                Sockets_.Erase(*key);
            }

            bool failed = false;
            for (auto& addr : target) {
                try {
                    TIcmpSocket::TRef pool(MakeHolder<TIcmpSocket>(this, CopyRemoteAddr(addr)));
                    Sockets_.Push(pool.Release());
                } catch(...) {
                    static TLimitedLogger logger(Logger_);
                    logger << TLOG_WARNING << CurrentExceptionMessage() << Endl;
                    failed = true;
                }
            }

            if (failed) {
                ythrow yexception() << "can't bind to all given addresses";
            }
        }

        inline EProtocol Protocol() const noexcept {
            return EProtocol::ICMP;
        }

    private:
        TLog& Logger_;
        TContExecutor* Executor_;

        mutable TIcmpPacket::TPool PacketPool_;
        mutable TProbeGenerator::TPool GeneratorPool_;
        mutable TOnDemandBuffer::TPool BufferPool_;

        TIcmpSocket::TMapType Sockets_;
    };

    TIcmpPollerService::TIcmpPollerService(TLog& logger, TContExecutor* executor)
        : Impl(MakeHolder<TImpl>(logger, executor))
    {
    }

    TIcmpPollerService::~TIcmpPollerService()
    {
    }

    void TIcmpPollerService::Stop() noexcept {
        Y_VERIFY(Impl);
        Impl.Destroy();
    }

    void TIcmpPollerService::SyncAddresses(const TVector<TNetworkAddress>& addresses) {
        Y_VERIFY(Impl);
        Impl->SyncAddresses(addresses);
    }

    void TIcmpPollerService::ScheduleChecks(TCont* cont, const TVector<TProbeConfig>& configs, TVector<TProbeReport>& reports) {
        Y_VERIFY(Impl);
        Impl->ScheduleChecks(cont, configs, reports);
    }
}
