#include "service.h"
#include "handler.h"

#include <infra/netmon/agent/common/generator.h>
#include <infra/netmon/agent/common/settings.h>

#include <util/generic/xrange.h>
#include <util/system/spinlock.h>

namespace NNetmon {
    enum LinkServiceType {
        BACKBONE = 0,
        FASTBONE = 1
    };

    class TLinkPollerService::TImpl {
    private:
        class TLinkService : public TIntrusiveListItem<TLinkService> {
        public:
            using TRef = THolder<TLinkService>;
            using TListType = TIntrusiveListWithAutoDelete<TLinkService, TDelete>;

            static constexpr double LINK_TIMEOUT_MS = 1.0;

            /*
                The ToS values of CS3, CS4 DSCP classes to backbone and
                CS1, CS2 DSCP classes to fastbone addresses
            */
            std::array<int, 2> TosValues_;

            enum EPollerState {
                SENDING = 0,
                RECEIVING = 1
            };

            struct TStatus {
                ui32 TypeOfService;
                ui64 Received = 0llu;
                ui64 Lost = 0llu;

                ui64 MaxHwRtt = 0llu; // us
                ui64 MaxSysRtt = 0llu; // us
                ui64 MaxUserRtt = 0llu; // us

                TStatus(ui32 tos = 0) : TypeOfService(tos) {}
            };

            TLinkService(const TImpl* parent,
                         LinkServiceType type,
                         const NAddr::IRemoteAddrRef& addr,
                         const NAddr::TIPv6Addr& srcIp)
                : Parent_(parent)
                , IOHandler_(Parent_->Logger_, addr, srcIp)
                , Cont_(nullptr)
                , Type_(type)
                , Loop_(Parent_->Executor_, this, "link_socket")
            {
                switch (Type_) {
                    case LinkServiceType::BACKBONE:
                        TosValues_ = {CS3, CS4};
                        break;
                    case LinkServiceType::FASTBONE:
                        TosValues_ = {CS1, CS2};
                        break;
                }
                for (auto tos: TosValues_) {
                    Statuses_.emplace_back(tos);
                }
                SyncAddresses();
                Loop_.Start();
            }

            inline const NAddr::IRemoteAddr* Addr() const noexcept {
                return IOHandler_.Addr().Get();
            }

            void ReportStatus(TVector<TProbeReport>& reports) {
                for (auto& status: Statuses_) {
                    // No packets were sent
                    if (!status.Received && !status.Lost) {
                        continue;
                    }

                    auto dstIp = CopyRemoteAddr(*DstIp_);

                    double rtt = -1.0;
                    if (status.Received) {
                        if (status.MaxHwRtt) {
                            rtt = status.MaxHwRtt;
                        } else if (status.MaxSysRtt) {
                            rtt = status.MaxSysRtt;
                        } else {
                            rtt = status.MaxUserRtt;
                        }
                    }

                    reports.emplace_back(
                            1, /* NOC_SLA_PROBE */
                            EProtocol::LINK_POLLER,
                            dstIp, /* srcIp may be set from options, but dstIp is always real host IP */
                            dstIp,
                            status.TypeOfService,
                            status.Received,
                            0,
                            0,
                            status.Lost,
                            0,
                            rtt,
                            nullptr,
                            false,
                            TMaybe<NAddr::TOpaqueAddr>(Nothing())
                    );

                    status.Received = 0;
                    status.Lost = 0;

                    status.MaxHwRtt = 0.0;
                    status.MaxSysRtt = 0.0;
                    status.MaxUserRtt = 0.0;
                }
            }

            inline void SyncAddresses() {
                SrcPort_ = Parent_->SrcPort_;
                DstPort_ = Parent_->DstPort_;
                switch (Type_) {
                    case LinkServiceType::BACKBONE:
                        SrcMac_ = Parent_->BbSrcMac_.Get();
                        DstMac_ = Parent_->BbDstMac_.Get();
                        SrcIp_ = Parent_->BbSrcIp_.Get();
                        DstIp_ = Parent_->BbDstIp_.Get();
                        break;
                    case LinkServiceType::FASTBONE:
                        SrcMac_ = Parent_->FbSrcMac_.Get();
                        DstMac_ = Parent_->FbDstMac_.Get();
                        SrcIp_ = Parent_->FbSrcIp_.Get();
                        DstIp_ = Parent_->FbDstIp_.Get();
                        break;
                }
            }

        private:
            inline void Run(TCont* cont) noexcept {
                Cont_ = cont;

                while (!Cont_->Cancelled()) {
                    try {
                        OneShot();
                    } catch (...) {
                        static TLimitedLogger logger(Parent_->Logger_);
                        logger << TLOG_WARNING << CurrentExceptionMessage() << Endl;
                    }
                }

                Cont_ = nullptr;
            }

            inline void OneShot() {
                TLinkPacket::TRef txPacket;
                TLinkPacket::TRef rxPacket;

                int state = EPollerState::SENDING;
                TInstant deadline, epollDeadline;
                ui16 opFilter = 0;
                TMaybeIOStatus ioStatus;
                ui64 probeId = 0llu;
                int indexTos, tos;

                const bool useHwTimestamps = TSettings::Get()->GetUseHwTimestamps();

                while (!Cont_->Cancelled()) {
                    if (DstMac_ &&
                        ::memcmp(
                                IOHandler_.Addr()->Addr(),
                                DstMac_->Addr(),
                                DstMac_->Len()
                        )) {

                        IOHandler_.Reopen(CopyRemoteAddr(*DstMac_), *SrcIp_);
                        probeId = 0llu;
                    }

                    indexTos = probeId % TosValues_.size();
                    tos = TosValues_[indexTos];

                    switch (state) {
                        case EPollerState::SENDING:
                            txPacket = TLinkPacket::Make(
                                Parent_->PacketPool_, Parent_->BufferPool_,
                                *SrcIp_, SrcPort_, *SrcMac_,
                                *DstIp_, DstPort_, *DstMac_,
                                tos
                            );
                            txPacket->Stats().ProbeId = probeId;
                            ioStatus = IOHandler_.Write(*txPacket);
                            if (ioStatus &&
                                !ioStatus->Status() &&
                                ioStatus->Processed() == txPacket->Size()) {
                                TUnistat::Instance().PushSignalUnsafe(ESignals::PacketSent, 1);
                                auto now = TInstant::Now();
                                if (LastSentTime) {
                                    PushSignal(EPushSignals::LinkSendInterval, (now - LastSentTime).MilliSeconds());
                                }
                                LastSentTime = now;
                                opFilter |= CONT_POLL_READ;
                                state = EPollerState::RECEIVING;
                            } else {
                                opFilter = 0;
                                Statuses_[indexTos].Lost += 1;
                                probeId += 1;
                                state = EPollerState::SENDING;
                            }

                            deadline = TInstant::Now() + TDuration::MilliSeconds(100);
                            epollDeadline = TInstant::Now() + TDuration::MilliSeconds(1);
                            break;

                        case EPollerState::RECEIVING:
                            if (!rxPacket) {
                                rxPacket = TLinkPacket::Make(
                                    Parent_->PacketPool_, Parent_->BufferPool_,
                                    *SrcIp_, SrcPort_, *SrcMac_,
                                    *DstIp_, DstPort_, *DstMac_,
                                    tos
                                );
                                rxPacket->UserTimestamps().SentTime = txPacket->UserTimestamps().SentTime;
                            }

                            if (!rxPacket->Valid()) {
                                do {
                                    ioStatus = IOHandler_.Read(*rxPacket);
                                    rxPacket->Validate(probeId);
                                } while (ioStatus.Defined() && !rxPacket->Valid());
                            }

                            if ((useHwTimestamps && !rxPacket->HwTimestamps().SentTime) ||
                                !rxPacket->SysTimestamps().SentTime) {
                                do {
                                    ioStatus = IOHandler_.GetTxTs(probeId, *rxPacket);
                                } while (ioStatus.Defined());
                            }

                            if (TInstant::Now() >= deadline) {
                                Statuses_[indexTos].Lost += 1;
                                opFilter = 0;
                                probeId += 1;
                                state = EPollerState::SENDING;
                                rxPacket = nullptr;
                            } else if (rxPacket->Valid() &&
                                       (!useHwTimestamps ||
                                        rxPacket->HwTimestamps().SentTime && rxPacket->HwTimestamps().RecvTime) &&
                                       rxPacket->SysTimestamps().SentTime && rxPacket->SysTimestamps().RecvTime) {
                                ui64 rtt;

                                if (useHwTimestamps) {
                                    rtt = Max(rxPacket->HwTimestamps().RecvTime - rxPacket->HwTimestamps().SentTime, 0ul);
                                    PushSignal(EPushSignals::LinkRttHw, rtt);
                                    Statuses_[indexTos].MaxHwRtt = Max(Statuses_[indexTos].MaxHwRtt, rtt);
                                }

                                rtt = Max(rxPacket->SysTimestamps().RecvTime - rxPacket->SysTimestamps().SentTime, 0ul);
                                PushSignal(EPushSignals::LinkRttSys, rtt);
                                Statuses_[indexTos].MaxSysRtt = Max(Statuses_[indexTos].MaxSysRtt, rtt);

                                rtt = Max(rxPacket->UserTimestamps().RecvTime - rxPacket->UserTimestamps().SentTime,
                                          0ul);
                                PushSignal(EPushSignals::LinkRttUser, rtt);
                                Statuses_[indexTos].MaxUserRtt = Max(Statuses_[indexTos].MaxUserRtt, rtt);

                                Statuses_[indexTos].Received += 1;
                                opFilter = 0;
                                probeId += 1;
                                state = EPollerState::SENDING;
                                rxPacket = nullptr;
                            } else {
                                opFilter = CONT_POLL_READ;
                            }
                            break;
                    }

                    /*
                        Avoid yielding precious execution context
                        on critical path while waiting for switch reply
                    */
                    auto now = TInstant::Now();
                    if (state == EPollerState::RECEIVING && now < epollDeadline) {
                        auto timeout = TDuration::MilliSeconds(5 * LINK_TIMEOUT_MS);
                        IOHandler_.PollDWithoutYield(timeout, deadline);
                    } else {
                        if (opFilter) {
                            IOHandler_.PollD(Cont_, opFilter, deadline);
                        } else {
                            Cont_->SleepD(deadline);
                        }
                    }
                }
            }

            const TImpl* const Parent_;
            TLinkIOHandler IOHandler_;
            TCont* Cont_;
            LinkServiceType Type_;

            TInstant LastSentTime;

            TSimpleContLoop<TLinkService, &TLinkService::Run> Loop_;
            TVector<TLinkService::TStatus> Statuses_;

            NAddr::TOpaqueAddr* SrcMac_;
            NAddr::TIPv6Addr* SrcIp_;
            ui16 SrcPort_;
            NAddr::TOpaqueAddr* DstMac_;
            NAddr::TIPv6Addr* DstIp_;
            ui16 DstPort_;
        };

    public:
        inline TImpl(TLog& logger, TContExecutor* executor)
            : Logger_(logger)
            , Executor_(executor)
            , PacketPool_(TDefaultAllocator::Instance())
            , BufferPool_(TDefaultAllocator::Instance())
        {
        }

        void SyncAddresses(ui16 srcPort, ui16 dstPort,
                           int bbIfIndex, const TString& bbSrcMac, const TString& bbDstMac,
                           const TString& bbSrcIp, const TString& bbDstIp,
                           int fbIfIndex, const TString& fbSrcMac, const TString& fbDstMac,
                           const TString& fbSrcIp, const TString& fbDstIp) {
            SrcPort_ = srcPort;
            DstPort_ = dstPort;
            BbSrcMac_ = ParseMac(bbSrcMac, bbIfIndex);
            BbDstMac_ = ParseMac(bbDstMac, bbIfIndex);
            BbSrcIp_ = ParseIp(bbSrcIp);
            BbDstIp_ = ParseIp(bbDstIp);
            FbSrcMac_ = ParseMac(fbSrcMac, fbIfIndex);
            FbDstMac_ = ParseMac(fbDstMac, fbIfIndex);
            FbSrcIp_ = ParseIp(fbSrcIp);
            FbDstIp_ = ParseIp(fbDstIp);
            if (BbSrcMac_ && BbDstMac_ && BbSrcIp_ && BbDstIp_) {
                if (!BbLinkService_) {
                    BbLinkService_ = MakeHolder<TLinkService>(this,
                                                              LinkServiceType::BACKBONE,
                                                              CopyRemoteAddr(*BbDstMac_.Get()),
                                                              *BbSrcIp_);
                }
                BbLinkService_->SyncAddresses();
            }
            if (FbSrcMac_ && FbDstMac_ && FbSrcIp_ && FbDstIp_) {
                if (!FbLinkService_) {
                    FbLinkService_ = MakeHolder<TLinkService>(this,
                                                              LinkServiceType::FASTBONE,
                                                              CopyRemoteAddr(*FbDstMac_.Get()),
                                                              *FbSrcIp_);
                }
                FbLinkService_->SyncAddresses();
            }
        }

        void ReportStatus(TVector<TProbeReport>& reports) {
            if (BbLinkService_) {
                BbLinkService_->ReportStatus(reports);
            }
            if (FbLinkService_) {
                FbLinkService_->ReportStatus(reports);
            }
        }

    private:
        inline THolder<NAddr::TIPv6Addr> ParseIp(const TString& s) {
            sockaddr_in6 sin6;
            ::memset(&sin6, 0, sizeof(sockaddr_in6));
            sin6.sin6_family = AF_INET6;
            if (::inet_pton(AF_INET6, s.c_str(), &sin6.sin6_addr) != 1) {
                return nullptr;
            }
            return MakeHolder<NAddr::TIPv6Addr>(sin6);
        }

        inline THolder<NAddr::TOpaqueAddr> ParseMac(const TString& s, int ifIndex) {
            ui8 mac[6];
            if (::sscanf(s.c_str(),
                         "%02hhx:%02hhx:%02hhx:%02hhx:%02hhx:%02hhx",
                         &mac[0], &mac[1],
                         &mac[2], &mac[3],
                         &mac[4], &mac[5]) != 6) {
                return nullptr;
            }

            TLlAddress ll(ifIndex, mac);
            return MakeHolder<NAddr::TOpaqueAddr>(&ll);
        }

        TLog& Logger_;
        TContExecutor* Executor_;

        mutable TLinkPacket::TPool PacketPool_;
        mutable TOnDemandBuffer::TPool BufferPool_;

        ui16 SrcPort_;
        ui16 DstPort_;
        THolder<NAddr::TOpaqueAddr> BbSrcMac_;
        THolder<NAddr::TOpaqueAddr> BbDstMac_;
        THolder<NAddr::TIPv6Addr> BbSrcIp_;
        THolder<NAddr::TIPv6Addr> BbDstIp_;
        THolder<NAddr::TOpaqueAddr> FbSrcMac_;
        THolder<NAddr::TOpaqueAddr> FbDstMac_;
        THolder<NAddr::TIPv6Addr> FbSrcIp_;
        THolder<NAddr::TIPv6Addr> FbDstIp_;
        THolder<TLinkService> BbLinkService_;
        THolder<TLinkService> FbLinkService_;
    };

    TLinkPollerService::TLinkPollerService(TLog& logger, TContExecutor* executor)
        : Impl(MakeHolder<TImpl>(logger, executor))
    {
    }

    TLinkPollerService::~TLinkPollerService()
    {
    }

    void TLinkPollerService::Stop() noexcept {
        Y_VERIFY(Impl);
        Impl.Destroy();
    }

    void TLinkPollerService::SyncAddresses(ui16 srcPort, ui16 dstPort,
                                           int bbIfIndex, const TString& bbSrcMac, const TString& bbDstMac,
                                           const TString& bbSrcIp, const TString& bbDstIp,
                                           int fbIfIndex, const TString& fbSrcMac, const TString& fbDstMac,
                                           const TString& fbSrcIp, const TString& fbDstIp) {
        Y_VERIFY(Impl);
        Impl->SyncAddresses(srcPort, dstPort,
                            bbIfIndex, bbSrcMac, bbDstMac, bbSrcIp, bbDstIp,
                            fbIfIndex, fbSrcMac, fbDstMac, fbSrcIp, fbDstIp);
    }

    void TLinkPollerService::ReportStatus(TVector<TProbeReport>& reports) {
        Y_VERIFY(Impl);
        return Impl->ReportStatus(reports);
    }
}
