#pragma once

#include <infra/netmon/agent/common/io.h>
#include <infra/netmon/agent/common/packet.h>

#include <util/generic/xrange.h>
#include <util/stream/mem.h>

#include <netinet/tcp.h>

namespace NNetmon {
    namespace {
        const ui32 NETMON_TCP_MAGIC = 0x72557255;
    }

    class TTcpPacket : public TPacket, public TIntrusiveListItem<TTcpPacket>, public TObjectFromPool<TTcpPacket> {
    public:
        enum class EType {
            Request = 0,
            Reply = 1
        };

        using TRef = THolder<TTcpPacket>;
        using TListType = TIntrusiveListWithAutoDelete<TTcpPacket, TDelete>;

        using TPacket::TPacket;

        template <typename... Args>
        static inline TRef Make(TPool& pool, Args&&... args) {
            return TRef(new (&pool) TTcpPacket(std::forward<Args>(args)...));
        }

        TTcpPacket(TOnDemandBuffer::TPool& pool, const NAddr::IRemoteAddr& addr,
                   ui16 dstPort, ui16 srcPort, int tos = 0, i32 timeToLive=-1)
            : TPacket(pool, addr, tos, timeToLive)
            , DstPort_(dstPort)
            , SrcPort_(srcPort)
            , Type_(TTcpPacket::EType::Request)
        {
        }

        inline void Fill() noexcept {
            tcphdr* const tcp = reinterpret_cast<tcphdr*>(Data());
            std::size_t offset = 0;

            if (Size() < sizeof(tcphdr)) {
                Size() = sizeof(tcphdr);
            }

            ::memset(tcp, 0, sizeof(tcphdr));

            tcp->source = HostToInet(SrcPort_);
            tcp->dest = HostToInet(DstPort_);
            tcp->doff = 0x5; // 5 qwords, simple tcp header
            tcp->res1 = 0x0;
            tcp->res2 = 0x0;
            tcp->window = 0x0;

            tcp->seq = HostToInet(NETMON_TCP_MAGIC);
            tcp->ack_seq = Type_ == TTcpPacket::EType::Reply ? HostToInet(NETMON_TCP_MAGIC) : 0;

            tcp->rst = 1;

            if (Type_ == TTcpPacket::EType::Reply) {
                tcp->ack = 1;
            } else {
                tcp->syn = 1;
            }

            tcp->urg_ptr = 0x0;
            tcp->check = 0x0;

            offset += sizeof(tcphdr);
            ui8* body = reinterpret_cast<ui8*>(Data()) + offset;

            if (Length_ > offset) {
                TMemoryOutput stream(body, Length_ - offset);
                try {
                    Stats_.Save(&stream);
                    stream.Finish();

                    offset += static_cast<std::size_t>(std::distance(
                                                       Buf_.Data() + offset, stream.Buf()));
                    Length_ = Max(offset, Length_);
                } catch (...) {
                    Truncated_ = true;
                }

                FillData();
            }
        }

        inline void FillChecksum(ui16 csum) noexcept {
            Y_VERIFY(Size() >= sizeof(tcphdr));
            tcphdr* const tcp = reinterpret_cast<tcphdr*>(Data());
            tcp->check = csum;
        }

        inline ui16 DstPort() const noexcept {
            return DstPort_;
        }

        inline ui16& DstPort() noexcept {
            return DstPort_;
        }

        inline ui16 SrcPort() const noexcept {
            return SrcPort_;
        }

        inline ui16& SrcPort() noexcept {
            return SrcPort_;
        }

        inline TTcpPacket::EType Type() const noexcept {
            return Type_;
        }

        inline TTcpPacket::EType& Type() noexcept {
            return Type_;
        }

        virtual std::size_t GetDataOffset() const override {
            return TPacket::GetDataOffset() + sizeof(tcphdr);
        }

    private:
        ui16 DstPort_;
        ui16 SrcPort_;

        EType Type_;
    };

    class TTcpIOHandler : public TNonCopyable {
    public:
        using TProbeIdType = ui8;
        using TPacket = TTcpPacket;

        TTcpIOHandler(TLog& logger, const NAddr::IRemoteAddrRef& addr, ui16 minPort, ui16 maxPort);
        ~TTcpIOHandler();

        const NAddr::IRemoteAddrRef& Addr() const noexcept;

        TMaybeIOStatus Read(TTcpPacket& packet) noexcept;
        TMaybeIOStatus Write(TTcpPacket& packet) noexcept;

        void PollD(TCont* cont, ui16 opFilter, const TInstant& deadline);
        void PollT(TCont* cont, ui16 opFilter, const TDuration& duration);

        static ui32 MaxProbeId() noexcept;

        void Close() noexcept;

        inline ui16 MinPort() const noexcept {
            return MinPort_;
        }

        inline ui16 MaxPort() const noexcept {
            return MaxPort_;
        }

    private:
        class TImpl;
        THolder<TImpl> Impl;

        ui16 MinPort_;
        ui16 MaxPort_;
    };
}
