#include "handler.h"

#include <infra/netmon/agent/common/utils.h>

namespace NNetmon {
    TUdpIOHandler::TUdpIOHandler(TLog& logger, const NAddr::IRemoteAddrRef& addr)
        : Logger_(logger)
        , Socket_(::socket(addr->Addr()->sa_family, SOCK_DGRAM, 0))
        , Addr_(addr)
        , Handler_(Logger_, Socket_)
    {
        if (Socket_ == INVALID_SOCKET) {
            ythrow TSystemError() << TStringBuf("can't create socket for ")
                                  << NAddr::PrintHostAndPort(*Addr_);
        }

        // don't enable SO_REUSEADDR because udp can be bound to same port in this case
        // see https://lwn.net/Articles/542728/ for details

        FixIPv6ListenSocket(Socket_);
        SetNonBlock(Socket_);

#ifdef SO_RXQ_OVFL // since Linux 2.6.33
        CheckedSetSockOpt(Socket_, SOL_SOCKET, SO_RXQ_OVFL, 1, "overflow detection");
#endif

#ifdef SO_TIMESTAMPNS // only on Linux
        CheckedSetSockOpt(Socket_, SOL_SOCKET, SO_TIMESTAMPNS, 1, "kernel timestamps");
#endif

        if (Addr_->Addr()->sa_family == AF_INET6) {
            CheckedSetSockOpt(Socket_, IPPROTO_IPV6, IPV6_RECVTCLASS, 1, "ipv6 recv tos");
        }

        DoNotFragment(Addr_->Addr()->sa_family, Socket_);
        EnableErrorQueue(Addr_->Addr()->sa_family, Socket_);

        if (bind(Socket_, Addr_->Addr(), Addr_->Len()) < 0) {
            ythrow TSystemError() << TStringBuf("bind to ")
                                  << NAddr::PrintHostAndPort(*Addr_)
                                  << TStringBuf(" failed");
        }

        SocketAddr_ = NAddr::GetSockAddr(Socket_);
    }

    TMaybeIOStatus TUdpIOHandler::Read(TUdpPacket& packet) noexcept {
        auto result(Handler_.Read(packet));

        if (result.Defined() && result->Processed()) {
            packet.LoadStats();
        }

        return result;
    }

    TMaybeIOStatus TUdpIOHandler::Write(TUdpPacket& packet) noexcept {
        const auto timeToLive(packet.TimeToLive());
        if (timeToLive != LastTimeToLive) {
            SetTimeToLive(Addr_->Addr()->sa_family, Socket_, timeToLive);
            LastTimeToLive = timeToLive;
        }
        packet.FillData();

        packet.SaveStats();
        return Handler_.Write(packet);
    }

    void TUdpIOHandler::Close() noexcept {
        try {
            Socket_.Close();
        } catch(...) {
        }
    }

    ui16 TUdpIOHandler::SocketPort() const noexcept {
        return ExtractPort(SocketAddr_);
    }
}
