#pragma once

#include <infra/netmon/agent/common/io.h>
#include <infra/netmon/agent/common/packet.h>

#include <util/stream/mem.h>

namespace NNetmon {
    class TUdpPacket : public TPacket, public TIntrusiveListItem<TUdpPacket>, public TObjectFromPool<TUdpPacket> {
    public:
        using TRef = THolder<TUdpPacket>;
        using TListType = TIntrusiveListWithAutoDelete<TUdpPacket, TDelete>;

        using TPacket::TPacket;

        template <typename... Args>
        static inline TRef Make(TPool& pool, Args&&... args) {
            return TRef(new (&pool) TUdpPacket(std::forward<Args>(args)...));
        }

        inline void LoadStats() noexcept {
            // FIXME: properly check for result
            TMemoryInput stream(Buf_.Data(), Length_);
            try {
                Stats_.Load(&stream);
            } catch(...) {
                Truncated_ = true;
            }
        }

        inline void SaveStats() noexcept {
            // FIXME: properly check for result
            TMemoryOutput stream(Buf_.Data(), Buf_.Capacity());
            try {
                Stats_.Save(&stream);
                stream.Finish();
                Length_ = Max(static_cast<std::size_t>(std::distance(Buf_.Data(), stream.Buf())), Length_);
            } catch(...) {
                Truncated_ = true;
            }
        }
    };

    class TUdpIOHandler {
    public:
        using TProbeIdType = ui32;
        using TPacket = TUdpPacket;

        TUdpIOHandler(TLog& logger, const NAddr::IRemoteAddrRef& addr);

        TMaybeIOStatus Read(TUdpPacket& packet) noexcept;
        TMaybeIOStatus Write(TUdpPacket& packet) noexcept;

        inline void PollD(TCont* cont, ui16 opFilter, const TInstant& deadline) {
            Handler_.PollD(cont, opFilter, deadline);
        }
        inline void PollT(TCont* cont, ui16 opFilter, const TDuration& duration) {
            Handler_.PollT(cont, opFilter, duration);
        }

        void Close() noexcept;

        inline ui64 RcvdTime() const noexcept {
            return Handler_.RcvdTime();
        }
        inline ui64 SentTime() const noexcept {
            return Handler_.SentTime();
        }

        inline const NAddr::IRemoteAddrRef& Addr() const noexcept {
            return Addr_;
        }
        inline const NAddr::IRemoteAddrRef& SocketAddr() const noexcept {
            return SocketAddr_;
        }

        ui16 SocketPort() const noexcept;

        inline operator SOCKET() const noexcept {
            return Socket_;
        }

    private:
        TLog& Logger_;
        TSocketHolder Socket_;
        NAddr::IRemoteAddrRef Addr_;
        NAddr::IRemoteAddrRef SocketAddr_;
        TIOHandler Handler_;
        i32 LastTimeToLive = -1;
    };
}
