#include "service.h"
#include "handler.h"

#include <infra/netmon/agent/common/generator.h>
#include <infra/netmon/agent/common/settings.h>

#include <library/cpp/coroutine/engine/events.h>
#include <library/cpp/histogram/hdr/histogram.h>

#include <util/random/random.h>
#include <util/generic/deque.h>

namespace NNetmon {
    class TUdpPollerService::TImpl {
    private:
        class TUdpSocket;
        class TUdpSocketPool;

        class TIOStrategy {
        public:
            virtual ~TIOStrategy() = default;
            virtual TUdpSocket& GetSocket() noexcept = 0;
        };

        class TSingleSocketIOStrategy : public TIOStrategy {
        public:
            TSingleSocketIOStrategy(TUdpSocket& socket)
                : Socket_(socket)
            { }

            TUdpSocket& GetSocket() noexcept override {
                return Socket_;
            }

        private:
            TUdpSocket& Socket_;
        };

        class TMultiSocketIOStrategy : public TIOStrategy {
        public:
            TMultiSocketIOStrategy(TUdpSocketPool& socketPool)
                : SocketPool_(socketPool)
            { }

            TUdpSocket& GetSocket() noexcept override {
                return *SocketPool_.GetNextSocket();
            }

        private:
            TUdpSocketPool& SocketPool_;
        };

        class TProbeGenerator : public TBaseProbeGenerator<TUdpIOHandler>, public TIntrusiveHashItem<TProbeGenerator>, public TObjectFromPool<TProbeGenerator> {
        public:
            using TRef = THolder<TProbeGenerator>;
            using TMapType = TIntrusiveHashWithAutoDelete<TProbeGenerator, TOps>;

            template <typename... Args>
            static inline TRef Make(TPool& pool, Args&&... args) {
                return TRef(new (&pool) TProbeGenerator(std::forward<Args>(args)...));
            }

            inline TProbeGenerator(const TImpl* parent, const TProbeConfig& config, TLimitedLogger& logger, TContExecutor* executor, TIOStrategy& ioStrategy)
                : TBaseProbeGenerator(config, parent->Protocol(), logger, executor, RandomNumber<TUdpIOHandler::TProbeIdType>())
                , Parent_(parent)
                , IOStrategy_(ioStrategy)
            { }

            ~TProbeGenerator() {
                for (auto* socket : Sockets_) {
                    socket->UnregisterGenerator(this);
                }
            }

            bool ValidatePacket(const TUdpPacket& received, TUdpPacket& sent) noexcept override {
                if (received.Truncated()) {
                    return false;
                }

                Y_VERIFY(received.Stats().Seqno == sent.Stats().Seqno);

                if (received.Stats().Signature != Parent_->Signature_) {
                    return false;
                }

                if (received.Stats() != sent.Stats()) {
                    return false;
                }

                TPacket::TStats::Copy(received.Stats(), sent.Stats());

                return true;
            }

            TUdpPacket::TRef CreatePacket(ui16 seqno) noexcept override {
                TUdpPacket::TRef packet(TUdpPacket::Make(
                    Parent_->PacketPool_,
                    Parent_->BufferPool_,
                    *Config_.TargetAddr,
                    Config_.TypeOfService,
                    Config_.TimeToLive
                ));

                auto& stats = packet->Stats();
                stats.Signature = Parent_->Signature_;
                stats.ProbeId = ProbeId_;
                stats.SourceSentTime = TInstant::Now().MicroSeconds();
                stats.Seqno = seqno;

                packet->Size() = Min(Config_.PacketSize, MAX_PACKET_LENGTH);
                return packet;
            }

            TUdpIOHandler& GetIOHandler() noexcept override {
                auto& socket = IOStrategy_.GetSocket();
                socket.RegisterGenerator(this);
                Sockets_.insert(&socket);
                return socket.GetIOHandler();
            }

        private:
            const TImpl* const Parent_;
            TIOStrategy& IOStrategy_;
            THashSet<TUdpSocket*> Sockets_;
        };

        class TUdpSocket : public TIntrusiveListItem<TUdpSocket> {
        public:
            using TRef = THolder<TUdpSocket>;
            using TListType = TIntrusiveListWithAutoDelete<TUdpSocket, TDelete>;

            TUdpSocket(const TImpl* parent, const NAddr::IRemoteAddrRef& addr)
                : Parent_(parent)
                , IOHandler_(Parent_->Logger_, addr)
                , Cont_(nullptr)
                , IOStrategy_(*this)
                , Loop_(Parent_->Executor_, this, "udp_socket")
            {
                Loop_.Start();
            }

            void WaitForGenerators(TCont* waiter) {
                if (Cont_) {
                    // this method should allow existing generators to finish their work
                    while (!Generators_.empty()) {
                        auto generator = Generators_.begin();
                        if (generator->second->Generator().Cancelled()) {
                            Generators_.erase(generator);
                            continue;
                        }

                        generator->second->Generator().Wait(waiter);
                    }
                }
            }

            void Wait(TCont* waiter) {
                if (Cont_) {
                    IOHandler_.Close();
                    // join socket itself
                    Cont_->Cancel();
                    waiter->Join(Cont_);
                }
            }

            inline void RegisterGenerator(TBaseProbeGenerator<TUdpIOHandler>* generator) noexcept {
                TWaitingGeneratorMap::insert_ctx ctx = nullptr;
                const auto& key = generator->Key();
                if (!Generators_.contains(key, ctx)) {
                    Generators_.emplace_direct(ctx, key, generator);
                }
            }
            inline void UnregisterGenerator(TBaseProbeGenerator<TUdpIOHandler>* generator) noexcept {
                Generators_.erase(generator->Key());
            }

            inline ui16 GetPort() noexcept {
                return IOHandler_.SocketPort();
            }
            inline TIOStrategy& GetIOStrategy() noexcept {
                return IOStrategy_;
            }
            inline TUdpIOHandler& GetIOHandler() noexcept {
                return IOHandler_;
            }

        private:
            inline void Run(TCont* cont) noexcept {
                Cont_ = cont;

                while (!Cont_->Cancelled()) {
                    try {
                        OneShot();
                    } catch(...) {
                        static TLimitedLogger logger(Parent_->Logger_);
                        logger << TLOG_WARNING << CurrentExceptionMessage() << Endl;
                    }
                }

                Cont_ = nullptr;
            }

            inline void OneShot() {
                TUdpPacket::TRef packet(TUdpPacket::Make(Parent_->PacketPool_, Parent_->BufferPool_));

                while (!Cont_->Cancelled()) {
                    auto ioStatus(IOHandler_.Read(*packet));
                    if (ioStatus) {
                        packet->Size() = ioStatus->Checked();
                        if (!packet->Size()) {
                            ythrow yexception() << TStringBuf("no data was read");
                        } else {
                            packet->Stats().SourceReceivedTime = IOHandler_.RcvdTime();
                            DispatchPacket(*packet);
                            return;
                        }
                    } else {
                        // would block on read
                        IOHandler_.PollT(Cont_, CONT_POLL_READ, TDuration::Max());
                    }
                }
            }

            inline void DispatchPacket(const TUdpPacket& packet) {
                const TProbeGenerator::TKey key{
                    MakeAtomicShared<NAddr::TOpaqueAddr>(packet.ClientAddrData()),
                    packet.Stats().ProbeId
                };
                auto it(Generators_.find(key));
                if (it != Generators_.end()) {
                    it->second->Generator().OnPacketAvailable(packet);
                } else {
                    PushSignal(EPushSignals::PacketMartian, 1);
                    ythrow yexception() << TStringBuf("Unknown packet from ") << NAddr::PrintHostAndPort(*key.first)
                                        << TStringBuf(" received by ") << NAddr::PrintHostAndPort(*IOHandler_.SocketAddr())
                                        << TStringBuf(", probe id ") << key.second;
                }
            }

            const TImpl* const Parent_;
            TUdpIOHandler IOHandler_;
            TCont* Cont_;

            using TWaitingGeneratorMap = THashMap<TProbeGenerator::TKey,
                                                  TBaseProbeGenerator<TUdpIOHandler>*,
                                                  TProbeGenerator::THash,
                                                  TProbeGenerator::TEqualKey>;
            TWaitingGeneratorMap Generators_;

            TSingleSocketIOStrategy IOStrategy_;

            TSimpleContLoop<TUdpSocket, &TUdpSocket::Run> Loop_;
        };

        class TUdpSocketPool : public TIntrusiveHashItem<TUdpSocketPool> {
        public:
            using TRef = THolder<TUdpSocketPool>;

            struct TOps: public TAddrIntrHashOps {
                static inline const NAddr::IRemoteAddr& ExtractKey(const TUdpSocketPool& obj) noexcept {
                    return *obj.Addr_;
                }
            };

            using TMapType = TIntrusiveHashWithAutoDelete<TUdpSocketPool, TOps>;

            TUdpSocketPool(const TImpl* parent, const NAddr::IRemoteAddrRef& addr, std::size_t socketCount)
                : Parent_(parent)
                , Addr_(addr)
                , SocketCount_(socketCount)
                , Cont_(nullptr)
                , LastUsedSocket_(ActiveSockets_.End())
                , IOStrategy_(*this)
                , Loop_(Parent_->Executor_, this, "udp_gc")
            {
                Initialize();
                Loop_.Start();
            }

            inline TUdpSocket* GetSocketByPort(const TProbeConfig& config, ui16 port) {
                for (auto& socket : ActiveSockets_) {
                    if (socket.GetPort() == port) {
                        return &socket;
                    }
                }
                for (auto& socket : DyingSockets_) {
                    if (socket.GetPort() == port) {
                        return &socket;
                    }
                }

                // no socket found, create a new one
                try {
                    TUdpSocket::TRef udpSocket(MakeHolder<TUdpSocket>(Parent_, config.SourceAddr));
                    DyingSockets_.PushBack(udpSocket.Release());
                    return DyingSockets_.Back();
                } catch (...) {
                    static TLimitedLogger logger(Parent_->Logger_);
                    logger << TLOG_WARNING << CurrentExceptionMessage() << Endl;
                    return nullptr;
                }
            }

            inline TUdpSocket* GetNextSocket() {
                if (LastUsedSocket_ == ActiveSockets_.End()) {
                    LastUsedSocket_ = ActiveSockets_.Begin();
                    Y_VERIFY(LastUsedSocket_ != ActiveSockets_.End());
                }
                auto result(LastUsedSocket_);
                ++LastUsedSocket_;
                return &(*result);
            }

            TBaseProbeGenerator<TUdpIOHandler>* Attach(const TProbeConfig& config) {
                static TLimitedLogger logger(Parent_->Logger_);

                TIOStrategy* ioStrategy;
                const ui16 port(ExtractPort(config.SourceAddr));

                if (port || !TSettings::Get()->GetUdpUseMultiPortProbes()) {
                    auto* socket = (port ? GetSocketByPort(config, port) : GetNextSocket());
                    if (!socket) {
                        return nullptr;
                    }
                    ioStrategy = &socket->GetIOStrategy();
                } else {
                    ioStrategy = &IOStrategy_;
                }
                TProbeGenerator::TRef generator(TProbeGenerator::Make(
                        Parent_->GeneratorPool_, Parent_, config, logger, Parent_->Executor_, *ioStrategy));
                if (Generators_.Has(generator->Key())) {
                    // only one generator with given (ip, port, probe_id) can exist
                    return nullptr;
                } else {
                    generator->Generator().Subscribe([this] (TBaseProbeGenerator<TUdpIOHandler>* generator) noexcept {
                        Generators_.Erase(TProbeGenerator::TOps::ExtractKey(*generator));
                    });
                    Generators_.Push(generator.Get());
                    generator->Generator().Start();
                    return generator.Release();
                }
            }

        private:
            void Initialize() {
                // let's pre-create all required sockets
                for (const auto idx : xrange(SocketCount_)) {
                    Y_UNUSED(idx);

                    TUdpSocket::TRef udpSocket(MakeHolder<TUdpSocket>(Parent_, Addr_));
                    ActiveSockets_.PushBack(udpSocket.Release());
                }
                LastUsedSocket_ = ActiveSockets_.Begin();
            }

            inline void Run(TCont* cont) noexcept {
                Cont_ = cont;

                while (!Cont_->Cancelled()) {
                    try {
                        OneShot();
                    } catch(...) {
                        static TLimitedLogger logger(Parent_->Logger_);
                        logger << TLOG_WARNING << CurrentExceptionMessage() << Endl;
                    }
                }

                for (auto& sock : ActiveSockets_) {
                    sock.Wait(Cont_);
                }
                for (auto& sock : DyingSockets_) {
                    sock.Wait(Cont_);
                }

                Cont_ = nullptr;
            }

            inline void OneShot() {
                Y_VERIFY(ActiveSockets_.Size() > 1);
                Cont_->SleepT(TDuration::Seconds(1));

                // first, insert new one to save total socket count
                TUdpSocket::TRef newSocket(MakeHolder<TUdpSocket>(Parent_, Addr_));
                ActiveSockets_.PushBack(newSocket.Release());

                // let's take one socket and bind it to another host
                if (ActiveSockets_.Front() == LastUsedSocket_.Item()) {
                    ++LastUsedSocket_;
                }
                DyingSockets_.PushBack(ActiveSockets_.PopFront());

                while (!DyingSockets_.Empty()) {
                    // socket can be removed from list only after close
                    auto* socketToDestroy(DyingSockets_.Front());
                    socketToDestroy->WaitForGenerators(Cont_);
                    TUdpSocket::TRef garbage(socketToDestroy);
                    garbage->Unlink();
                    garbage->Wait(Cont_);
                }
            }

            const TImpl* const Parent_;
            const NAddr::IRemoteAddrRef Addr_;
            const std::size_t SocketCount_;
            TCont* Cont_;

            TUdpSocket::TListType ActiveSockets_;
            TUdpSocket::TListType DyingSockets_;
            TUdpSocket::TListType::TIterator LastUsedSocket_;

            TProbeGenerator::TMapType Generators_;

            TMultiSocketIOStrategy IOStrategy_;

            TSimpleContLoop<TUdpSocketPool, &TUdpSocketPool::Run> Loop_;
        };

    public:
        inline TImpl(TLog& logger, TContExecutor* executor)
            : Logger_(logger)
            , Executor_(executor)
            , Signature_(RandomNumber<ui32>())
            , PacketPool_(TDefaultAllocator::Instance())
            , GeneratorPool_(TDefaultAllocator::Instance())
            , BufferPool_(TDefaultAllocator::Instance())
        {
        }

        TBaseProbeGenerator<TUdpIOHandler>* Attach(const TProbeConfig& config) {
            if (config.SourceAddr->Addr()->sa_family != config.TargetAddr->Addr()->sa_family) {
                return nullptr;
            }

            const auto sourceAddrWithoutPort(SetPort(*config.SourceAddr));
            const auto it(SocketPools_.Find(*sourceAddrWithoutPort));
            return it != SocketPools_.End() ? it->Attach(config) : nullptr;
        }

        void ScheduleChecks(TCont* cont, const TVector<TProbeConfig>& configs, TVector<TProbeReport>& reports) {
            TProbeCollector<TImpl> collector(this, cont, configs);
            collector.Dump(reports);
        }

        void SyncAddresses(const TVector<TNetworkAddress>& addresses) {
            TAddrSet target;
            for (auto& addr : addresses) {
                for (auto it(addr.Begin()); it != addr.End(); ++it) {
                    target.emplace(&*it);
                }
            }

            TVector<NAddr::IRemoteAddrRef> keysToDelete;
            for (auto& pool : SocketPools_) {
                const auto& key(TUdpSocketPool::TOps::ExtractKey(pool));
                auto it(target.find(key));
                if (it.IsEnd()) {
                    keysToDelete.emplace_back(CopyRemoteAddr(key));
                } else {
                    target.erase(it);
                }
            }
            for (const auto& key : keysToDelete) {
                SocketPools_.Erase(*key);
            }

            bool failed = false;
            for (auto& addr : target) {
                try {
                    TUdpSocketPool::TRef pool(MakeHolder<TUdpSocketPool>(
                        this, CopyRemoteAddr(addr), TSettings::Get()->GetUdpSocketCount()));
                    SocketPools_.Push(pool.Release());
                } catch(...) {
                    static TLimitedLogger logger(Logger_);
                    logger << TLOG_WARNING << CurrentExceptionMessage() << Endl;
                    failed = true;
                }
            }

            if (failed) {
                ythrow yexception() << "can't bind to all given addresses";
            }
        }

        inline EProtocol Protocol() const noexcept {
            return EProtocol::UDP;
        }

    private:
        TLog& Logger_;
        TContExecutor* Executor_;
        const ui32 Signature_;

        mutable TUdpPacket::TPool PacketPool_;
        mutable TProbeGenerator::TPool GeneratorPool_;
        mutable TOnDemandBuffer::TPool BufferPool_;

        TUdpSocketPool::TMapType SocketPools_;
    };

    TUdpPollerService::TUdpPollerService(TLog& logger, TContExecutor* executor)
        : Impl(MakeHolder<TImpl>(logger, executor))
    {
    }

    TUdpPollerService::~TUdpPollerService()
    {
    }

    void TUdpPollerService::Stop() noexcept {
        Y_VERIFY(Impl);
        Impl.Destroy();
    }

    void TUdpPollerService::SyncAddresses(const TVector<TNetworkAddress>& addresses) {
        Y_VERIFY(Impl);
        Impl->SyncAddresses(addresses);
    }

    void TUdpPollerService::ScheduleChecks(TCont* cont, const TVector<TProbeConfig>& configs, TVector<TProbeReport>& reports) {
        Y_VERIFY(Impl);
        Impl->ScheduleChecks(cont, configs, reports);
    }
}
