#include "service.h"

#include <infra/netmon/agent/common/loop.h>
#include <infra/netmon/agent/common/utils.h>
#include <infra/netmon/agent/common/metrics.h>

#include <library/cpp/coroutine/dns/async.h>
#include <library/cpp/coroutine/dns/helpers.h>
#include <library/cpp/coroutine/dns/coro.h>

#include <util/generic/is_in.h>

#include <contrib/libs/c-ares/include/ares.h>

namespace NNetmon {
    class TResolverService::TImpl {
    public:
        inline TImpl(TLog& logger, TContExecutor* executor)
            : Logger_(logger)
            , Executor_(executor)
            , Resolver_(Executor_, NAsyncDns::TOptions().SetMaxRequests(8))
            , RequestPool_(TDefaultAllocator::Instance())
            , Cont_(nullptr)
            , Loop_(Executor_, this, "resolver")
        {
            Loop_.Start();
        }

    private:
        class TRequest : public TIntrusiveListItem<TRequest>, public TObjectFromPool<TRequest> {
        public:
            using TRef = THolder<TRequest>;
            using TListType = TIntrusiveList<TRequest>;

            template <typename... Args>
            static inline TRef Make(TPool& pool, Args&&... args) {
                return TRef(new (&pool) TRequest(std::forward<Args>(args)...));
            }

            // Returns an empty array on NXDOMAIN, returns Nothing on error.
            inline TMaybe<TAddrs> Execute() {
                Parent_->Resolver_.Resolve(NAsyncDns::TNameRequest(Name_.data(), Family_, &Callback_));

                // We may get ENODATA status for AF_UNSPEC requests if, e.g.,
                // there is no A record for Name_, but an AAAA record exists.
                // If no records exist (Result is empty), expect ENOTFOUND.
                TVector<int> goodStatuses = {ARES_SUCCESS, ARES_ENOTFOUND, ARES_ENODATA};
                if (!Callback_.Result) {
                    goodStatuses = {ARES_SUCCESS, ARES_ENOTFOUND};
                }

                for (auto status : Callback_.Status) {
                    if (!IsIn(goodStatuses, status)) {
                        Parent_->Logger_ << TLOG_DEBUG << "Can't resolve " << Name_ << "[" << Family_ <<  "]: "
                                         << ares_strerror(status);
                        return Nothing();
                    }
                }

                return Callback_.Result;
            }

            inline TCont* Cont() noexcept {
                Y_VERIFY(Cont_);
                return Cont_;
            }

        private:
            TRequest(TImpl* parent, const TString& name, int port, int family)
                : Parent_(parent)
                , Name_(name)
                , Port_(port)
                , Family_(family)
                , Cont_(Parent_->Executor_->Running())
                , Callback_(Port_)
            {
            }

            TImpl* Parent_;
            const TString Name_;
            const int Port_;
            const int Family_;
            TCont* Cont_;

            NAsyncDns::TResolveAddr Callback_;
        };

    public:
        inline TMaybe<TAddrs> Resolve(const TString& name, int port, int family) {
            Y_VERIFY(Cont_ != nullptr);

            TRequest::TRef req(TRequest::Make(RequestPool_, this, name, port, family));
            Requests_.PushBack(req.Get());

            return req->Execute();
        }

        inline void BeforeStop() noexcept {
            // wait for requests here because join on cancelled cont will cancel resolver cont,
            // this leads to use-after-free in c-ares callbacks
            while (!Requests_.Empty()) {
                Executor_->Running()->Join(Requests_.Front()->Cont());
            }

            Executor_->Running()->Yield();
        }

    private:
        inline void Run(TCont* cont) noexcept {
            Cont_ = cont;
            DoRun();
            Cont_ = nullptr;
        }

        inline void DoRun() noexcept {
            Y_VERIFY(Cont_ != nullptr);

            while (!Cont_->Cancelled()) {
                Cont_->SleepI();
            }

            Y_VERIFY(Requests_.Empty());
        }

        TLog& Logger_;
        TContExecutor* Executor_;
        NAsyncDns::TContResolver Resolver_;

        TRequest::TPool RequestPool_;
        TRequest::TListType Requests_;

        TCont* Cont_;

        TSimpleContLoop<TImpl, &TImpl::Run> Loop_;
    };

    TResolverService::TResolverService(TLog& logger, TContExecutor* executor)
    {
        Impl.Reset(MakeHolder<TImpl>(logger, executor));
    }

    TResolverService::~TResolverService()
    {
    }

    void TResolverService::Stop() noexcept {
        Y_VERIFY(Impl);
        Impl->BeforeStop();
        Impl.Destroy();
    }

    TMaybe<TAddrs> TResolverService::Resolve(const TString& name, int port, int family) {
        Y_VERIFY(Impl);
        return Impl->Resolve(name, port, family);
    }
}
