#include <infra/netmon/api/aggregator_api.h>
#include <infra/netmon/topology/settings.h>
#include <infra/netmon/metrics.h>
#include <infra/netmon/settings.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/generic/algorithm.h>
#include <util/generic/queue.h>
#include <util/stream/zlib.h>
#include <util/string/ascii.h>
#include <util/string/builder.h>
#include <util/system/hostname.h>

namespace NNetmon {
    namespace {
        TString EscapeString(const TStringBuf& s) {
            TString ret(s);
            ret.to_lower();
            for (TString::iterator it = ret.begin(); it != ret.end(); ++it) {
                if (!IsAsciiAlnum(*it)) {
                    *it = '_';
                }
            }
            return ret;
        }

        NMonitoring::TMetricRegistry MakeRegistry() {
            NMonitoring::TLabels labels({{"aggregator", HostName()}});
            for (const auto& [k, v] : TSettings::Get()->GetSolomonLabels()) {
                labels.Add(k, v);
            }
            return NMonitoring::TMetricRegistry(labels);
        }

        void ProcessSwitchMetrics(const TTopology::TSwitchRef& switchRef,
                                  const TGenericPacketCounters<ui64>& counters,
                                  std::initializer_list<NMonitoring::TLabel> extraLabels,
                                  NMonitoring::TMetricRegistry& registry) {
            if (auto value = CalculateNocSlaMetric(counters)) {
                NMonitoring::TLabels labels(extraLabels);
                labels.Add("sensor"sv, "sla");
                labels.Add("host"sv, switchRef->GetName());
                labels.Add("queue"sv, switchRef->GetLine().GetName());
                labels.Add("dc"sv, switchRef->GetDatacenter().GetName());

                registry.Gauge(labels)->Set(*value);
            }

            if (auto value = CalculateNocSlaTosMetric(counters)) {
                NMonitoring::TLabels labels(extraLabels);
                labels.Add("sensor"sv, "sla_tos");
                labels.Add("host"sv, switchRef->GetName());
                labels.Add("queue"sv, switchRef->GetLine().GetName());
                labels.Add("dc"sv, switchRef->GetDatacenter().GetName());

                registry.Gauge(labels)->Set(*value);
            }
        }

        void ProcessSwitchRtt(const TTopology::TSwitchRef& switchRef,
                              const TGenericRttCounters<ui64>& counters,
                              double percentile,
                              const TStringBuf& sensor,
                              std::initializer_list<NMonitoring::TLabel> extraLabels,
                              NMonitoring::TMetricRegistry& registry) {
            TSampleHistogram hist(*counters.Buckets);
            TSampleHistogram::TBuckets cumulativeCount;
            hist.FillCumulativeCount(cumulativeCount);

            if (auto value = hist.GetPercentileValue(percentile, cumulativeCount)) {
                NMonitoring::TLabels labels(extraLabels);
                labels.Add("sensor"sv, sensor);
                labels.Add("host"sv, switchRef->GetName());
                labels.Add("queue"sv, switchRef->GetLine().GetName());
                labels.Add("dc"sv, switchRef->GetDatacenter().GetName());

                registry.Gauge(labels)->Set(*value);
            }
        }

        void ProcessSwitchMetrics(const TTopology::TSwitchRef& switchRef,
                                  const TGenericRttCounters<ui64>& counters,
                                  std::initializer_list<NMonitoring::TLabel> extraLabels,
                                  NMonitoring::TMetricRegistry& registry) {
            ProcessSwitchRtt(switchRef, counters, 0.5,  "rtt50"sv, extraLabels, registry);
            ProcessSwitchRtt(switchRef, counters, 0.9,  "rtt90"sv, extraLabels, registry);
            ProcessSwitchRtt(switchRef, counters, 0.95, "rtt95"sv, extraLabels, registry);
            ProcessSwitchRtt(switchRef, counters, 0.99, "rtt99"sv, extraLabels, registry);
        }

        template <class TMetrics>
        void ProcessMetricsMap(const TInstant& timestamp,
                               const TMetrics& metricsMap,
                               const TStringBuf& linkPoller,
                               const NMonitoring::IMetricEncoderPtr& encoder,
                               const TTopologyStorage& topologyStorage,
                               const TInfraUpdater& infra) {
            auto registry = MakeRegistry();
            THashMap<TTopology::TDatacenterRef, bool> downtimeByDc;

            for (const auto& item : metricsMap) {
                auto switchRef = topologyStorage.FindSwitch(item.first);
                if (!switchRef) {
                    continue;
                }

                auto dc = switchRef.GetDatacenter();
                if (!downtimeByDc.contains(dc)) {
                    downtimeByDc[dc] = infra.IsInfraEventOngoing(dc->GetName());
                }
                if (downtimeByDc[dc]) {
                    continue;
                }

                for (auto& direct : {std::make_pair("tx"sv, item.second.Tx),
                                    std::make_pair("rx"sv, item.second.Rx)}) {
                    for (auto& tc : {std::make_pair("cs1"sv, direct.second.CS1),
                                    std::make_pair("cs2"sv, direct.second.CS2),
                                    std::make_pair("cs3"sv, direct.second.CS3),
                                    std::make_pair("cs4"sv, direct.second.CS4)}) {
                        ProcessSwitchMetrics(switchRef, tc.second,
                                             {{"direction"sv,   direct.first},
                                              {"tclass"sv,      tc.first},
                                              {"link_poller"sv, linkPoller}},
                                             registry);
                    }
                    ProcessSwitchMetrics(switchRef, direct.second.BB4,
                                         {{"direction"sv,   direct.first},
                                          {"network"sv,     "bb4"sv},
                                          {"link_poller"sv, linkPoller}},
                                         registry);
                }
            }
            registry.Accept(timestamp, encoder.Get());
        }

        void ProcessLinkPollerMetricsMap(const TInstant& timestamp,
                                         const TLinkPollerMetricsUpdater::TMetricsMap& metricsMap,
                                         const TStringBuf& linkPoller,
                                         const NMonitoring::IMetricEncoderPtr& encoder,
                                         const TTopologyStorage& topologyStorage,
                                         const TInfraUpdater& infra) {
            auto registry = MakeRegistry();
            THashMap<TTopology::TDatacenterRef, bool> downtimeByDc;

            for (const auto& item : metricsMap) {
                auto switchRef = topologyStorage.FindSwitch(item.first);
                if (!switchRef) {
                    continue;
                }

                auto dc = switchRef.GetDatacenter();
                if (!downtimeByDc.contains(dc)) {
                    downtimeByDc[dc] = infra.IsInfraEventOngoing(dc->GetName());
                }
                if (downtimeByDc[dc]) {
                    continue;
                }

                for (auto & cs : {std::make_pair(item.second.CS1, "cs1"sv),
                                  std::make_pair(item.second.CS2, "cs2"sv),
                                  std::make_pair(item.second.CS3, "cs3"sv),
                                  std::make_pair(item.second.CS4, "cs4"sv)}) {
                    auto value = CalculateNocSlaMetric(cs.first);
                    if (!value.Defined()) {
                        continue;
                    }

                    auto *gauge = registry.Gauge({
                                                         {"sensor"sv,      "sla"sv},
                                                         {"host"sv,        switchRef->GetName()},
                                                         {"queue"sv,       switchRef->GetLine().GetName()},
                                                         {"dc"sv,          switchRef->GetDatacenter().GetName()},
                                                         {"direction"sv,   "local"sv},
                                                         {"tclass"sv,      cs.second},
                                                         {"link_poller"sv, linkPoller}
                                                 });
                    gauge->Set(*value);
                }
            }
            registry.Accept(timestamp, encoder.Get());
        }

        template <class TMetricsUpdater>
        void ProcessMetricsFromUpdater(TMetricsUpdater& updater,
                                       const std::pair<TInstant, TInstant>& interval,
                                       const TStringBuf& linkPoller,
                                       const NMonitoring::IMetricEncoderPtr& encoder,
                                       const TTopologyStorage& topologyStorage,
                                       const TInfraUpdater& infra) {
            auto metrics = updater.GetPastMetrics(interval.first, interval.second);
            for (const auto& item : metrics) {
                // series with rounded timestamps are more efficiently stored in Solomon
                auto timestamp = RoundInstant(item.first, TDuration::Seconds(5));

                if constexpr (std::is_same_v<TMetricsUpdater, TLinkPollerMetricsUpdater>) {
                    ProcessLinkPollerMetricsMap(timestamp, *item.second, linkPoller, encoder, topologyStorage, infra);
                } else {
                    ProcessMetricsMap(timestamp, *item.second, linkPoller, encoder, topologyStorage, infra);
                }
            }
        }

        template <class TProbeCounters>
        void ProcessProbeCounterMetricsVerbose(NJsonWriter::TBuf& response,
                                               const TProbeCounters& counters) {
            response
                .WriteKey("metric")
                .WriteDouble(CalculateNocSlaMetric(counters).GetOrElse(0))
                .WriteKey("success")
                .WriteULongLong(counters.Success)
                .WriteKey("failed")
                .WriteULongLong(counters.Failed)
                .WriteKey("metric_tos")
                .WriteDouble(CalculateNocSlaTosMetric(counters).GetOrElse(0));
        }

        template <class TSlaCounters>
        void ProcessSlaCounterMetricsVerbose(NJsonWriter::TBuf& response,
                                             const TSlaCounters& counters) {
            response
                .WriteKey(TStringBuf("cs3"))
                .BeginObject();
            ProcessProbeCounterMetricsVerbose(response, counters.CS3);
            response
                .EndObject()
                .WriteKey(TStringBuf("cs4"))
                .BeginObject();
            ProcessProbeCounterMetricsVerbose(response, counters.CS4);
            response
                .EndObject();
        }

        template <class TMetrics>
        void ProcessSwitchMetricsVerbose(NJsonWriter::TBuf& response,
                                         const TMetrics& metrics) {
            response
                .WriteKey(TStringBuf("rx"))
                .BeginObject();
            ProcessSlaCounterMetricsVerbose(response, metrics.Rx);
            response
                .EndObject()
                .WriteKey(TStringBuf("tx"))
                .BeginObject();
            ProcessSlaCounterMetricsVerbose(response, metrics.Tx);
            response
                .EndObject();
        }


        void ProcessLinkPollerMetricsVerbose(NJsonWriter::TBuf& response,
                                             const TLinkPollerMetricsUpdater::TMetrics& metrics) {
            response
                    .WriteKey(TStringBuf("cs1"))
                    .BeginObject();
            ProcessProbeCounterMetricsVerbose(response, metrics.CS1);
            response
                    .EndObject()
                    .WriteKey(TStringBuf("cs2"))
                    .BeginObject();
            ProcessProbeCounterMetricsVerbose(response, metrics.CS2);
            response
                    .EndObject();
            response
                    .WriteKey(TStringBuf("cs3"))
                    .BeginObject();
            ProcessProbeCounterMetricsVerbose(response, metrics.CS3);
            response
                    .EndObject()
                    .WriteKey(TStringBuf("cs4"))
                    .BeginObject();
            ProcessProbeCounterMetricsVerbose(response, metrics.CS4);
            response
                    .EndObject();
        }
    } // end namespace

    // api/v1/ping handler

    void TPingReply::Process() {
        if (!GetServerContext().GetExpressionStorage().IsReady()) {
            ythrow yexception() << "expressions aren't ready yet";
        }
        if (!GetServerContext().GetWalleUpdater().IsReady()) {
            ythrow yexception() << "wall-e cache isn't ready yet";
        }
        if (!GetServerContext().GetSeenHostsUpdater().IsReady()) {
            ythrow yexception() << "seen hosts cache isn't ready yet";
        }
        if (!GetServerContext().GetSliceCollector().IsReady()) {
            ythrow yexception() << "slices aren't collected yet";
        }
        if (!GetServerContext().GetProbeDumper().IsReady()) {
            ythrow yexception() << "too many probes aren't dumped";
        }

        GetResponse().BeginObject()
            .WriteKey(TStringBuf("status"))
            .WriteString("OK")
            .EndObject();
    }

    // api/v1/tags handler

    void TTagsReply::Process() {
        if (!GetServerContext().GetExpressionStorage().IsReady()) {
            ythrow yexception() << "expressions aren't ready yet";
        }
        if (!GetServerContext().GetSliceCollector().IsReady()) {
            ythrow yexception() << "slices aren't collected yet";
        }

        GetResponse().BeginObject()
            .WriteKey(TStringBuf("tags"))
            .BeginList();

        for (const auto& aggregator : GetServerContext().GetAggregatorMaintainer().GetAggregators()) {
            for (const auto& expression : GetServerContext().GetExpressionStorage().Find(aggregator->GetKey().GetExpressionId())) {
                GetResponse().BeginObject()
                    .WriteKey(TStringBuf("tag"))
                    .WriteString(expression.Id)
                    .WriteKey(TStringBuf("network"))
                    .WriteString(NetworkToString(aggregator->GetKey().GetNetwork()))
                    .WriteKey(TStringBuf("protocol"))
                    .WriteString(ProtocolToString(aggregator->GetKey().GetProtocol()))
                    .EndObject();
            }
        }

        GetResponse().EndList()
            .EndObject();
    }

    // api/v1/seen_hosts handler

    void TSeenHostsReply::Process() {
        const auto expressionId(GetRequestData().GetExpressionId());
        const auto stateMap(GetServerContext().GetSeenHostsMaintainer().GetStateMap());
        const auto it(stateMap->States.find(expressionId));

        GetResponse()
            .BeginObject()
            .WriteKey(TStringBuf("hosts"));

        if (!it.IsEnd() &&
            GetServerContext().GetTopologyStorage().IsExpressionResolved(expressionId) &&
            GetServerContext().GetWalleUpdater().IsReady())
        {
            const auto& state(it->second);
            const auto& walleUpdater(GetServerContext().GetWalleUpdater());
            auto maintenanceCounter(walleUpdater.GetHostCountUnderMaintenance(expressionId));

            GetResponse()
                .BeginObject();

            GetResponse()
                .WriteKey(TStringBuf("dc"))
                .BeginList();
            for (const auto& dc : state.CountByDc) {
                auto it = maintenanceCounter->ByDc.find(dc.first);
                ui64 maintenanceCount = (it.IsEnd() ? 0 : it->second);
                maintenanceCount = Min(dc.second.second, maintenanceCount);

                GetResponse()
                    .BeginObject()
                        .WriteKey(TStringBuf("name"))
                        .WriteString(dc.first->GetName())
                        .WriteKey(TStringBuf("seen"))
                        .WriteULongLong(dc.second.first)
                        .WriteKey(TStringBuf("total"))
                        .WriteULongLong(dc.second.second - maintenanceCount)
                        .WriteKey(TStringBuf("maintenance"))
                        .WriteULongLong(maintenanceCount)
                    .EndObject();
            }
            GetResponse()
                .EndList();

            GetResponse()
                .WriteKey(TStringBuf("queue"))
                .BeginList();
            for (const auto& queue : state.CountByQueue) {
                auto it = maintenanceCounter->ByQueue.find(queue.first);
                ui64 maintenanceCount = (it.IsEnd() ? 0 : it->second);
                maintenanceCount = Min(queue.second.second, maintenanceCount);

                GetResponse()
                    .BeginObject()
                        .WriteKey(TStringBuf("name"))
                        .WriteString(queue.first->GetName())
                        .WriteKey(TStringBuf("seen"))
                        .WriteULongLong(queue.second.first)
                        .WriteKey(TStringBuf("total"))
                        .WriteULongLong(queue.second.second - maintenanceCount)
                        .WriteKey(TStringBuf("maintenance"))
                        .WriteULongLong(maintenanceCount)
                    .EndObject();
            }
            GetResponse()
                .EndList();

            ui64 maintenanceTotal = Min(state.TotalHosts, maintenanceCounter->Total);
            GetResponse()
                .WriteKey(TStringBuf("seen"))
                .WriteULongLong(state.Hosts.size())
                .WriteKey(TStringBuf("total"))
                .WriteULongLong(state.TotalHosts - maintenanceTotal)
                .WriteKey(TStringBuf("maintenance"))
                .WriteULongLong(maintenanceTotal)
                .WriteKey(TStringBuf("timestamp"))
                .WriteULongLong(stateMap->Timestamp.Seconds())
                .EndObject();
        } else {
            GetResponse().WriteNull();
        }

        GetResponse()
            .EndObject();
    }

    // api/v1/terminated_hosts handler

    void TTerminatedHostsReply::Process() {
        const auto& topologyStorage(GetServerContext().GetTopologyStorage());

        auto state(GetServerContext().GetTerminatedHostsUpdater().GetState());
        const auto& terminatedHosts(state->GetHosts());

        GetResponse()
            .BeginObject()
            .WriteKey(TStringBuf("hosts"))
            .BeginList();
        for (auto hostId : terminatedHosts) {
            auto foundHost = topologyStorage.FindHost(hostId);
            if (foundHost) {
                GetResponse()
                    .WriteString(foundHost->GetName());
            }
        }
        GetResponse()
            .EndList()
            .EndObject();
    }

    // api/v1/interested_hosts handler

    void TInterestedHostsReply::Process() {
        auto interestedHosts(GetServerContext().GetProbeScheduleMaintainer().GetInterestedHosts());

        auto state(GetServerContext().GetTerminatedHostsUpdater().GetState());
        const auto& terminatedHosts(state->GetHosts());

        GetResponse()
            .BeginObject()
            .WriteKey(TStringBuf("hosts"))
            .BeginList();
        for (auto host : *interestedHosts) {
            if (!terminatedHosts.contains(host->GetReducedId())) {
                GetResponse()
                    .WriteString(host->GetName());
            }
        }
        GetResponse()
            .EndList()
            .EndObject();
    }

    // api/v1/history_limits handler

    void THistoryLimitsReply::Process() {
        GetResponse().BeginObject()
            .WriteKey(TStringBuf("dc"))
            .WriteULongLong(TSettings::Get()->GetDcHistoryTimeToLive().Seconds())
            .WriteKey(TStringBuf("queue"))
            .WriteULongLong(TSettings::Get()->GetQueueHistoryTimeToLive().Seconds())
            .WriteKey(TStringBuf("switch"))
            .WriteULongLong(TSettings::Get()->GetSwitchHistoryTimeToLive().Seconds())
            .EndObject();
    }

    // api/v1/queue_mapping handler

    void TQueueMappingReply::Process() {
        GetResponse()
            .BeginObject();

        for (const auto& pair : TTopologySettings::Get()->GetQueueMapping()) {
            GetResponse()
                .WriteKey(pair.first)
                .WriteString(pair.second);
        }

        GetResponse()
            .EndObject();
    }

    // api/v1/replace_dead_hosts handler

    void TReplaceDeadHostsReply::ParseRequest(THttpInput& input) {
        TBufferedZLibDecompress decompressor(&input, ZLib::ZLib);
        TBufferOutput output;
        TransferData(&decompressor, &output);

        const TBuffer& buf = output.Buffer();
        if (!buf.Empty()) {
            NJson::ReadJsonFastTree(TStringBuf(buf.Data(), buf.Size()), &Request, true);
        }
    }

    void TReplaceDeadHostsReply::Process() {
        const auto& topologyStorage = GetServerContext().GetTopologyStorage();
        auto& walleUpdater = GetServerContext().GetWalleUpdater();
        const auto& input = Request.GetMapSafe();

        if (!input.contains("hosts")) {
            ythrow yexception() << "hosts not set";
        }
        const auto& hostsArray = input.at("hosts").GetArraySafe();

        TTopologyStorage::THostSet deadHosts;
        deadHosts.reserve(hostsArray.size());

        for (const auto& value : hostsArray) {
            auto host = topologyStorage.FindHost(value.GetStringSafe());
            if (host) {
                deadHosts.emplace(std::move(host));
            }
        }

        walleUpdater.ReplaceDeadHosts(deadHosts);
        INFO_LOG << "Processed replace_dead_hosts request, new size: " << deadHosts.size() << Endl;
    }

    // api/v1/quality handler

    void TQualityStatsReply::Process() {
        if (!GetServerContext().GetExpressionStorage().IsReady()) {
            ythrow yexception() << "expressions aren't ready yet";
        }
        if (!GetServerContext().GetSliceCollector().IsReady()) {
            ythrow yexception() << "slices aren't collected yet";
        }

        GetResponse().BeginList();

        const auto& whitelist = TSettings::Get()->GetWhitelistedExpressions();
        const auto seenHosts = GetServerContext().GetSeenHostsMaintainer().GetStateMap();

        for (const auto& aggregator : GetServerContext().GetAggregatorMaintainer().GetAggregators()) {
            for (const auto& expression : GetServerContext().GetExpressionStorage().Find(aggregator->GetKey().GetExpressionId())) {
                bool whitelisted((whitelist.find(expression.Id) != whitelist.end()) ||
                                 GetServerContext().GetUserDataStorage().IsQualitySignalsEnabled(expression.Id));
                SeenHostsToJson(expression, seenHosts, whitelisted);
                ProbesToJson(*aggregator, expression, whitelisted);
                if (whitelisted) {
                    RttHgramsToJson(*aggregator, expression);
                    LineMaintainerToJson(*aggregator, expression);
                }
            }
        }

        auto AddAbsolute = [&](const TStringBuf name, ui64 value) {
            GetResponse()
                .BeginList()
                .WriteString(TStringBuilder() << name << TStringBuf("_axxx"))
                .WriteULongLong(value)
                .EndList();
        };

        const auto stats(GetServerContext().GetHostStats());
        AddAbsolute(TStringBuf("agents_total_bare"), stats.TotalBare);
        AddAbsolute(TStringBuf("agents_seen_bare"), stats.SeenBare);
        AddAbsolute(TStringBuf("agents_total_virtual"), stats.TotalVirtual);
        AddAbsolute(TStringBuf("agents_seen_virtual"), stats.SeenVirtual);
        AddAbsolute(TStringBuf("agents_with_version"), stats.WithVersion);
        AddAbsolute(TStringBuf("agents_lastest_version"), stats.LastestVersion);

        GetResponse().EndList();
    }

    void TQualityStatsReply::SeenHostsToJson(const TExpressionStorage::TResult& expression, const TSeenHostsMaintainer::TStateMap::TConstValueRef seenHosts, bool detailed) {
        const auto it(seenHosts->States.find(expression.ExpressionId));
        if (!it.IsEnd()) {
            const auto& state(it->second);

            GetResponse()
                .BeginList()
                .WriteString(TString::Join(EscapeString(expression.Id), "_seen_hosts_axxx"))
                .WriteULongLong(state.Hosts.size())
                .EndList()
                .BeginList()
                .WriteString(TString::Join(EscapeString(expression.Id), "_total_hosts_axxx"))
                .WriteULongLong(state.TotalHosts)
                .EndList();

            if (detailed) {
                auto AddHosts = [&](const TString& name, const std::pair<ui64, ui64>& hosts) {
                    GetResponse()
                        .BeginList()
                        .WriteString(TString::Join(EscapeString(expression.Id), "_", name, "_seen_hosts_axxx"))
                        .WriteULongLong(hosts.first)
                        .EndList()
                        .BeginList()
                        .WriteString(TString::Join(EscapeString(expression.Id), "_", name, "_total_hosts_axxx"))
                        .WriteULongLong(hosts.second)
                        .EndList();
                };

                for (const auto& dc : state.CountByDc) {
                    AddHosts(dc.first->GetName(), dc.second);
                }

                for (const auto& queue : state.CountByQueue) {
                    AddHosts(queue.first->GetName(), queue.second);
                }
            }
        }
    }

    void TQualityStatsReply::ProbesToJson(const TProbeAggregator& aggregator, const TExpressionStorage::TResult& expression, bool detailed) {
        TString prefix = GetSignalNamePrefix(aggregator, expression);

        ui64 probeCount = 0;
        ui64 deadProbeCount = 0;

        const auto index(aggregator.GetDatacenterIndex());
        if (index && (index->GetGenerated() + TSettings::Get()->GetDcAggregationInterval() * 3 >= TInstant::Now())) {
            const auto& tree = index->GetTree();
            for (auto it(tree.Begin()); it != tree.End(); ++it) {
                const auto& state(*it);

                probeCount += state.GetProbeCount();
                deadProbeCount += state.GetDeadProbeCount();

                if (detailed) {
                    TString signalPrefix;
                    if (state.GetSource() == state.GetTarget()) {
                        signalPrefix = TStringBuilder()
                                    << prefix
                                    << EscapeString(state.GetSource().GetName()) << "_";
                    } else {
                        signalPrefix = TStringBuilder()
                                    << prefix
                                    << EscapeString(state.GetSource().GetName()) << "_"
                                    << EscapeString(state.GetTarget().GetName()) << "_";
                    }

                    auto AddULongLong = [&](const TStringBuf name, ui64 value) {
                    TString signalName = TStringBuilder() << signalPrefix << name << TStringBuf("_axxx");
                    GetResponse()
                        .BeginList()
                        .WriteString(signalName)
                        .WriteULongLong(value)
                        .EndList();
                    };

                    AddULongLong(TStringBuf("probes"), state.GetProbeCount());
                    AddULongLong(TStringBuf("dead_probes"), state.GetDeadProbeCount());
                }
            }
        }

        GetResponse()
            .BeginList()
            .WriteString(TString::Join(prefix, "probes_axxx"))
            .WriteULongLong(probeCount)
            .EndList()
            .BeginList()
            .WriteString(TString::Join(prefix, "dead_probes_axxx"))
            .WriteULongLong(deadProbeCount)
            .EndList();
    }

    void TQualityStatsReply::RttHgramsToJson(const TProbeAggregator& aggregator, const TExpressionStorage::TResult& expression) {
        TString prefix = GetSignalNamePrefix(aggregator, expression);

        const auto index(aggregator.GetDatacenterIndex());
        if (index && (index->GetGenerated() + TSettings::Get()->GetDcAggregationInterval() * 3 >= TInstant::Now())) {
            const auto& tree = index->GetTree();

            for (auto it(tree.Begin()); it != tree.End(); ++it) {
                const auto& src = it->GetSource().GetName();
                const auto& dst = it->GetTarget().GetName();

                /* Same dcs are handled in LineMaintainerToJson */
                if (src != dst) {

                    /* Since rtt is a sample histogram,
                       we can call cross-dc rtt histogram also as cross-line :) */

                    TString signalName = TString::Join(
                                            prefix, src, "_",
                                            dst, "_crossline_rtt_ahhh"
                                        );

                    RttSignalToJson(signalName, (*it).GetRttHistogram());
                }
            }
        }
    }

    void TQualityStatsReply::LineMaintainerToJson(const TProbeAggregator& aggregator, const TExpressionStorage::TResult& expression) {
        TString prefix = GetSignalNamePrefix(aggregator, expression);

        const auto index(aggregator.GetLineIndex());
        if (!index || (index->GetGenerated() + TSettings::Get()->GetLineAggregationInterval() * 3 < TInstant::Now())) {
            return;
        }

        THashMap<TString, TVector<ui64>> intraAndCrossDcConn;
        THashMap<TString, TSampleHistogram> crosslineRtt;
        THashMap<TString, TSampleHistogram> intralineRtt;

        const auto& tree = index->GetTree();
        TVector<ui64> lineConn;
        lineConn.resize(LineConnWeights.size());

        for (auto it(tree.Begin()); it != tree.End(); ++it) {
            /* Gather multiline connectivity percentage in single histogram */
            auto connHgram = it->GetConnectivityHistogram().GetValues();
            if (connHgram) {
                // merging conn100 metric
                double conn = connHgram->at(CONNECTIVITY_BUCKET_COUNT - 1);
                auto bucket = std::lower_bound(LineConnWeights.begin(), LineConnWeights.end(), conn);
                auto index = std::distance(LineConnWeights.begin(), bucket);

                if (LineConnWeights[index] > conn && index > 0) {
                    --index;
                }

                lineConn[index] += 1;

                const auto& src_dc = it->GetSource().GetDatacenter().GetName();
                const auto& tgt_dc = it->GetTarget().GetDatacenter().GetName();
                const TString dcPartName = (src_dc == tgt_dc ? src_dc : TString::Join(src_dc, "_", tgt_dc));

                auto& dcConn = *(intraAndCrossDcConn.emplace(
                                    std::piecewise_construct,
                                    std::forward_as_tuple(dcPartName),
                                    std::forward_as_tuple(LineConnWeights.size())
                                ).first);

                dcConn.second[index] += 1;
            }

            /* Merge two separate cross- and intra- line rtt hgrams for same dc */
            const auto& dc = it->GetSource().GetDatacenter().GetName();
            if (dc != it->GetTarget().GetDatacenter().GetName()) {
                continue;
            }

            const auto& src = it->GetSource().GetName();
            const auto& tgt = it->GetTarget().GetName();

            if (src == tgt) {
                intralineRtt[dc].Merge(it->GetRttHistogram());
            } else {
                crosslineRtt[dc].Merge(it->GetRttHistogram());
            }
        }

        TString signalName = TString::Join(prefix, "line_conn_ahhh");
        ConnSignalToJson(signalName, lineConn);

        for (const auto& dcConn : intraAndCrossDcConn) {
            signalName = TString::Join(prefix, dcConn.first, "_line_conn_ahhh");
            ConnSignalToJson(signalName, dcConn.second);
        }

        for (const auto& dc_rtt : crosslineRtt) {
            signalName = TString::Join(
                            prefix, dc_rtt.first, "_",
                            dc_rtt.first, "_crossline_rtt_ahhh"
                        );

            RttSignalToJson(signalName, dc_rtt.second);
        }

        for (const auto& dc_rtt : intralineRtt) {
            signalName = TString::Join(
                            prefix, dc_rtt.first, "_",
                            dc_rtt.first, "_intraline_rtt_ahhh"
                        );

            RttSignalToJson(signalName, dc_rtt.second);
        }
    }

    void TQualityStatsReply::RttSignalToJson(const TString& signalName, const TSampleHistogram& rtt_histogram) {
        auto ctx = GetResponse().BeginList().WriteString(signalName).BeginList();

        for (size_t i = 0; i < RTT_BUCKET_COUNT; i++) {
            ctx.BeginList()
                .WriteDouble(GetBucketWeights()[i])
                .WriteDouble(rtt_histogram.RawBuckets()[i])
                .EndList();

        }

        ctx.EndList().EndList();
    }

    void TQualityStatsReply::ConnSignalToJson(const TString& signalName, const TVector<ui64>& conn_histogram) {
        auto ctx = GetResponse().BeginList().WriteString(signalName).BeginList();

        for(size_t i = 0; i < LineConnWeights.size(); ++i) {
            ctx.BeginList()
                .WriteDouble(LineConnWeights[i])
                .WriteDouble(conn_histogram[i])
                .EndList();

        }

        ctx.EndList().EndList();
    }

    TString TQualityStatsReply::GetSignalNamePrefix(const TProbeAggregator& aggregator, const TExpressionStorage::TResult& expression) const {
        return TString::Join(
                   EscapeString(expression.Id), "_",
                   NetworkToString(aggregator.GetKey().GetNetwork()), "_",
                   ProtocolToString(aggregator.GetKey().GetProtocol()), "_"
              );
    }

    // api/v1/hosts handler

    template <class TContext>
    void HostToJson(const THost& host, NJsonWriter::TBuf& response, const TContext& context) {
        response
            .BeginObject()
            .WriteKey(TStringBuf("name"))
            .WriteString(host.GetName())
            .WriteKey(TStringBuf("id"))
            .WriteString(::ToString(host.GetReducedId()))
            .WriteKey(TStringBuf("dc"))
            .WriteString(host.GetDatacenter().GetName())
            .WriteKey(TStringBuf("queue"))
            .WriteString(host.GetLine().GetName())
            .WriteKey(TStringBuf("switch"))
            .WriteString(host.GetSwitch().GetName())
            .WriteKey(TStringBuf("virtual"))
            .WriteBool(host.IsVirtual());

        if constexpr (std::is_same_v<TContext, TAggregatorContext>) {
            response
                .WriteKey(TStringBuf("agent_version"))
                .WriteString(context.GetFinishedTaskIndex().GetVersion(host.GetName()));
        }

        response
            .WriteKey(TStringBuf("interfaces"))
            .BeginObject();

        for (const auto& iface : host.GetInterfaces()) {
            response
                .WriteKey(iface->GetName())
                .BeginObject()
                .WriteKey(TStringBuf("name"))
                .WriteString(iface->GetName())
                .WriteKey(TStringBuf("id"))
                .WriteString(::ToString(iface->GetReducedId()))
                .WriteKey(TStringBuf("switch"))
                .WriteString(iface->GetSwitch().GetName())
                .WriteKey(TStringBuf("network_type"))
                .WriteString(NetworkToString(iface->GetNetworkType()))
                .WriteKey(TStringBuf("ipv6addr"))
                .WriteString(iface->GetIpv6().ToString())
                .WriteKey(TStringBuf("ipv4addr"))
                .WriteString(iface->GetIpv4().ToString())
                .EndObject();
        }

        response
            .EndObject();

        if constexpr (std::is_same_v<TContext, TAggregatorContext>) {
            response
                .WriteKey(TStringBuf("expressions"))
                .BeginObject();

                const auto selector(context.GetTopologyStorage().GetTopologySelector());

                TSet<TExpressionId> expressionIds;
                for (const auto& iface : host.GetInterfaces()) {
                    const auto sourceExpressionIds(selector->FindSourceExpressions(*iface));
                    if (sourceExpressionIds) {
                        expressionIds.insert(sourceExpressionIds->begin(), sourceExpressionIds->end());
                    }
                    const auto targetExpressionIds(selector->FindTargetExpressions(*iface));
                    if (targetExpressionIds) {
                        expressionIds.insert(targetExpressionIds->begin(), targetExpressionIds->end());
                    }
                }

                for (const auto& expressionId : expressionIds) {
                for (const auto& expressionState : context.GetExpressionStorage().Find(expressionId)) {
                    response
                        .WriteKey(expressionState.Id)
                        .WriteString(expressionState.Expression);
                }
            }
            response.EndObject();
        }
        response
            .WriteKey(TStringBuf("owners"))
            .BeginList();

        for (auto owner : host.GetOwners()) {
            response.WriteString(owner);
        }

        response
            .EndList()
            .EndObject();
    }

    void THostsReply::Process() {
        const auto hosts(GetRequestData().GetMultipleHosts());

        GetResponse()
            .BeginObject()
            .WriteKey(TStringBuf("hosts"))
            .BeginObject();

        for (const auto& host : hosts) {
            GetResponse().WriteKey(host->GetName());
            HostToJson(*host, GetResponse(), GetServerContext());
        }

        GetResponse()
            .EndObject()
            .EndObject();
    }

    void TResolverHostsReply::Process() {
        const auto hosts(GetRequestData().GetMultipleHosts());

        GetResponse()
            .BeginObject()
            .WriteKey(TStringBuf("hosts"))
            .BeginObject();

        for (const auto& host : hosts) {
            GetResponse().WriteKey(host->GetName());
            HostToJson(*host, GetResponse(), GetServerContext());
        }

        GetResponse()
            .EndObject()
            .EndObject();
    }

    // api/v1/find_by_version handler

    void TFindByVersionReply::Process() {
        GetResponse()
            .BeginObject()
            .WriteKey(TStringBuf("hosts"))
            .BeginList();

        for (const auto& host : GetServerContext().GetFinishedTaskIndex().FindByVersion(GetRequestData().GetVersion())) {
            GetResponse().WriteString(host);
        }

        GetResponse()
            .EndList()
            .EndObject();
    }

    void TAgentReportReply::Process() {
        GetResponse()
            .BeginObject()
            .WriteKey(TStringBuf("agents"))
            .BeginList();

        GetServerContext().GetFinishedTaskIndex().ForEachVersion([this](const NClient::TVersionResult& info, TInstant generated) {
            GetResponse()
                .BeginObject()
                .WriteKey(TStringBuf("host"))
                .WriteString(info.GetHost())
                .WriteKey(TStringBuf("version"))
                .WriteString(info.GetVersion())
                .WriteKey(TStringBuf("platform"))
                .WriteString(info.GetPlatform())
                .WriteKey(TStringBuf("generated"))
                .WriteString(::ToString(generated.Seconds()))
                .WriteKey(TStringBuf("interfaces"))
                .BeginList();

            for (const auto& iface : info.GetLocalInterfaces()) {
                GetResponse()
                    .BeginObject()
                    .WriteKey(TStringBuf("fqdn"))
                    .WriteString(iface.GetFQDN())
                    .WriteKey(TStringBuf("address"))
                    .WriteString(iface.GetAddress())
                    .WriteKey(TStringBuf("mask"))
                    .WriteString(iface.GetMask())
                    .WriteKey(TStringBuf("mac"))
                    .WriteString(iface.GetMAC())
                    .EndObject();
            }

            GetResponse()
                .EndList()
                .EndObject();
        });

        GetResponse()
            .EndList()
            .EndObject();
    }

    // api/v1/coverage handler

    void TCoverageReply::Process() {
        const auto stats(GetServerContext().GetHostStats());
        const auto versions(GetServerContext().GetFinishedTaskIndex().VersionHistogram());

        GetResponse()
            .BeginObject()
            .WriteKey(TStringBuf("bare"))
            .BeginObject()
            .WriteKey(TStringBuf("total"))
            .WriteULongLong(stats.TotalBare)
            .WriteKey(TStringBuf("seen"))
            .WriteULongLong(stats.SeenBare)
            .EndObject()
            .WriteKey(TStringBuf("virtual"))
            .BeginObject()
            .WriteKey(TStringBuf("total"))
            .WriteULongLong(stats.TotalVirtual)
            .WriteKey(TStringBuf("seen"))
            .WriteULongLong(stats.SeenVirtual)
            .EndObject()
            .WriteKey(TStringBuf("versions"))
            .BeginObject();

        for (const auto& pair : versions) {
            GetResponse()
                .WriteKey(pair.first)
                .WriteInt(pair.second);
        }

        GetResponse()
            .EndObject()
            .EndObject();
    }

    // api/v1/switch_metrics

    void TSwitchMetricsReply::Process() {
        const auto& infra = GetServerContext().GetInfraUpdater();
        const auto& topologyStorage = GetServerContext().GetTopologyStorage();
        const auto& interval = GetRequestInterval();
        INFO_LOG << "Received switch metrics request with interval from " << interval.first << " to " << interval.second << Endl;

        // link_poller = false
        ProcessMetricsFromUpdater(GetServerContext().GetInterSwitchMetricsUpdater(),
                                  interval, "false"sv, GetEncoder(), topologyStorage, infra);
        ProcessMetricsFromUpdater(GetServerContext().GetInterSwitchRttMetricsUpdater(),
                                  interval, "false"sv, GetEncoder(), topologyStorage, infra);

        // link_poller = true
        ProcessMetricsFromUpdater(GetServerContext().GetInterSwitchLPMetricsUpdater(),
                                  interval, "true"sv, GetEncoder(), topologyStorage, infra);
        ProcessMetricsFromUpdater(GetServerContext().GetInterSwitchRttLPMetricsUpdater(),
                                  interval, "true"sv, GetEncoder(), topologyStorage, infra);
        ProcessMetricsFromUpdater(GetServerContext().GetLinkPollerMetricsUpdater(),
                                  interval, "true"sv, GetEncoder(), topologyStorage, infra);
    }

    // api/v1/inter_switch_metrics_verbose

    void TInterSwitchMetricsVerboseReply::Process() {
        const auto& topology = GetServerContext().GetTopologyStorage();
        const auto& metricsUpdater = GetServerContext().GetInterSwitchMetricsUpdater();
        auto metricsMap = metricsUpdater.GetCurrentMetrics();

        GetResponse()
            .BeginList();

        for (const auto& item : *metricsMap) {
            auto switchRef = topology.FindSwitch(item.first);
            if (!switchRef) {
                continue;
            }

            GetResponse()
                .BeginObject()
                .WriteKey(TStringBuf("switch"))
                .WriteString(switchRef->GetName());

            ProcessSwitchMetricsVerbose(GetResponse(), item.second);

            GetResponse().EndObject();
        }

        GetResponse()
            .EndList();
    }

    // api/v1/link_poller_metrics_verbose

    void TLinkPollerMetricsVerboseReply::Process() {
        const auto& topology = GetServerContext().GetTopologyStorage();
        const auto& metricsUpdater = GetServerContext().GetLinkPollerMetricsUpdater();
        auto metricsMap = metricsUpdater.GetCurrentMetrics();

        GetResponse().BeginList();

        for (const auto& item : *metricsMap) {
            auto switchRef = topology.FindSwitch(item.first);
            if (!switchRef) {
                continue;
            }

            GetResponse()
                .BeginObject()
                .WriteKey(TStringBuf("switch"))
                .WriteString(switchRef->GetName());

            ProcessLinkPollerMetricsVerbose(GetResponse(), item.second);

            GetResponse().EndObject();
        }

        GetResponse().EndList();
    }
}
