#pragma once

#include <infra/netmon/api/aggregator_request.h>
#include <infra/netmon/library/web_server.h>
#include <infra/netmon/topology/topology_storage.h>

#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/http/misc/parsed_request.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>

#include <util/generic/strbuf.h>

namespace NNetmon {
    template <class TDispatcher>
    class TBaseAggregatorReply: public TJsonHttpReply<TDispatcher, TAggregatorContext> {
    public:
        TBaseAggregatorReply(TAggregatorContext& serverContext, const TServiceRequest::TRef& clientContext)
            : TJsonHttpReply<TDispatcher, TAggregatorContext>(serverContext, clientContext)
            , RequestData(serverContext)
        {
        }

        void ParseRequest(THttpInput& input) override {
            TJsonHttpReply<TDispatcher, TAggregatorContext>::ParseRequest(input);
            RequestData.Parse(this->GetRequest());
        }
        inline const TRequestData& GetRequestData() const {
            return RequestData;
        }

    private:
        TRequestData RequestData;
    };

    class TSolomonFetcherReply: public THttpReply<THttpDispatcher, TAggregatorContext> {
    public:
        TSolomonFetcherReply(TAggregatorContext& serverContext, const TServiceRequest::TRef& clientContext)
            : THttpReply<THttpDispatcher, TAggregatorContext>(serverContext, clientContext)
            , Output(Response)
            , Encoder(NMonitoring::EncoderSpackV1(&Output,
                                                  NMonitoring::ETimePrecision::SECONDS,
                                                  NMonitoring::ECompression::ZSTD))
            , Interval(TInstant(), TInstant::Now())
        {
        }

        void PreprocessRequest(THttpInput&) {
        }

        void ParseRequest(THttpInput& input) {
            auto parsedRequest = TParsedHttpFull(input.FirstLine());
            if (parsedRequest.Cgi.Empty()) {
                return;
            }
            TCgiParameters params(parsedRequest.Cgi);
            const auto& nowParam = params.Get("now");
            const auto& periodParam = params.Get("period");
            TInstant now;
            TDuration period;

            if (nowParam && TInstant::TryParseIso8601(nowParam, now) &&
                periodParam && TDuration::TryParse(periodParam, period)) {
                Interval = {now - period, now};
            }
        }

        void WriteResponse(THttpOutput& output) {
            Encoder->Close();
            output << THttpResponse()
                .SetContentType(TStringBuf("application/x-solomon-spack"))
                .SetContent(Response);
        }

        const NMonitoring::IMetricEncoderPtr& GetEncoder() {
            return Encoder;
        }

        const std::pair<TInstant, TInstant>& GetRequestInterval() const {
            return Interval;
        }

        virtual void Process() = 0;

    private:
        TString Response;
        TStringOutput Output;
        NMonitoring::IMetricEncoderPtr Encoder;

        std::pair<TInstant, TInstant> Interval;
    };

    class TImmediateAggregatorReply: public TBaseAggregatorReply<THttpDispatcher> {
    public:
        using TBaseAggregatorReply::TBaseAggregatorReply;

        virtual void Process() = 0;
    };

    class TFutureAggregatorReply: public TBaseAggregatorReply<THttpFutureDispatcher> {
    public:
        using TBaseAggregatorReply::TBaseAggregatorReply;

        virtual NThreading::TFuture<void> Process() = 0;
    };

    class TResolverReply: public TJsonHttpReply<THttpDispatcher, TResolverContext> {
    public:
        TResolverReply(TResolverContext& serverContext, const TServiceRequest::TRef& clientContext)
            : TJsonHttpReply<THttpDispatcher, TResolverContext>(serverContext, clientContext)
            , RequestData(serverContext)
        {
        }

        void ParseRequest(THttpInput& input) override {
            TJsonHttpReply<THttpDispatcher, TResolverContext>::ParseRequest(input);
            RequestData.Parse(this->GetRequest());
        }
        inline const TResolverRequestData& GetRequestData() const {
            return RequestData;
        }

        virtual void Process() = 0;

    private:
        TResolverRequestData RequestData;
    };

    class TPingReply: public TImmediateAggregatorReply {
    public:
        using TImmediateAggregatorReply::TImmediateAggregatorReply;

        void Process() override;
    };

    class TTagsReply: public TImmediateAggregatorReply {
    public:
        using TImmediateAggregatorReply::TImmediateAggregatorReply;

        void Process() override;
    };

    class TSeenHostsReply: public TImmediateAggregatorReply {
    public:
        using TImmediateAggregatorReply::TImmediateAggregatorReply;

        void Process() override;
    };

    class TTerminatedHostsReply: public TImmediateAggregatorReply {
    public:
        using TImmediateAggregatorReply::TImmediateAggregatorReply;

        void Process() override;
    };

    class TInterestedHostsReply: public TCompressedJsonHttpReply<THttpDispatcher, TAggregatorContext> {
    public:
        using TCompressedJsonHttpReply::TCompressedJsonHttpReply;

        void Process();
    };

    class THistoryLimitsReply: public TImmediateAggregatorReply {
    public:
        using TImmediateAggregatorReply::TImmediateAggregatorReply;

        void Process() override;
    };

    class TQueueMappingReply: public TImmediateAggregatorReply {
    public:
        using TImmediateAggregatorReply::TImmediateAggregatorReply;

        void Process() override;
    };

    class TReplaceDeadHostsReply: public THttpReply<THttpDispatcher, TAggregatorContext> {
    public:
        using THttpReply<THttpDispatcher, TAggregatorContext>::THttpReply;

        void PreprocessRequest(THttpInput&) { }
        void ParseRequest(THttpInput& input);
        void Process();
        void WriteResponse(THttpOutput& output) {
            output << THttpResponse();
            output.Finish();
        }
    private:
        NJson::TJsonValue Request;
    };

    class TQualityStatsReply: public TImmediateAggregatorReply {
    public:
        using TImmediateAggregatorReply::TImmediateAggregatorReply;

        void Process() override;

    private:
        void MaintainerToJson(const TProbeAggregator& maintainer, const TExpressionStorage::TResult& expression);
        void LineMaintainerToJson(const TProbeAggregator& maintainer, const TExpressionStorage::TResult& expression);
        void RttHgramsToJson(const TProbeAggregator& maintainer, const TExpressionStorage::TResult& expression);
        void SeenHostsToJson(const TExpressionStorage::TResult& expression, const TSeenHostsMaintainer::TStateMap::TConstValueRef seenHosts, bool detailed);
        void ProbesToJson(const TProbeAggregator& aggregator, const TExpressionStorage::TResult& expression, bool detailed);
        void StateToJson(const TString& prefix, const TDatacenterPairState& state);
        void RttSignalToJson(const TString& signalName, const TSampleHistogram& rtt_histogram);
        void ConnSignalToJson(const TString& signalName, const TVector<ui64>& conn_histogram);
        TString GetSignalNamePrefix(const TProbeAggregator& aggregator, const TExpressionStorage::TResult& expression) const;

        /* Buckets for connectivity crossline histogram */
        const TVector<double> LineConnWeights = {
            0.0, 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9,
            0.905, 0.91, 0.915, 0.920, 0.925, 0.930, 0.935, 0.940, 0.945,
            0.95, 0.9525, 0.955, 0.9575, 0.96, 0.9625, 0.965, 0.9675,
            0.97, 0.9725, 0.975, 0.9775,
            0.98, 0.98125, 0.9825, 0.98375, 0.985, 0.98625, 0.9875, 0.98875,
            0.99, 0.99125, 0.9925, 0.99375, 0.995, 0.99625, 0.9975, 0.99875,
            0.999, 0.9999, 1.0
        };
    };

    class THostsReply: public TImmediateAggregatorReply {
    public:
        using TImmediateAggregatorReply::TImmediateAggregatorReply;

        void Process() override;
    };

    class TResolverHostsReply: public TResolverReply {
    public:
        using TResolverReply::TResolverReply;

        void Process() override;
    };

    class TFindByVersionReply: public TImmediateAggregatorReply {
    public:
        using TImmediateAggregatorReply::TImmediateAggregatorReply;

        void Process() override;
    };

    class TAgentReportReply: public TImmediateAggregatorReply {
    public:
        using TImmediateAggregatorReply::TImmediateAggregatorReply;

        void Process() override;
    };

    class TCoverageReply: public TImmediateAggregatorReply {
    public:
        using TImmediateAggregatorReply::TImmediateAggregatorReply;

        void Process() override;
    };

    class TSwitchMetricsReply: public TSolomonFetcherReply {
    public:
        using TSolomonFetcherReply::TSolomonFetcherReply;

        void Process() override;
    };

    class TInterSwitchMetricsVerboseReply: public TImmediateAggregatorReply {
    public:
        using TImmediateAggregatorReply::TImmediateAggregatorReply;

        void Process() override;
    };

    class TLinkPollerMetricsVerboseReply: public TImmediateAggregatorReply {
    public:
        using TImmediateAggregatorReply::TImmediateAggregatorReply;

        void Process() override;
    };

    using TPingHandler = THttpHandler<TPingReply>;
    using TTagsHandler = THttpHandler<TTagsReply>;
    using TSeenHostsHandler = THttpHandler<TSeenHostsReply>;
    using TTerminatedHostsHandler = THttpHandler<TTerminatedHostsReply>;
    using TInterestedHostsHandler = THttpHandler<TInterestedHostsReply>;
    using THistoryLimitsHandler = THttpHandler<THistoryLimitsReply>;
    using TQueueMappingHandler = THttpHandler<TQueueMappingReply>;

    using TReplaceDeadHostsHandler = THttpHandler<TReplaceDeadHostsReply>;

    using TQualityStatsHandler = THttpHandler<TQualityStatsReply>;
    using THostsHandler = THttpHandler<THostsReply>;
    using TResolverHostsHandler = THttpHandler<TResolverHostsReply>;
    using TFindByVersionHandler = THttpHandler<TFindByVersionReply>;
    using TAgentReportHandler = THttpHandler<TAgentReportReply>;
    using TCoverageHandler = THttpHandler<TCoverageReply>;
    using TSwitchMetricsHandler = THttpHandler<TSwitchMetricsReply>;
    using TInterSwitchMetricsVerboseHandler = THttpHandler<TInterSwitchMetricsVerboseReply>;
    using TLinkPollerMetricsVerboseHandler = THttpHandler<TLinkPollerMetricsVerboseReply>;
};
