#include <infra/netmon/api/aggregator_request.h>
#include <infra/netmon/settings.h>

#include <library/cpp/json/writer/json_value.h>
#include <library/cpp/json/json_reader.h>

#include <util/datetime/constants.h>
#include <util/generic/ymath.h>

namespace NNetmon {
    namespace {
        const char* DATETIME_FORMAT = "%Y-%m-%d %H:%M:%S";
        const char* DATE_FORMAT = "%Y-%m-%d";

        TInstant DateTimeFromJson(long long timestamp) {
            TInstant ts;
            if (timestamp > 0) {
                ts = TInstant::Seconds(timestamp);
            } else {
                ts = TInstant::Now() - TDuration::Seconds(Abs(timestamp));
            }
            return ts;
        }

        TInstant DateTimeFromJson(const NJson::TJsonValue& obj) {
            return DateTimeFromJson(obj.GetIntegerSafe());
        }

        TVector<TTopology::THostRef> GetMultipleHosts(const TTopologyStorage& topologyStorage,
                                                      const TVector<TString>& multipleHosts,
                                                      const TVector<ui64>& multipleHostIds) {
            TVector<TTopology::THostRef> hosts;
            for (const auto& hostName : multipleHosts) {
                const auto host(topologyStorage.FindHost(hostName));
                if (!host) {
                    const auto iface(topologyStorage.FindHostInterface(hostName));
                    if (!iface) {
                        ythrow TNotFoundError() << "host not found";
                    } else {
                        hosts.emplace_back(iface.GetHost());
                    }
                } else {
                    hosts.emplace_back(host);
                }
            }
            for (const auto hostId : multipleHostIds) {
                const auto host(topologyStorage.FindHost(hostId));
                if (!host) {
                    const auto iface(topologyStorage.FindHostInterface(hostId));
                    if (!iface) {
                        ythrow TNotFoundError() << "host not found";
                    } else {
                        hosts.emplace_back(iface.GetHost());
                    }
                } else {
                    hosts.emplace_back(host);
                }
            }
            return hosts;
        }
    } // end namespace

    TAggregatorContext::THostStats TAggregatorContext::GetHostStats() const noexcept {
        const auto topology(GetTopologyStorage().GetTopology());
        const auto seenHosts(GetSeenHostsUpdater().GetHosts());
        const auto versions(GetFinishedTaskIndex().VersionHistogram());

        ui64 totalBareCounter = 0;
        ui64 totalVirtualCounter = 0;
        ui64 withVersionCounter = 0;
        ui64 lastestVersionCounter = 0;
        ui64 seenBareCounter = 0;
        ui64 seenVirtualCounter = 0;

        topology->ForEachHost([&](const THost& host) {
            ui64 seen = (seenHosts->find(host).IsEnd()) ? 0 : 1;
            if (host.IsVirtual()) {
                totalVirtualCounter++;
                seenVirtualCounter += seen;
            } else {
                totalBareCounter++;
                seenBareCounter += seen;
            }
        });

        TAgentVersion lastestVersion(0, 0, 0, 0);
        for (const auto& pair : versions) {
            auto version(ParseAgentVersion(pair.first));
            if (lastestVersion < version) {
                lastestVersion = version;
                lastestVersionCounter = pair.second;
            }
            withVersionCounter += pair.second;
        }

        return {
            totalBareCounter,
            totalVirtualCounter,
            withVersionCounter,
            lastestVersionCounter,
            seenBareCounter,
            seenVirtualCounter
        };
    }

    void TResolverRequestData::Parse(const NJson::TJsonValue& input) {
        if (!input.IsDefined()) {
            return;
        }
        const auto& topologyStorage(Context.GetTopologyStorage());
        for (const auto& it : input.GetMapSafe()) {
            if (it.first == TStringBuf("hosts")) {
                for (const auto& host : it.second.GetArraySafe()) {
                    if (!topologyStorage.IsValidName(host.GetStringSafe()))
                        ythrow TValidationError() << "host isn't valid";
                    MultipleHosts.emplace_back(host.GetStringSafe());
                }
            } else if (it.first == TStringBuf("host_ids")) {
                for (const auto& host : it.second.GetArraySafe()) {
                    MultipleHostIds.emplace_back(host.GetUIntegerSafe());
                }
            } else {
                ythrow TValidationError() << "unknown key " << it.first << " given";
            }
        }
    }

    TVector<TTopology::THostRef> TRequestData::GetMultipleHosts() const {
        return ::NNetmon::GetMultipleHosts(Context.GetTopologyStorage(), MultipleHosts, MultipleHostIds);
    }

    void TRequestData::Parse(const NJson::TJsonValue& input) {
        if (!input.IsDefined()) {
            return;
        }
        const auto& topologyStorage(Context.GetTopologyStorage());
        for (const auto& it : input.GetMapSafe()) {
            if (it.first == TStringBuf("fqdn") || it.first == TStringBuf("host")) {
                Host = it.second.GetStringSafe();
                if (!topologyStorage.IsValidName(Host))
                    ythrow TValidationError() << "host isn't valid";
            } else if (it.first == TStringBuf("hosts")) {
                for (const auto& host : it.second.GetArraySafe()) {
                    if (!topologyStorage.IsValidName(host.GetStringSafe()))
                        ythrow TValidationError() << "host isn't valid";
                    MultipleHosts.emplace_back(host.GetStringSafe());
                }
            } else if (it.first == TStringBuf("host_ids")) {
                for (const auto& host : it.second.GetArraySafe()) {
                    MultipleHostIds.emplace_back(host.GetUIntegerSafe());
                }
            } else if (it.first == TStringBuf("source_dc")) {
                SourceDc = it.second.GetStringSafe();
                if (!topologyStorage.IsValidName(SourceDc))
                    ythrow TValidationError() << "source dc isn't valid";
            } else if (it.first == TStringBuf("target_dc")) {
                TargetDc = it.second.GetStringSafe();
                if (!topologyStorage.IsValidName(TargetDc))
                    ythrow TValidationError() << "target dc isn't valid";
            } else if (it.first == TStringBuf("source_queue")) {
                SourceQueue = it.second.GetStringSafe();
                if (!topologyStorage.IsValidName(SourceQueue))
                    ythrow TValidationError() << "source queue isn't valid";
            } else if (it.first == TStringBuf("target_queue")) {
                TargetQueue = it.second.GetStringSafe();
                if (!topologyStorage.IsValidName(TargetQueue))
                    ythrow TValidationError() << "target queue isn't valid";
            } else if (it.first == TStringBuf("source_switch")) {
                SourceSwitch = it.second.GetStringSafe();
                if (!topologyStorage.IsValidName(SourceSwitch))
                    ythrow TValidationError() << "source switch isn't valid";
            } else if (it.first == TStringBuf("target_switch")) {
                TargetSwitch = it.second.GetStringSafe();
                if (!topologyStorage.IsValidName(TargetSwitch))
                    ythrow TValidationError() << "target switch isn't valid";
            } else if (it.first == TStringBuf("protocol")) {
                const TString& protocol = it.second.GetStringSafe();
                if (protocol == TStringBuf("icmp")) {
                    ProtocolType = ICMP;
                } else if (protocol == TStringBuf("udp")) {
                    ProtocolType = UDP;
                } else if (protocol == TStringBuf("tcp")) {
                    ProtocolType = TCP;
                } else {
                    ythrow TValidationError() << "unknown protocol " << protocol << " given";
                }
            } else if (it.first == TStringBuf("network")) {
                const TString& network = it.second.GetStringSafe();
                if (network == TStringBuf("bb4")) {
                    NetworkType = BACKBONE4;
                } else if (network == TStringBuf("bb6")) {
                    NetworkType = BACKBONE6;
                } else if (network == TStringBuf("fb6")) {
                    NetworkType = FASTBONE6;
                } else if (network == TStringBuf("fb-cs1")) {
                    NetworkType = FASTBONE_CS1;
                } else if (network == TStringBuf("fb-cs2")) {
                    NetworkType = FASTBONE_CS2;
                } else if (network == TStringBuf("bb-cs3")) {
                    NetworkType = BACKBONE_CS3;
                } else if (network == TStringBuf("bb-cs4")) {
                    NetworkType = BACKBONE_CS4;
                } else if (network == TStringBuf("mtn-bb6")) {
                    NetworkType = SPARSE_BACKBONE6;
                } else if (network == TStringBuf("mtn-fb6")) {
                    NetworkType = SPARSE_FASTBONE6;
                } else {
                    ythrow TValidationError() << "unknown network " << network << " given";
                }
            } else if (it.first == TStringBuf("generated")) {
                Generated = DateTimeFromJson(it.second);
            } else if (it.first == TStringBuf("since")) {
                Since = DateTimeFromJson(it.second);
            } else if (it.first == TStringBuf("until")) {
                Until = DateTimeFromJson(it.second);
            } else if (it.first == TStringBuf("tag")) {
                Expression = it.second.GetStringSafe();
                if (!ValidateGroup(Expression)) {
                    ythrow TValidationError() << "invalid tag '" << Expression << "' given";
                }
                const auto& expressionStorage(Context.GetExpressionStorage());
                auto state(expressionStorage.Get(Expression));
                if (!state.Defined()) {
                    ythrow TValidationError() << "expression '" << Expression << "' does not exist";
                }
                ExpressionId =  state->ExpressionId;
            } else if (it.first == TStringBuf("expression")) {
                Expression = it.second.GetStringSafe();
                const auto& expressionStorage(Context.GetExpressionStorage());
                auto state(expressionStorage.Get(Expression));
                if (!state.Defined()) {
                    ythrow TValidationError() << "expression '" << Expression << "' does not exist";
                }

                ExpressionId = state->ExpressionId;
            } else if (it.first == TStringBuf("limit")) {
                Limit = it.second.GetUIntegerSafe();
            } else if (it.first == TStringBuf("min_score")) {
                MinScore = it.second.GetDouble();
            } else if (it.first == TStringBuf("max_score")) {
                MaxScore = it.second.GetDouble();
            } else if (it.first == TStringBuf("min_rtt")) {
                MinRtt = it.second.GetDouble() / 1000.0;
            } else if (it.first == TStringBuf("max_rtt")) {
                MaxRtt = it.second.GetDouble() / 1000.0;
            } else if (it.first == TStringBuf("sort_by")) {
                const TString& sortBy = it.second.GetStringSafe();
                TString field;
                SortOrder = SORT_ASCENDING;
                if (sortBy.StartsWith('-')) {
                    SortOrder = SORT_DESCENDING;
                    field = sortBy.substr(1);
                } else if (sortBy.StartsWith('+')) {
                    field = sortBy.substr(1);
                } else {
                    field = sortBy;
                }
                if (field == TStringBuf("score")) {
                    SortField = SORT_SCORE;
                } else if (field == TStringBuf("rtt")) {
                    SortField = SORT_RTT;
                } else {
                    ythrow TValidationError() << "unknown field " << field << " given";
                }
            } else if (it.first == TStringBuf("version")) {
                Version = it.second.GetStringSafe();
            } else {
                ythrow TValidationError() << "unknown key " << it.first << " given";
            }
        }
        if (Until == TInstant::Zero()) {
            Until = TInstant::Now();
        }
        if (Since == TInstant::Zero()) {
            Since = TInstant::Now() - TSettings::Get()->GetDcAggregationWindow();
        }
    }

    TTopology::THostRef TRequestData::GetHost() const {
        const auto host(Context.GetTopologyStorage().FindHost(GetHostName()));
        if (!host) {
            const auto iface(Context.GetTopologyStorage().FindHostInterface(GetHostName()));
            if (iface) {
                return iface.GetHost();
            } else {
                ythrow TNotFoundError() << "host not found";
            }
        }
        return host;
    }

    TVector<TTopology::THostRef> TResolverRequestData::GetMultipleHosts() const {
        return ::NNetmon::GetMultipleHosts(Context.GetTopologyStorage(), MultipleHosts, MultipleHostIds);
    }

    TTopology::THostInterfaceRef TRequestData::GetHostInterface() const {
        const auto iface(Context.GetTopologyStorage().FindHostInterface(GetHostName()));
        if (!iface) {
            ythrow TNotFoundError() << "host interface not found";
        }
        return iface;
    }

    const TString& TRequestData::GetVersion() const {
        if (Version.empty()) {
            ythrow TNotFoundError() << "version not given";
        }
        return Version;
    }

    TTopology::TDatacenterRef TRequestData::GetSourceDatacenter(bool safe) const {
        return Context.GetTopologyStorage().FindDatacenter(GetSourceDatacenterName(safe));
    }

    TTopology::TDatacenterRef TRequestData::GetTargetDatacenter(bool safe) const {
        return Context.GetTopologyStorage().FindDatacenter(GetTargetDatacenterName(safe));
    }

    TTopology::TLineRef TRequestData::GetSourceLine(bool safe) const {
        auto dcName = GetSourceDatacenterName(false);

        if (dcName.empty()) {
            return Context.GetTopologyStorage().FindQueue(GetSourceLineName(safe));
        } else {
            return Context.GetTopologyStorage().FindQueue(dcName, GetSourceLineName(safe));
        }
    }

    TTopology::TLineRef TRequestData::GetTargetLine(bool safe) const {
        auto dcName = GetTargetDatacenterName(false);

        if (dcName.empty()) {
            return Context.GetTopologyStorage().FindQueue(GetTargetLineName(safe));
        } else {
            return Context.GetTopologyStorage().FindQueue(dcName, GetTargetLineName(safe));
        }
    }

    TTopology::TSwitchRef TRequestData::GetSourceSwitch(bool safe) const {
        auto& dcName = GetSourceDatacenterName(false);
        auto& lineName = GetSourceLineName(false);
        auto& switchName = GetSourceSwitchName(safe);

        if (dcName.empty() || lineName.empty()) {
            return Context.GetTopologyStorage().FindSwitch(switchName);
        } else {
            return Context.GetTopologyStorage().FindSwitch(dcName, lineName, switchName);
        }
    }

    TTopology::TSwitchRef TRequestData::GetTargetSwitch(bool safe) const {
        auto& dcName = GetTargetDatacenterName(false);
        auto& lineName = GetTargetLineName(false);
        auto& switchName = GetTargetSwitchName(safe);

        if (dcName.empty() || lineName.empty()) {
            return Context.GetTopologyStorage().FindSwitch(switchName);
        } else {
            return Context.GetTopologyStorage().FindSwitch(dcName, lineName, switchName);
        }
    }

    TDatacenterPairKey TRequestData::GetDatacenterPair(bool safe) const {
        const auto source(GetSourceDatacenter(safe));
        const auto target(GetTargetDatacenter(safe));
        TDatacenterPairKey key(target, source);
        if (safe && !key.IsValid()) {
            ythrow TNotFoundError() << "datacenter not found";
        }
        return key;
    }

    TLinePairKey TRequestData::GetLinePair(bool safe) const {
        const auto source(GetSourceLine(safe));
        const auto target(GetTargetLine(safe));
        TLinePairKey key(target, source);
        if (safe && !key.IsValid()) {
            ythrow TNotFoundError() << "queue not found";
        }
        return key;
    }

    TSwitchPairKey TRequestData::GetSwitchPair(bool safe) const {
        const auto source(GetSourceSwitch(safe));
        const auto target(GetTargetSwitch(safe));
        TSwitchPairKey key(target, source);
        if (safe && !key.IsValid()) {
            ythrow TNotFoundError() << "switch not found";
        }
        return key;
    }

    const TString& TRequestData::GetHostName() const {
        if (!Host)
            ythrow TValidationError() << "host not set";
        return Host;
    }

    const TString& TRequestData::GetSourceDatacenterName(bool safe) const {
        if (safe && !SourceDc)
            ythrow TValidationError() << "source_dc not set";
        return SourceDc;
    }

    const TString& TRequestData::GetTargetDatacenterName(bool safe) const {
        if (safe && !TargetDc)
            ythrow TValidationError() << "target_dc not set";
        return TargetDc;
    }

    const TString& TRequestData::GetSourceLineName(bool safe) const {
        if (safe && !SourceQueue)
            ythrow TValidationError() << "source_queue not set";
        return SourceQueue;
    }

    const TString& TRequestData::GetTargetLineName(bool safe) const {
        if (safe && !TargetQueue)
            ythrow TValidationError() << "target_queue not set";
        return TargetQueue;
    }

    const TString& TRequestData::GetSourceSwitchName(bool safe) const {
        if (safe && !SourceSwitch)
            ythrow TValidationError() << "source_switch not set";
        return SourceSwitch;
    }

    const TString& TRequestData::GetTargetSwitchName(bool safe) const {
        if (safe && !TargetSwitch)
            ythrow TValidationError() << "target_switch not set";
        return TargetSwitch;
    }

    bool TRequestData::HasSourceDc() const {
        return !!GetSourceDatacenter(false);
    }

    bool TRequestData::HasTargetDc() const {
        return !!GetTargetDatacenter(false);
    }

    bool TRequestData::HasSourceQueue() const {
        return !!GetSourceLine(false);
    }

    bool TRequestData::HasTargetQueue() const {
        return !!GetTargetLine(false);
    }

    bool TRequestData::HasSourceSwitch() const {
        return !!GetSourceSwitch(false);
    }

    bool TRequestData::HasTargetSwitch() const {
        return !!GetTargetSwitch(false);
    }

    EProtocolType TRequestData::GetProtocol() const {
        if (ProtocolType == NIL_PROTOCOL)
            ythrow TValidationError() << "protocol not set";
        return ProtocolType;
    }

    ENetworkType TRequestData::GetNetwork() const {
        if (NetworkType == NIL_NETWORK)
            ythrow TValidationError() << "network not set";
        return NetworkType;
    }

    const TString& TRequestData::GetExpression() const {
        if (!Expression)
            ythrow TValidationError() << "expression not set";
        return Expression;
    }

    TExpressionId TRequestData::GetExpressionId() const {
        if (!ExpressionId)
            ythrow TValidationError() << "expression not set";
        return ExpressionId;
    }

    TProbeAggregatorKey TRequestData::GetAggregatorKey() const {
        return TProbeAggregatorKey{
            GetExpressionId(),
            GetNetwork(),
            GetProtocol()
        };
    }

    TInstant TRequestData::GetGenerated() const {
        if (Generated == TInstant::Zero())
            ythrow TValidationError() << "generated not set";
        return Generated;
    }

    const TInstant& TRequestData::GetSince() const {
        if (Since == TInstant::Zero())
            ythrow TValidationError() << "since not set";
        return Since;
    }

    TString TRequestData::GetSinceHuman() const {
        struct tm currentTime;
        GetSince().LocalTime(&currentTime);
        return Strftime(DATETIME_FORMAT, &currentTime);
    }

    TString TRequestData::GetSinceHumanDate() const {
        struct tm currentTime;
        GetSince().LocalTime(&currentTime);
        return Strftime(DATE_FORMAT, &currentTime);
    }

    const TInstant& TRequestData::GetUntil() const {
        if (Until == TInstant::Zero())
            ythrow TValidationError() << "until not set";
        return Until;
    }

    TString TRequestData::GetUntilHuman() const {
        struct tm currentTime;
        GetUntil().LocalTime(&currentTime);
        return Strftime(DATETIME_FORMAT, &currentTime);
    }

    TString TRequestData::GetUntilHumanDate() const {
        struct tm currentTime;
        (GetSince() + TDuration::Seconds(SECONDS_IN_DAY)).LocalTime(&currentTime);
        return Strftime(DATE_FORMAT, &currentTime);
    }

    ui64 TRequestData::GetLimit() const {
        if (!Limit)
            ythrow TValidationError() << "limit not set";
        return Limit;
    }

    double TRequestData::GetMinScore() const {
        if (MinScore > MaxScore)
            ythrow TValidationError() << "min_score is greather than max_score";
        return MinScore;
    }

    double TRequestData::GetMaxScore() const {
        if (MinScore > MaxScore)
            ythrow TValidationError() << "min_score is greather than max_score";
        return MaxScore;
    }

    bool TRequestData::HasMinRtt() const {
        return !IsNan(MinRtt);
    }

    bool TRequestData::HasMaxRtt() const {
        return !IsNan(MaxRtt);
    }

    double TRequestData::GetMinRtt() const {
        if (!HasMinRtt())
            ythrow TValidationError() << "min_rtt not defined";
        if (HasMaxRtt() && MinRtt > MaxRtt)
            ythrow TValidationError() << "min_rtt is greather than max_rtt";
        return MinRtt;
    }

    double TRequestData::GetMaxRtt() const {
        if (!HasMaxRtt())
            ythrow TValidationError() << "max_rtt not defined";
        if (HasMinRtt() && MinRtt > MaxRtt)
            ythrow TValidationError() << "min_rtt is greather than max_rtt";
        return MaxRtt;
    }

    TRequestData::ESortField TRequestData::GetSortField() const {
        return SortField;
    }

    TRequestData::ESortOrder TRequestData::GetSortOrder() const {
        return SortOrder;
    }

    TProbeAggregator::TRef TRequestData::GetProbeAggregator() const {
        const TProbeAggregatorKey key{GetExpressionId(), GetNetwork(), GetProtocol()};
        try {
            return Context.GetAggregatorMaintainer().GetAggregator(key);
        } catch (...) {
            ythrow TNotFoundError() << CurrentExceptionMessage();
        }
    }
}
