#pragma once

#include <infra/netmon/metrics_updater.h>
#include <infra/netmon/probe_aggregator_maintainer.h>
#include <infra/netmon/probe_dumper.h>
#include <infra/netmon/probe_schedule_maintainer.h>
#include <infra/netmon/tasks/finished_task_index.h>
#include <infra/netmon/topology/clients/walle.h>
#include <infra/netmon/seen_hosts.h>
#include <infra/netmon/terminated_hosts.h>
#include <infra/netmon/history_requester.h>
#include <infra/netmon/user_data_storage.h>
#include <infra/netmon/topology/topology_storage.h>
#include <infra/netmon/library/api_handler_helpers.h>

#include <cmath>

namespace NNetmon {
    class TAggregatorContext : public TNonCopyable {
    public:
        TAggregatorContext(
                const TExpressionStorage& expressionStorage,
                const TUserDataStorage& userDataStorage,
                const TTopologyStorage& topologyStorage,
                const TSliceCollector& sliceCollector,
                const TSlicerDumper& probeDumper,
                const TProbeAggregatorMaintainer& aggregatorMaintainer,
                const TSeenHostsUpdater& seenHostsUpdater,
                const TSeenHostsMaintainer& seenHostsMaintainer,
                const TTerminatedHostsUpdater& terminatedHostsUpdater,
                TWalleUpdater& walleUpdater,
                const TInfraUpdater& infraUpdater,
                const TFinishedTaskIndex& finishedTaskIndex,
                const THistoryRequester& historyRequester,
                const TProbeScheduleMaintainer& probeScheduleMaintainer,
                TInterSwitchMetricsUpdater& switchSlaMetricsUpdater,
                TInterSwitchRttMetricsUpdater& switchSlaRttMetricsUpdater,
                TLinkPollerMetricsUpdater& linkPollerMetricsUpdater,
                TInterSwitchMetricsUpdater& switchSlaLPMetricsUpdater,
                TInterSwitchRttMetricsUpdater& switchSlaRttLPMetricsUpdater)
            : ExpressionStorage(expressionStorage)
            , UserDataStorage(userDataStorage)
            , TopologyStorage(topologyStorage)
            , SliceCollector(sliceCollector)
            , ProbeDumper(probeDumper)
            , AggregatorMaintainer(aggregatorMaintainer)
            , SeenHostsUpdater(seenHostsUpdater)
            , SeenHostsMaintainer(seenHostsMaintainer)
            , TerminatedHostsUpdater(terminatedHostsUpdater)
            , WalleUpdater(walleUpdater)
            , InfraUpdater(infraUpdater)
            , FinishedTaskIndex(finishedTaskIndex)
            , HistoryRequester(historyRequester)
            , ProbeScheduleMaintainer(probeScheduleMaintainer)
            , InterSwitchMetricsUpdater(switchSlaMetricsUpdater)
            , InterSwitchRttMetricsUpdater(switchSlaRttMetricsUpdater)
            , LinkPollerMetricsUpdater(linkPollerMetricsUpdater)
            , InterSwitchLPMetricsUpdater(switchSlaLPMetricsUpdater)
            , InterSwitchRttLPMetricsUpdater(switchSlaRttLPMetricsUpdater)
        {
        }

        inline const TExpressionStorage& GetExpressionStorage() const noexcept {
            return ExpressionStorage;
        }
        inline const TUserDataStorage& GetUserDataStorage() const noexcept {
            return UserDataStorage;
        }
        inline const TTopologyStorage& GetTopologyStorage() const noexcept {
            return TopologyStorage;
        }
        inline const TSliceCollector& GetSliceCollector() const noexcept {
            return SliceCollector;
        }
        inline const TSlicerDumper& GetProbeDumper() const noexcept {
            return ProbeDumper;
        }
        inline const TProbeAggregatorMaintainer& GetAggregatorMaintainer() const noexcept {
            return AggregatorMaintainer;
        }
        inline const TSeenHostsUpdater& GetSeenHostsUpdater() const noexcept {
            return SeenHostsUpdater;
        }
        inline const TSeenHostsMaintainer& GetSeenHostsMaintainer() const noexcept {
            return SeenHostsMaintainer;
        }
        inline const TTerminatedHostsUpdater& GetTerminatedHostsUpdater() const noexcept {
            return TerminatedHostsUpdater;
        }
        inline const TWalleUpdater& GetWalleUpdater() const noexcept {
            return WalleUpdater;
        }
        inline TWalleUpdater& GetWalleUpdater() noexcept {
            return WalleUpdater;
        }
        inline const TInfraUpdater& GetInfraUpdater() const noexcept {
            return InfraUpdater;
        }
        inline const TFinishedTaskIndex& GetFinishedTaskIndex() const noexcept {
            return FinishedTaskIndex;
        }
        inline const THistoryRequester& GetHistoryRequester() const noexcept {
            return HistoryRequester;
        }
        inline const TProbeScheduleMaintainer& GetProbeScheduleMaintainer() const noexcept {
            return ProbeScheduleMaintainer;
        }
        inline TInterSwitchMetricsUpdater& GetInterSwitchMetricsUpdater() noexcept {
            return InterSwitchMetricsUpdater;
        }
        inline TInterSwitchRttMetricsUpdater& GetInterSwitchRttMetricsUpdater() noexcept {
            return InterSwitchRttMetricsUpdater;
        }
        inline TLinkPollerMetricsUpdater& GetLinkPollerMetricsUpdater() noexcept {
            return LinkPollerMetricsUpdater;
        }
        inline TInterSwitchMetricsUpdater& GetInterSwitchLPMetricsUpdater() noexcept {
            return InterSwitchLPMetricsUpdater;
        }
        inline TInterSwitchRttMetricsUpdater& GetInterSwitchRttLPMetricsUpdater() noexcept {
            return InterSwitchRttLPMetricsUpdater;
        }

        struct THostStats {
            const ui64 TotalBare = 0;
            const ui64 TotalVirtual = 0;
            const ui64 WithVersion = 0;
            const ui64 LastestVersion = 0;
            const ui64 SeenBare = 0;
            const ui64 SeenVirtual = 0;
        };

        THostStats GetHostStats() const noexcept;

    private:
        const TExpressionStorage& ExpressionStorage;
        const TUserDataStorage& UserDataStorage;
        const TTopologyStorage& TopologyStorage;
        const TSliceCollector& SliceCollector;
        const TSlicerDumper& ProbeDumper;
        const TProbeAggregatorMaintainer& AggregatorMaintainer;
        const TSeenHostsUpdater& SeenHostsUpdater;
        const TSeenHostsMaintainer& SeenHostsMaintainer;
        const TTerminatedHostsUpdater& TerminatedHostsUpdater;
        TWalleUpdater& WalleUpdater;
        const TInfraUpdater& InfraUpdater;
        const TFinishedTaskIndex& FinishedTaskIndex;
        const THistoryRequester& HistoryRequester;
        const TProbeScheduleMaintainer& ProbeScheduleMaintainer;
        TInterSwitchMetricsUpdater& InterSwitchMetricsUpdater;
        TInterSwitchRttMetricsUpdater& InterSwitchRttMetricsUpdater;
        TLinkPollerMetricsUpdater& LinkPollerMetricsUpdater;
        TInterSwitchMetricsUpdater& InterSwitchLPMetricsUpdater;
        TInterSwitchRttMetricsUpdater& InterSwitchRttLPMetricsUpdater;
    };

    class TResolverContext : public TNonCopyable {
    public:
        TResolverContext(const TTopologyStorage& topologyStorage)
            : TopologyStorage(topologyStorage)
        {
        }

        inline const TTopologyStorage& GetTopologyStorage() const noexcept {
            return TopologyStorage;
        }
    private:
        const TTopologyStorage& TopologyStorage;
    };

    class TResolverRequestData {
    public:
        TResolverRequestData(TResolverContext& context)
            : Context(context)
        {
        }

        void Parse(const NJson::TJsonValue& input);
        TVector<TTopology::THostRef> GetMultipleHosts() const;

    private:
        const TResolverContext& Context;
        TVector<TString> MultipleHosts;
        TVector<ui64> MultipleHostIds;
    };

    class TRequestData {
    public:
        enum ESortOrder {
            SORT_DESCENDING,
            SORT_ASCENDING
        };

        enum ESortField {
            SORT_SCORE,
            SORT_RTT
        };

        TRequestData(TAggregatorContext& context)
            : Context(context)
        {
        }

        void Parse(const NJson::TJsonValue& input);

        TTopology::THostRef GetHost() const;
        TVector<TTopology::THostRef> GetMultipleHosts() const;
        TTopology::THostInterfaceRef GetHostInterface() const;

        const TString& GetVersion() const;

        TTopology::TDatacenterRef GetSourceDatacenter(bool safe=true) const;
        TTopology::TDatacenterRef GetTargetDatacenter(bool safe=true) const;
        TTopology::TLineRef GetSourceLine(bool safe=true) const;
        TTopology::TLineRef GetTargetLine(bool safe=true) const;
        TTopology::TSwitchRef GetSourceSwitch(bool safe=true) const;
        TTopology::TSwitchRef GetTargetSwitch(bool safe=true) const;

        TDatacenterPairKey GetDatacenterPair(bool safe=true) const;
        TLinePairKey GetLinePair(bool safe=true) const;
        TSwitchPairKey GetSwitchPair(bool safe=true) const;

        const TString& GetHostName() const;
        const TString& GetSourceDatacenterName(bool safe=true) const;
        const TString& GetTargetDatacenterName(bool safe=true) const;
        const TString& GetSourceLineName(bool safe=true) const;
        const TString& GetTargetLineName(bool safe=true) const;
        const TString& GetSourceSwitchName(bool safe=true) const;
        const TString& GetTargetSwitchName(bool safe=true) const;

        bool HasSourceDc() const;
        bool HasTargetDc() const;
        bool HasSourceQueue() const;
        bool HasTargetQueue() const;
        bool HasSourceSwitch() const;
        bool HasTargetSwitch() const;

        EProtocolType GetProtocol() const;
        ENetworkType GetNetwork() const;
        const TString& GetExpression() const;
        TExpressionId GetExpressionId() const;
        TProbeAggregatorKey GetAggregatorKey() const;

        TInstant GetGenerated() const;

        const TInstant& GetSince() const;
        TString GetSinceHuman() const;
        TString GetSinceHumanDate() const;

        const TInstant& GetUntil() const;
        TString GetUntilHuman() const;
        TString GetUntilHumanDate() const;

        ui64 GetLimit() const;

        double GetMinScore() const;
        double GetMaxScore() const;

        bool HasMinRtt() const;
        bool HasMaxRtt() const;
        double GetMinRtt() const;
        double GetMaxRtt() const;

        ESortField GetSortField() const;
        ESortOrder GetSortOrder() const;

        TProbeAggregator::TRef GetProbeAggregator() const;

        const TTopologyStorage& GetTopologyStorage() const {
            return Context.GetTopologyStorage();
        }
        TTopologySelector::TBox::TConstValueRef GetTopologySelector() const {
            return GetTopologyStorage().GetTopologySelector();
        }

        const TTopologyStorage::THostSetBox::TConstValueRef GetDeadHosts() const {
            return Context.GetWalleUpdater().GetHosts();
        }
        const TTopologyStorage::THostSetBox::TConstValueRef GetSeenHosts() const {
            return Context.GetSeenHostsUpdater().GetHosts();
        }

    private:
        const TAggregatorContext& Context;

        TString Host;
        TVector<TString> MultipleHosts;
        TVector<ui64> MultipleHostIds;

        TString Version;

        TString SourceDc;
        TString TargetDc;
        TString SourceQueue;
        TString TargetQueue;
        TString SourceSwitch;
        TString TargetSwitch;

        EProtocolType ProtocolType = NIL_PROTOCOL;
        ENetworkType NetworkType = NIL_NETWORK;

        TString Expression;
        TExpressionId ExpressionId = 0;

        TInstant Generated;
        TInstant Since;
        TInstant Until;

        ui64 Limit = 1000;

        double MinScore = 0.0;
        double MaxScore = 1.0;

        double MinRtt = NAN;
        double MaxRtt = NAN;

        ESortField SortField = SORT_SCORE;
        ESortOrder SortOrder = SORT_ASCENDING;
    };
};
