#include <infra/netmon/api/client_api.h>
#include <infra/netmon/topology/settings.h>
#include <infra/netmon/terminated_hosts.h>
#include <infra/netmon/probe_dumper.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/metrics.h>

#include <util/generic/algorithm.h>
#include <util/generic/xrange.h>

#if defined(_linux_)
#   include <netinet/ip.h>
#else
#   define IPTOS_CLASS_CS0 0x0
#   define IPTOS_CLASS_CS1 0x20
#   define IPTOS_CLASS_CS2 0x40
#   define IPTOS_CLASS_CS3 0x60
#   define IPTOS_CLASS_CS4 0x80
#endif

namespace NNetmon {
    TVector<TPacketSlaCounters> PacketCounters;
    TVector<TRttSlaCounters> RttCounters;

    namespace {
        const ui64 MICROSECONDS_IN_SECOND = 1000000UL;
        const ui64 MILLISECONDS_IN_SECOND = 1000UL;

        inline bool AreKnownQueues(const TTopology::THostRef sourceHost,
                                   const TTopology::THostRef targetHost)
        {
            const auto& filter(TTopologySettings::Get()->GetKnownQueuesFilter());
            return TTopologySettings::Get()->GetUsePodAsQueue() ||
                   (filter.Check(sourceHost->GetLine().GetName()) && filter.Check(targetHost->GetLine().GetName()));
        }

        inline NClient::EProtocol ProtocolToProto(NNetmon::EProtocolType protocol) {
            switch (protocol) {
                case NNetmon::EProtocolType::ICMP:
                    return NClient::EProtocol::ICMP;
                case NNetmon::EProtocolType::UDP:
                    return NClient::EProtocol::UDP;
                case NNetmon::EProtocolType::TCP:
                    return NClient::EProtocol::TCP;
                case NNetmon::EProtocolType::LINK_POLLER:
                    return NClient::EProtocol::LINK_POLLER;

                default:
                    return NClient::EProtocol::UNKNOWN_PROTOCOL;
            }
        }

        inline NClient::ETrafficClass TrafficClassToProto(NNetmon::ETrafficClassType trafficClass) {
            switch (trafficClass) {
                case NNetmon::ETrafficClassType::CS0:
                    return NClient::ETrafficClass::CS0;
                case NNetmon::ETrafficClassType::CS1:
                    return NClient::ETrafficClass::CS1;
                case NNetmon::ETrafficClassType::CS2:
                    return NClient::ETrafficClass::CS2;
                case NNetmon::ETrafficClassType::CS3:
                    return NClient::ETrafficClass::CS3;
                case NNetmon::ETrafficClassType::CS4:
                    return NClient::ETrafficClass::CS4;
                default:
                    // there are no special value, return default backbone tc
                    return NClient::ETrafficClass::CS3;
            }
        }

        struct TIpv4ScheduleConfig {
            static const TVector<NNetmon::ETrafficClassType>& GetTrafficClasses() {
                static TVector<NNetmon::ETrafficClassType> trafficClasses = {CS0};
                return trafficClasses;
            }

            static NClient::EFamily GetFamily() {
                return NClient::INET4;
            }

            static TMaybe<int> GetVlan() {
                return Nothing();
            }

            static TMaybe<NClient::TScheduledProbesResponse::TTarget> TargetToProto(const THost& host) {
                auto iface = host.GetIpv4Interface();
                if (!iface) {
                    return Nothing();
                }
                NClient::TScheduledProbesResponse::TTarget protoTarget;
                protoTarget.SetHostname(iface->GetName());
                protoTarget.SetIp(iface->GetIpv4().ToString());
                return protoTarget;
            }
        };

        // TDerived needs to implement the following static methods:
        //
        // const THostInterface* GetInterface(THost&)
        //   - chooses host interface to use in schedule
        //
        // TVector<NNetmon::ETrafficClassType> GetAllowedTrafficClasses()
        //   - returns all traffic classes that can be possibly used in this subschedule
        //
        // THashSet<NNetmon::ETrafficClassType> GetRequestedTrafficClasses()
        //   - returns all traffic classes that we need to build subschedules for
        //     (i.e. Settings::GetProbeScheduleTrafficClasses, GetProbeScheduleCrossDcTrafficClasses)
        //
        // TDerived also may reimplement GetVlan().
        template <class TDerived>
        struct TBaseIpv6ScheduleConfig {
            static const TVector<NNetmon::ETrafficClassType>& GetTrafficClasses() {
                static auto trafficClasses = GetTrafficClassesImpl();
                return trafficClasses;
            }

            static NClient::EFamily GetFamily() {
                return NClient::INET6;
            }

            static TMaybe<int> GetVlan() {
                return Nothing();
            }

            static TMaybe<NClient::TScheduledProbesResponse::TTarget> TargetToProto(const THost& host) {
                auto iface = TDerived::GetInterface(host);
                if (!iface) {
                    return Nothing();
                }
                NClient::TScheduledProbesResponse::TTarget protoTarget;
                protoTarget.SetHostname(iface->GetName());
                protoTarget.SetIp(iface->GetIpv6().ToString());
                return protoTarget;
            }

        private:
            static TVector<NNetmon::ETrafficClassType> GetTrafficClassesImpl() {
                auto trafficClasses = TDerived::GetAllowedTrafficClasses();
                auto requested = TDerived::GetRequestedTrafficClasses();
                EraseIf(trafficClasses, [&requested](NNetmon::ETrafficClassType tc) {
                    return !requested.contains(tc);
                });
                return trafficClasses;
            }
        };

        enum class EVlanPreference {
            DOM0_TO_DOM0, // use default bb/fb interfaces

            MTN_TO_MTN,   // use mtn vlan interface for targets in RTC,
                          // otherwise use default dom0 vlans;
                          // source iface is mtn vlan

            DOM0_TO_MTN,  // same, but use dom0 source interface
        };

        // TDerived needs to define GetRequestedTrafficClasses() and VlanPreference.
        template <class TDerived>
        struct TBaseBackboneScheduleConfig : public TBaseIpv6ScheduleConfig<TDerived> {
            static TMaybe<int> GetVlan() {
                if (TDerived::VlanPreference == EVlanPreference::MTN_TO_MTN) {
                    return EVlans::MTN_BACKBONE;
                } else {
                    return Nothing();
                }
            }

            static const THostInterface* GetInterface(const THost& host) {
                switch (TDerived::VlanPreference) {
                    case EVlanPreference::DOM0_TO_DOM0:
                        return host.GetBackboneInterface();
                    case EVlanPreference::MTN_TO_MTN: /* fallthrough */
                    case EVlanPreference::DOM0_TO_MTN:
                        return TSettings::Get()->GetVlanInversionSelector().Contains(host)
                               ? host.GetInterfaceByVlan(EVlans::MTN_BACKBONE)
                               : host.GetBackboneInterface();
                }
            }

            static TVector<NNetmon::ETrafficClassType> GetAllowedTrafficClasses() {
                return {CS0, CS3, CS4};
            }
        };

        // TDerived needs to define GetRequestedTrafficClasses() and VlanPreference.
        template <class TDerived>
        struct TBaseFastboneScheduleConfig : public TBaseIpv6ScheduleConfig<TDerived> {
            static TMaybe<int> GetVlan() {
                if (TDerived::VlanPreference == EVlanPreference::MTN_TO_MTN) {
                    return EVlans::MTN_FASTBONE;
                } else {
                    return Nothing();
                }
            }

            static const THostInterface* GetInterface(const THost& host) {
                switch (TDerived::VlanPreference) {
                    case EVlanPreference::DOM0_TO_DOM0:
                        return host.GetFastboneInterface();
                    case EVlanPreference::MTN_TO_MTN: /* fallthrough */
                    case EVlanPreference::DOM0_TO_MTN:
                        return TSettings::Get()->GetVlanInversionSelector().Contains(host)
                               ? host.GetInterfaceByVlan(EVlans::MTN_FASTBONE)
                               : host.GetFastboneInterface();
                }
            }

            static TVector<NNetmon::ETrafficClassType> GetAllowedTrafficClasses() {
                // disallow CS0, because agents distinguish bb/fb probes by traffic class
                return {CS1, CS2};
            }
        };

        template<EVlanPreference VlanPref>
        struct TIntraDcBackboneScheduleConfig : public TBaseBackboneScheduleConfig<TIntraDcBackboneScheduleConfig<VlanPref>> {
            static const auto VlanPreference = VlanPref;

            static THashSet<NNetmon::ETrafficClassType> GetRequestedTrafficClasses() {
                return TSettings::Get()->GetProbeScheduleTrafficClasses();
            }
        };

        template<EVlanPreference VlanPref>
        struct TIntraDcFastboneScheduleConfig : public TBaseFastboneScheduleConfig<TIntraDcFastboneScheduleConfig<VlanPref>> {
            static const auto VlanPreference = VlanPref;

            static THashSet<NNetmon::ETrafficClassType> GetRequestedTrafficClasses() {
                return TSettings::Get()->GetProbeScheduleTrafficClasses();
            }
        };

        struct TCrossDcBackboneScheduleConfig : public TBaseBackboneScheduleConfig<TCrossDcBackboneScheduleConfig> {
            static const auto VlanPreference = EVlanPreference::DOM0_TO_DOM0;

            static THashSet<NNetmon::ETrafficClassType> GetRequestedTrafficClasses() {
                return TSettings::Get()->GetProbeScheduleCrossDcTrafficClasses();
            }
        };

        struct TCrossDcFastboneScheduleConfig : public TBaseFastboneScheduleConfig<TCrossDcFastboneScheduleConfig> {
            static const auto VlanPreference = EVlanPreference::DOM0_TO_DOM0;

            static THashSet<NNetmon::ETrafficClassType> GetRequestedTrafficClasses() {
                return TSettings::Get()->GetProbeScheduleCrossDcTrafficClasses();
            }
        };

        struct TSparseBackboneScheduleConfig : public TBaseBackboneScheduleConfig<TSparseBackboneScheduleConfig> {
            static const auto VlanPreference = EVlanPreference::DOM0_TO_DOM0;

            static TVector<NNetmon::ETrafficClassType> GetAllowedTrafficClasses() {
                return {CS0};
            }

            static THashSet<NNetmon::ETrafficClassType> GetRequestedTrafficClasses() {
                return {CS0};
            }
        };

        struct TSparseFastboneScheduleConfig : public TBaseFastboneScheduleConfig<TSparseFastboneScheduleConfig> {
            static const auto VlanPreference = EVlanPreference::DOM0_TO_DOM0;

            static TVector<NNetmon::ETrafficClassType> GetAllowedTrafficClasses() {
                // we can't use cs0 probes here since agents interpret cs0 as backbone
                return {CS1};
            }

            static THashSet<NNetmon::ETrafficClassType> GetRequestedTrafficClasses() {
                return {CS1};
            }
        };

        template <class TConfig>
        TMaybe<NClient::TScheduledProbesResponse::TSubschedule> MakeSubschedule(const TVector<TTopology::THostRef>& targets) {
            if (targets.empty()) {
                return Nothing();
            }

            NClient::TScheduledProbesResponse::TSubschedule subschedule;
            subschedule.SetFamily(TConfig::GetFamily());
            subschedule.SetPacketCount(TSettings::Get()->GetScheduledProbePacketCount());

            for (auto trafficClass : TConfig::GetTrafficClasses()) {
                subschedule.AddTrafficClasses(TrafficClassToProto(trafficClass));
            }
            if (subschedule.GetTrafficClasses().empty()) {
                return Nothing();
            }

            if (auto vlan = TConfig::GetVlan()) {
                subschedule.SetVlan(*vlan);
            }

            auto& protoTargets(*subschedule.MutableTargets());
            protoTargets.Reserve(targets.size());
            for (const auto& target : targets) {
                if (auto protoTarget = TConfig::TargetToProto(*target)) {
                    protoTargets.Add()->Swap(&*protoTarget);
                }
            }

            if (protoTargets.empty()) {
                return Nothing();
            } else {
                return subschedule;
            }
        }
    }

    void TSendReportsReply::PreprocessRequest(THttpInput& input) {
        ui64 reportsCount = ExtractReportsCount(input.Headers());
        if (reportsCount) {
            TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::ApiNewClientReports, reportsCount);
            TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::ApiClientReportsCount, reportsCount);

            if (!GetServerContext().GetReportsRateLimiter().Check(reportsCount)) {
                TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::ApiDroppedClientReports, reportsCount);
                ythrow TTooManyRequestsError() << "too many reports from clients";
            }
        }
    }

    void TSendReportsReply::Process() {
        bool dumperReady(GetServerContext().GetDumper().IsReady());

        auto state(GetServerContext().GetTerminatedHostsUpdater().GetState());
        const auto& terminatedHosts(state->GetHosts());
        bool validProbeFound = false;
        auto linkPollerInfo = ELinkPollerInfo::NO_POLLER;
        TMaybe<TTopology::THostRef> sourceHost;

        const auto deadHosts(GetServerContext().GetWalleUpdater().GetHosts());
        const auto linkPollerHosts(GetServerContext().GetLinkPollerHostsUpdater().GetHosts());
        const auto linkPollerSwitches(GetServerContext().GetLinkPollerHostsUpdater().GetSwitches());

        ui64 now(TInstant::Now().MicroSeconds());

        // Parse reports for NOCSLA signal even if dumper subsystem isn't ready
        TDumperProbeBatch::TRecordVector dumperProbes;
        for (const auto& report : GetRequest().GetReports()) {
            const auto probe(ParseReport(
                report, terminatedHosts, *deadHosts,
                *linkPollerHosts, *linkPollerSwitches,
                now, linkPollerInfo
            ));
            if (!probe) {
                continue;
            }

            if (!validProbeFound) {
                sourceHost.ConstructInPlace(probe->GetSourceIface().GetHost());
                // reports from one agent should have the same source host, so remove it only once
                GetServerContext().GetTerminatedHostsUpdater().RemoveHost(*sourceHost);
                validProbeFound = true;
            }
            if (dumperReady) {
                dumperProbes.PushBack(TDumperProbeBatch::TRecord::Make(*probe, probe->GetGenerated()));
            }
        }

        if (sourceHost && linkPollerInfo != ELinkPollerInfo::NO_POLLER) {
            GetServerContext().GetLinkPollerHostsUpdater().AddHost(*sourceHost);
        }

        if (!dumperReady) {
            ythrow yexception() << "too many probes aren't dumped";
        }

        GetServerContext().GetLine().EnqueueAll(dumperProbes);
        GetResponse().SetAccepted(dumperProbes.size());
    }

    TMaybe<TDumperProbe> TSendReportsReply::ParseReport(const NClient::TProbeReport& report,
                                                        const TTopologyStorage::THostIdSet& terminatedHosts,
                                                        const TTopologyStorage::THostSet& deadHosts,
                                                        const TTopologyStorage::THostSet& linkPollerHosts,
                                                        const TLinkPollerHostsUpdater::TSwitchSet& linkPollerSwitches,
                                                        ui64 now,
                                                        ELinkPollerInfo& linkPollerInfo)
    {
        const auto& topology(GetServerContext().GetTopologyStorage());

        const auto sourceIface = topology.FindHostInterface(report.GetSource());
        const auto targetIface = topology.FindHostInterface(report.GetTarget());
        if (!sourceIface || !targetIface) {
            return Nothing();
        } else if (sourceIface->GetNetworkType() != targetIface->GetNetworkType()) {
            return Nothing();
        }

        const TTopology::THostRef sourceHost(sourceIface->GetHost());
        const TTopology::THostRef targetHost(targetIface->GetHost());
        if (!AreKnownQueues(sourceHost, targetHost)) {
            return Nothing();
        }

        const auto& sourceDc = sourceIface->GetDatacenter();
        const auto& targetDc = targetIface->GetDatacenter();

        EProtocolType protocol;
        switch (report.GetProtocol()) {
            case NClient::ICMP:
                protocol = EProtocolType::ICMP;
                break;
            case NClient::UDP:
                protocol = EProtocolType::UDP;
                break;
            case NClient::TCP:
                protocol = EProtocolType::TCP;
                break;
            case NClient::LINK_POLLER:
                protocol = EProtocolType::LINK_POLLER;
                break;
            default:
                protocol = EProtocolType::NIL_PROTOCOL;
                break;
        }

        ui16 family;
        ENetworkType network;
        switch (report.GetFamily()) {
            case NClient::INET4:
                // only backbone network can have v4 addresses
                network = ENetworkType::BACKBONE4;
                family = AF_INET;
                break;
            case NClient::INET6:
                network = sourceIface->GetNetworkType();
                if (i32 tos = report.GetTypeOfService()) {
                    // correct network type for link poller report with CS1 and CS2
                    if (tos < IPTOS_CLASS_CS3 && protocol == EProtocolType::LINK_POLLER) {
                        network = ENetworkType::FASTBONE6;
                    }
                    // IPTOS_CLASS_CS0 (no class) has zero value
                    if (network == ENetworkType::FASTBONE6 && tos == IPTOS_CLASS_CS1) {
                        network = ENetworkType::FASTBONE_CS1;
                    } else if (network == ENetworkType::FASTBONE6 && tos == IPTOS_CLASS_CS2) {
                        network = ENetworkType::FASTBONE_CS2;
                    } else if (network == ENetworkType::BACKBONE6 && tos == IPTOS_CLASS_CS3) {
                        network = ENetworkType::BACKBONE_CS3;
                    } else if (network == ENetworkType::BACKBONE6 && tos == IPTOS_CLASS_CS4) {
                        network = ENetworkType::BACKBONE_CS4;
                    } else {
                        network = ENetworkType::NIL_NETWORK;
                    }
                }
                family = AF_INET6;
                break;
            default:
                network = ENetworkType::NIL_NETWORK;
                family = 0;
                break;
        }

        if (network == ENetworkType::NIL_NETWORK || protocol == EProtocolType::NIL_PROTOCOL) {
            return Nothing();
        } else if (sourceIface->IsMtnVlan() && !TSettings::Get()->AreMtnVlanProbesEnabled()) {
            return Nothing();
        }

        const bool invertedVlans = sourceDc == targetDc &&        // TODO: invert cross-dc probes later
                                   sourceIface != targetIface &&  // link poller probes aren't inverted
                                   TSettings::Get()->GetVlanInversionSelector().Contains(*sourceHost);
        const bool sparseProbe = (
            (!invertedVlans && sourceIface->IsMtnVlan()) ||
            (invertedVlans && !sourceIface->IsMtnVlan())
        );
        if (sparseProbe) {
            if (sourceIface->GetNetworkType() == ENetworkType::BACKBONE6) {
                network = SPARSE_BACKBONE6;
            } else if (sourceIface->GetNetworkType() == ENetworkType::FASTBONE6) {
                network = SPARSE_FASTBONE6;
            }
        }

        const TIpAddress sourceAddress(family, report.GetSourceAddress().GetIp());
        const TIpAddress targetAddress(family, report.GetTargetAddress().GetIp());

        double rttMs = report.GetRoundTripTimeAverage() / (MICROSECONDS_IN_SECOND / MILLISECONDS_IN_SECOND);

        double score(-1.0);
        const double total(report.GetReceived() + report.GetLost());
        if (!report.GetFailed() && total > 0.0) {
            score = report.GetReceived() / total;
        } else if (!report.GetRoundTripTimeAverage()) {
            return Nothing();
        }

        if (!report.GetGenerated() || report.GetGenerated() > now) {
            // don't accept probes from future
            return Nothing();
        }

        if (terminatedHosts.contains(targetHost->GetReducedId()) ||
            deadHosts.contains(sourceHost) || deadHosts.contains(targetHost))
        {
            // mark probe from terminated and dead hosts with special score value
            score = -1.0;
        }

        auto scoreClass = GetScoreClass(score);

        bool isNocSlaProbe(report.GetType() == NClient::EProbeType::NOC_SLA_PROBE); // don't update counters on production
        bool updateLinkPollerCounters(isNocSlaProbe && (protocol == EProtocolType::ICMP || protocol == EProtocolType::LINK_POLLER));
        bool updateSlaCounters(false);
        bool updateCrossDcCounters(false);
        if (isNocSlaProbe && protocol == EProtocolType::UDP) {
            updateSlaCounters     = sourceDc == targetDc;
            updateCrossDcCounters = sourceDc != targetDc;
        }

        bool switchMetricsEnabled = !!TSettings::Get()->GetNocSlaSwitchMapCapacity();
        auto& switchSlaCounters = GetServerContext().GetSwitchSlaCounters();
        auto& switchSlaLPCounters = GetServerContext().GetSwitchSlaLPCounters();

        // FIXME: NETMON-338, just ignore local pinger probes by same src/dst host feature
        if (sourceIface->GetReducedId() != targetIface->GetReducedId()) {
            if (scoreClass == EProbeScore::SUCCESS) {
                TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::ProbeSuccessCount, 1);
            } else if (scoreClass == EProbeScore::FAILED) {
                TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::ProbeFailedCount, 1);
            } else if (scoreClass == EProbeScore::SEMIFAILED) {
                TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::ProbeSemiFailedCount, 1);
            }

            if (updateSlaCounters && scoreClass != EProbeScore::INVALID) {
                const auto& dcs = TSettings::Get()->GetProbeScheduleDcs();
                for (size_t i = 0; i < dcs.size(); ++i) {
                    const auto& dc = dcs[i];

                    if (dc == sourceDc.GetName()) {
                        IncrementCounter(PacketCounters[i], network,
                                         report.GetReceived(), report.GetLost(),
                                         report.GetTosChanged());
                        IncrementCounter(RttCounters[i], network, rttMs);
                        break;
                    }
                }

                // NETMON-564: don't update per-switch signals if there were link poller drops
                if (switchMetricsEnabled &&
                    linkPollerInfo != ELinkPollerInfo::HAS_DROPS &&
                    !sparseProbe)
                {
                    // For switches that contain some hosts with link poller enabled
                    // and some hosts with LP disabled, we create two signals:
                    //  - signal with label 'link_poller=true' selects only LP hosts
                    //  - signal with label 'link_poller=false' selects all hosts in switch
                    // For switches where all hosts have LP enabled, we only
                    // create one signal with label 'link_poller=true'.
                    if (!linkPollerSwitches.contains(sourceIface->GetSwitch())) {
                        switchSlaCounters.RegisterInterSwitchPackets(sourceIface->GetSwitch(),
                                                                     false, network,
                                                                     report.GetReceived(), report.GetLost(),
                                                                     report.GetTosChanged(), rttMs);
                    }
                    if (!linkPollerSwitches.contains(targetIface->GetSwitch())) {
                        switchSlaCounters.RegisterInterSwitchPackets(targetIface->GetSwitch(),
                                                                     true,  network,
                                                                     report.GetReceived(), report.GetLost(),
                                                                     report.GetTosChanged(), rttMs);
                    }
                    if (linkPollerHosts.contains(sourceIface->GetHost())) {
                        switchSlaLPCounters.RegisterInterSwitchPackets(sourceIface->GetSwitch(),
                                                                       false, network,
                                                                       report.GetReceived(), report.GetLost(),
                                                                       report.GetTosChanged(), rttMs);
                    }
                    if (linkPollerHosts.contains(targetIface->GetHost())) {
                        switchSlaLPCounters.RegisterInterSwitchPackets(targetIface->GetSwitch(),
                                                                       true, network,
                                                                       report.GetReceived(), report.GetLost(),
                                                                       report.GetTosChanged(), rttMs);
                    }
                }
            }
            if (updateCrossDcCounters && scoreClass != EProbeScore::INVALID) {
                auto& crossDcCounters = GetServerContext().GetCrossDcCounters();
                crossDcCounters.RegisterPackets(sourceDc, targetDc,
                                                network, report.GetReceived(), report.GetLost(),
                                                report.GetTosChanged(), rttMs);
            }
        } else { // sourceIface->GetReducedId() == targetIface->GetReducedId()
            if (updateLinkPollerCounters && switchMetricsEnabled && scoreClass != EProbeScore::INVALID) {
                switchSlaLPCounters.RegisterLinkPollerPackets(sourceIface->GetSwitch(),
                                                              network,
                                                              report.GetReceived(),
                                                              report.GetLost());

                auto& probeScheduleMaintainer = GetServerContext().GetProbeScheduleMaintainer();
                if (network == BACKBONE_CS3 || network == BACKBONE6) {
                    probeScheduleMaintainer.AddLinkPollerStats(TTopology::THostRef(&sourceIface->GetHost()),
                                                               report.GetReceived(), report.GetLost());
                    linkPollerInfo = report.GetLost() ? ELinkPollerInfo::HAS_DROPS : ELinkPollerInfo::NO_DROPS;
                }
            }
        }

        TUnistat::Instance().PushSignalUnsafe(
            ENetmonSignals::ProbeIncomingDelay, (now - report.GetGenerated()) / MICROSECONDS_IN_SECOND);

        return MakeMaybe<TDumperProbe>(
            sourceIface,
            targetIface,
            sourceAddress,
            targetAddress,
            report.GetSourceAddress().GetPort(),
            report.GetTargetAddress().GetPort(),
            network,
            protocol,
            score,
            report.GetRoundTripTimeAverage() / MICROSECONDS_IN_SECOND,
            report.GetGenerated() / MICROSECONDS_IN_SECOND
        );
    }

    void TExpandGroupReply::Process() {
        const auto groupName(GetRequest().GetExpression());
        if (groupName.empty()) {
            ythrow yexception() << "no group given";
        }
        if (!ValidateGroup(groupName)) {
            ythrow yexception() << "invalid group given";
        }

        const auto groupKey(TGroupKey::Skynet(groupName));
        const auto state(GetServerContext().GetGroupStorage().FetchGroup(groupKey));

        auto& group(*GetResponse().MutableGroup());
        group.SetReady(state.Ready);

        if (state.Hosts) {
            group.MutableHosts()->Reserve(state.Hosts->size());
            for (const auto& host : *state.Hosts) {
                group.AddHosts(host);
            }
        }
    }

    void TEnqueuedTasksReply::Process() {
        const auto& hostName(GetRequest().GetHost());
        if (!hostName) {
            ythrow yexception() << "no host given";
        }

        auto foundTasks(GetServerContext().GetEnqueuedTaskIndex().FindByHost(hostName));
        auto& responseTasks(*GetResponse().MutableTasks());
        responseTasks.Reserve(foundTasks.size());

        for (const auto& task : foundTasks) {
            responseTasks.Add()->CopyFrom(task);
        }
    }

    NThreading::TFuture<void> TFinishTasksReply::Process() {
        auto& enqueuedTasks(GetServerContext().GetEnqueuedTaskStorage());
        auto& finishedTasks(GetServerContext().GetFinishedTaskStorage());

        TVector<NThreading::TFuture<void>> futures;
        GetResponse().MutableSuccess()->Resize(GetRequest().TasksSize(), false);
        for (const auto idx : xrange(GetRequest().TasksSize())) {
            TFinishedTask task(GetRequest().GetTasks(idx));
            IAgentTask::TKey parentKey(task.ParentKey());

            if (parentKey) {
                // if parent key is specified there are two cases: remove enqueued task or replace finished task
                THolder<TAgentTaskMover> mover;
                TMaybe<IAgentTask::TKey> previousKey(GetServerContext().GetFinishedTaskIndex().KeyByParentKey(parentKey));
                if (previousKey.Defined()) {
                    mover = MakeHolder<TAgentTaskMover>(
                        finishedTasks,
                        finishedTasks,
                        previousKey.GetRef(),
                        task
                    );
                } else {
                    mover = MakeHolder<TAgentTaskMover>(
                        enqueuedTasks,
                        finishedTasks,
                        parentKey,
                        task
                    );
                }

                futures.emplace_back(mover->Execute().Apply(
                    [this, idx] (const NThreading::TFuture<bool>& future) {
                        GetResponse().MutableSuccess()->Set(idx, future.GetValue());
                    }
                ));
            } else {
                futures.emplace_back(finishedTasks.Put(task).Apply(
                    [this, idx] (const NThreading::TFuture<TAgentTaskStorage::EResultCode>& future) {
                        GetResponse().MutableSuccess()->Set(idx, future.GetValue() == TAgentTaskStorage::OK);
                    }
                ));
            }
        }

        return NThreading::WaitExceptionOrAll(futures);
    }

    void TProvisioningReply::Process() {
        if (!GetRequest().GetHost()) {
            ythrow yexception() << "no host given";
        }

        auto info(GetServerContext().GetTopologyUpdater().GetInfo());
        info->ToProto(*GetResponse().MutableTopologyInfo());
    }

    void TScheduledProbesReply::Process() {
        if (GetRequest().GetType() != NClient::EProbeType::NOC_SLA_PROBE) {
            ythrow yexception() << "invalid probe type";
        }

        const auto& hostName(GetRequest().GetHost());
        if (!hostName) {
            ythrow yexception() << "no host given";
        }

        const auto host(GetServerContext().GetTopologyStorage().FindHost(hostName));
        if (!host) {
            ythrow yexception() << "host not found: " << hostName;
        }

        if (!GetRequest().GetNotConsiderHostInNextSchedule()) {
            // this host is interested in schedule
            GetServerContext().GetTerminatedHostsUpdater().RemoveHost(host);
            GetServerContext().GetProbeScheduleMaintainer().AddInterestedHost(host);
        }

        const auto schedule(GetServerContext().GetProbeScheduleMaintainer().GetSchedule());
        if (schedule->empty()) {
            return;
        }

        GetResponse().SetType(NClient::EProbeType::NOC_SLA_PROBE);
        GetResponse().SetTtl(TSettings::Get()->GetProbeScheduleTtl().Seconds());
        for (const auto protocol : TSettings::Get()->GetProbeScheduleProtocols()) {
            GetResponse().AddProtocols(ProtocolToProto(protocol));
        }

        const auto it(schedule->find(host));
        if (it.IsEnd()) {
            return;
        }

        const bool invertVlans = TSettings::Get()->GetVlanInversionSelector().Contains(*host);

        // intra-dc
        if (invertVlans) {
            if (auto bbIntraDc = MakeSubschedule<TIntraDcBackboneScheduleConfig<EVlanPreference::MTN_TO_MTN>>(it->second.IntraDc)) {
                GetResponse().AddSubschedules()->Swap(bbIntraDc.Get());
            }
            if (auto fbIntraDc = MakeSubschedule<TIntraDcFastboneScheduleConfig<EVlanPreference::MTN_TO_MTN>>(it->second.IntraDc)) {
                GetResponse().AddSubschedules()->Swap(fbIntraDc.Get());
            }
        } else if (!TSettings::Get()->GetVlanInversionSelector().Empty()) {
            if (auto bbIntraDc = MakeSubschedule<TIntraDcBackboneScheduleConfig<EVlanPreference::DOM0_TO_MTN>>(it->second.IntraDc)) {
                GetResponse().AddSubschedules()->Swap(bbIntraDc.Get());
            }
            if (auto fbIntraDc = MakeSubschedule<TIntraDcFastboneScheduleConfig<EVlanPreference::DOM0_TO_MTN>>(it->second.IntraDc)) {
                GetResponse().AddSubschedules()->Swap(fbIntraDc.Get());
            }
        } else {
            if (auto bbIntraDc = MakeSubschedule<TIntraDcBackboneScheduleConfig<EVlanPreference::DOM0_TO_DOM0>>(it->second.IntraDc)) {
                GetResponse().AddSubschedules()->Swap(bbIntraDc.Get());
            }
            if (auto fbIntraDc = MakeSubschedule<TIntraDcFastboneScheduleConfig<EVlanPreference::DOM0_TO_DOM0>>(it->second.IntraDc)) {
                GetResponse().AddSubschedules()->Swap(fbIntraDc.Get());
            }
        }

        if (auto ipv4IntraDc = MakeSubschedule<TIpv4ScheduleConfig>(it->second.Ipv4IntraDc)) {
            GetResponse().AddSubschedules()->Swap(ipv4IntraDc.Get());
        }

        // cross-dc
        if (auto bbCrossDc = MakeSubschedule<TCrossDcBackboneScheduleConfig>(it->second.CrossDc)) {
            GetResponse().AddSubschedules()->Swap(bbCrossDc.Get());
        }
        if (auto fbCrossDc = MakeSubschedule<TCrossDcFastboneScheduleConfig>(it->second.CrossDc)) {
            GetResponse().AddSubschedules()->Swap(fbCrossDc.Get());
        }

        // intra-dc sparse
        if (invertVlans) {
            if (auto bbSparse = MakeSubschedule<TSparseBackboneScheduleConfig>(it->second.IntraDcSparse)) {
                GetResponse().AddSubschedules()->Swap(bbSparse.Get());
            }
            if (auto fbSparse = MakeSubschedule<TSparseFastboneScheduleConfig>(it->second.IntraDcSparse)) {
                GetResponse().AddSubschedules()->Swap(fbSparse.Get());
            }
        }
    }

    void TTerminatedHostReply::Process() {
        const auto& hostName(GetRequest().GetHost());
        if (!hostName) {
            ythrow yexception() << "no host given";
        }

        const auto host(GetServerContext().GetTopologyStorage().FindHost(hostName));
        if (!host) {
            ythrow yexception() << "host not found: " << hostName;
        }

        GetServerContext().GetTerminatedHostsUpdater().AddHost(host);
    }
}
