#pragma once

#include <infra/netmon/topology/clients/groups.h>
#include <infra/netmon/topology/topology_updater.h>
#include <infra/netmon/probe_dumper.h>
#include <infra/netmon/probe_schedule_maintainer.h>
#include <infra/netmon/terminated_hosts.h>
#include <infra/netmon/tasks/enqueued_task_index.h>
#include <infra/netmon/tasks/finished_task_index.h>
#include <infra/netmon/library/api_handler_helpers.h>
#include <infra/netmon/library/token_bucket.h>
#include <infra/netmon/link_poller_hosts.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/switch_metrics.h>

#include <infra/netmon/agent/idl/api.pb.h>

namespace NNetmon {
    class TReportsRateLimiter : public TNonCopyable {
    public:
        TReportsRateLimiter()
            : TokenBucket(TSettings::Get()->GetReportsRateLimiterCapacity(),
                          TSettings::Get()->GetReportsRateLimiterRefillRatePerMs())
        {
        }

        inline bool Check(size_t reportsCount) {
            return TokenBucket.TryConsume(reportsCount);
        }

    private:
        TTokenBucket TokenBucket;
    };

    class TClientApiContext : public TNonCopyable {
    public:
        TClientApiContext(TDumperProbeBatch::TRecordQueue& queue_,
                          const TSlicerDumper& dumper,
                          const TGroupStorage& groupStorage,
                          const TTopologyUpdater& topologyUpdater,
                          const TTopologyStorage& topologyStorage,
                          TProbeScheduleMaintainer& probeScheduleMaintainer,
                          TLinkPollerHostsUpdater& linkPollerHostsUpdater,
                          TTerminatedHostsUpdater& terminatedHostsUpdater,
                          const TWalleUpdater& walleUpdater,
                          TCrossDcCounters& crossDcCounters,
                          TSwitchSlaCounters& switchSlaCounters,
                          TSwitchSlaCounters& switchSlaLPCounters,
                          TAgentTaskStorage& enqueuedTaskStorage,
                          const TEnqueuedTaskIndex& enqueuedTaskIndex,
                          TAgentTaskStorage& finishedTaskStorage,
                          const TFinishedTaskIndex& finishedTaskIndex)
            : Queue(queue_)
            , Dumper(dumper)
            , GroupStorage(groupStorage)
            , TopologyUpdater(topologyUpdater)
            , TopologyStorage(topologyStorage)
            , WalleUpdater(walleUpdater)
            , ProbeScheduleMaintainer(probeScheduleMaintainer)
            , LinkPollerHostsUpdater(linkPollerHostsUpdater)
            , TerminatedHostsUpdater(terminatedHostsUpdater)
            , CrossDcCounters(crossDcCounters)
            , SwitchSlaCounters(switchSlaCounters)
            , SwitchSlaLPCounters(switchSlaLPCounters)
            , EnqueuedTaskStorage(enqueuedTaskStorage)
            , EnqueuedTaskIndex(enqueuedTaskIndex)
            , FinishedTaskStorage(finishedTaskStorage)
            , FinishedTaskIndex(finishedTaskIndex)
        {
        }

        inline TDumperProbeBatch::TRecordQueue& GetLine() const noexcept {
            return Queue;
        }
        inline const TSlicerDumper& GetDumper() const noexcept {
            return Dumper;
        }
        inline const TGroupStorage& GetGroupStorage() const noexcept {
            return GroupStorage;
        }
        inline const TTopologyUpdater& GetTopologyUpdater() const noexcept {
            return TopologyUpdater;
        }
        inline const TTopologyStorage& GetTopologyStorage() const noexcept {
            return TopologyStorage;
        }
        inline const TWalleUpdater& GetWalleUpdater() const noexcept {
            return WalleUpdater;
        }
        inline TProbeScheduleMaintainer& GetProbeScheduleMaintainer() noexcept {
            return ProbeScheduleMaintainer;
        }
        inline TLinkPollerHostsUpdater& GetLinkPollerHostsUpdater() noexcept {
            return LinkPollerHostsUpdater;
        }
        inline TTerminatedHostsUpdater& GetTerminatedHostsUpdater() noexcept {
            return TerminatedHostsUpdater;
        }
        inline TCrossDcCounters& GetCrossDcCounters() noexcept {
            return CrossDcCounters;
        }
        inline TSwitchSlaCounters& GetSwitchSlaCounters() noexcept {
            return SwitchSlaCounters;
        }
        inline TSwitchSlaCounters& GetSwitchSlaLPCounters() noexcept {
            return SwitchSlaLPCounters;
        }

        inline TAgentTaskStorage& GetEnqueuedTaskStorage() const noexcept {
            return EnqueuedTaskStorage;
        }
        inline const TEnqueuedTaskIndex& GetEnqueuedTaskIndex() const noexcept {
            return EnqueuedTaskIndex;
        }
        inline TAgentTaskStorage& GetFinishedTaskStorage() const noexcept {
            return FinishedTaskStorage;
        }
        inline const TFinishedTaskIndex& GetFinishedTaskIndex() const noexcept {
            return FinishedTaskIndex;
        }

        inline TReportsRateLimiter& GetReportsRateLimiter() noexcept {
            return ReportsRateLimiter;
        }

    private:
        TDumperProbeBatch::TRecordQueue& Queue;
        const TSlicerDumper& Dumper;

        const TGroupStorage& GroupStorage;
        const TTopologyUpdater& TopologyUpdater;
        const TTopologyStorage& TopologyStorage;

        const TWalleUpdater& WalleUpdater;
        TProbeScheduleMaintainer& ProbeScheduleMaintainer;
        TLinkPollerHostsUpdater& LinkPollerHostsUpdater;
        TTerminatedHostsUpdater& TerminatedHostsUpdater;

        TCrossDcCounters& CrossDcCounters;
        TSwitchSlaCounters& SwitchSlaCounters;
        TSwitchSlaCounters& SwitchSlaLPCounters;

        TAgentTaskStorage& EnqueuedTaskStorage;
        const TEnqueuedTaskIndex& EnqueuedTaskIndex;
        TAgentTaskStorage& FinishedTaskStorage;
        const TFinishedTaskIndex& FinishedTaskIndex;

        TReportsRateLimiter ReportsRateLimiter;
    };

    template <class TDispatcher, class TRequest, class TResponse>
    using TBaseClientApiReply = TProtoHttpReply<TDispatcher, TClientApiContext, TRequest, TResponse>;

    class TSendReportsReply: public TBaseClientApiReply<
            THttpDispatcher, NClient::TSendReportsRequest, NClient::TSendReportsResponse> {
    public:
        using TProtoHttpReply::TProtoHttpReply;

        void PreprocessRequest(THttpInput& input) override;

        void Process();

    private:
        enum class ELinkPollerInfo {
            NO_POLLER, // no link poller report has been parsed yet
            NO_DROPS,  // the last parsed link poller report is successful
            HAS_DROPS  // the last parsed link poller report has drops
        };

        TMaybe<TDumperProbe> ParseReport(const NClient::TProbeReport& report,
                                         const TTopologyStorage::THostIdSet& terminatedHosts,
                                         const TTopologyStorage::THostSet& deadHosts,
                                         const TTopologyStorage::THostSet& linkPollerHosts,
                                         const TLinkPollerHostsUpdater::TSwitchSet& linkPollerSwitches,
                                         ui64 now,
                                         ELinkPollerInfo& linkPollerInfo);
    };

    class TExpandGroupReply: public TBaseClientApiReply<
            THttpDispatcher, NClient::TExpandGroupRequest, NClient::TExpandGroupResponse> {
    public:
        using TProtoHttpReply::TProtoHttpReply;

        void Process();
    };

    class TEnqueuedTasksReply: public TBaseClientApiReply<
            THttpDispatcher, NClient::TEnqueuedTasksRequest, NClient::TEnqueuedTasksResponse> {
    public:
        using TProtoHttpReply::TProtoHttpReply;

        void Process();
    };

    class TFinishTasksReply: public TBaseClientApiReply<
            THttpFutureDispatcher, NClient::TFinishTasksRequest, NClient::TFinishTasksResponse> {
    public:
        using TProtoHttpReply::TProtoHttpReply;

        NThreading::TFuture<void> Process();
    };

    class TProvisioningReply: public TBaseClientApiReply<
            THttpDispatcher, NClient::TProvisioningRequest, NClient::TProvisioningResponse> {
    public:
        using TProtoHttpReply::TProtoHttpReply;

        void Process();
    };

    class TScheduledProbesReply: public TBaseClientApiReply<
            THttpDispatcher, NClient::TScheduledProbesRequest, NClient::TScheduledProbesResponse> {
    public:
        using TProtoHttpReply::TProtoHttpReply;

        void Process();
    };

    class TTerminatedHostReply: public TBaseClientApiReply<
            THttpDispatcher, NClient::TTerminatedHostRequest, NClient::TTerminatedHostResponse> {
    public:
        using TProtoHttpReply::TProtoHttpReply;

        void Process();
    };

    using TSendReportsHandler = THttpHandler<TSendReportsReply>;
    using TExpandGroupHandler = THttpHandler<TExpandGroupReply>;
    using TEnqueuedTasksHandler = THttpHandler<TEnqueuedTasksReply>;
    using TFinishTasksHandler = THttpHandler<TFinishTasksReply>;
    using TProvisioningHandler = THttpHandler<TProvisioningReply>;
    using TScheduledProbesHandler = THttpHandler<TScheduledProbesReply>;
    using TTerminatedHostHandler = THttpHandler<TTerminatedHostReply>;
}
