#include <infra/netmon/api/expression_api.h>
#include <infra/netmon/settings.h>

namespace NNetmon {
    namespace {
        void ExpressionToJson(const TExpressionStorage::TResult& expression, NJsonWriter::TBuf& response,
                              const TUserState& userState, const TExpressionContext& context)
        {
            const auto& aggregator(context.GetAggregatorMaintainer());

            const auto& sudoers(context.GetSudoers());
            const bool isSudoer = sudoers.contains(userState.Login);

            auto owners(expression.Owners);
            if (isSudoer) {
                for (const auto& sudoer: context.GetSudoers()) {
                    owners.emplace(sudoer);
                }
            }

            response
                .BeginObject()
                .WriteKey(TStringBuf("id"))
                .WriteString(expression.Id)
                .WriteKey(TStringBuf("expression"))
                .WriteString(expression.Expression)
                .WriteKey(TStringBuf("temporary"))
                .WriteBool(false)
                .WriteKey(TStringBuf("aggregated"))
                .WriteBool(aggregator.ExpressionExists(expression.ExpressionId))
                .WriteKey(TStringBuf("owners"))
                .BeginList();

            for (const auto& owner : owners) {
                response.WriteString(owner);
            }

            response
                .EndList()
                .WriteKey(TStringBuf("views"))
                .BeginList();

            for (const auto& view : aggregator.FindAggregators(expression.ExpressionId)) {
                const auto index(view->GetDatacenterIndex());
                if (index) {
                    response
                        .BeginObject()
                        .WriteKey(TStringBuf("network"))
                        .WriteString(NetworkToString(view->GetKey().GetNetwork()))
                        .WriteKey(TStringBuf("protocol"))
                        .WriteString(ProtocolToString(view->GetKey().GetProtocol()))
                        .WriteKey(TStringBuf("updated"))
                        .WriteULongLong(index->GetGenerated().Seconds())
                        .EndObject();
                }
            }

            response
                .EndList()
                .EndObject();
        }

        bool HasPermissions(const TString& id, const TUserState& userState,
                            const TExpressionContext& context,
                            bool modifyLimits=false)
        {
            const bool isSudoer = context.GetSudoers().contains(userState.Login);
            if (!modifyLimits && TSettings::Get()->AreExpressionsFreezed() && !isSudoer) {
                ythrow TUnauthorizedError() <<
                    "expressions are freezed, "
                    "please contact st:NETMONSUPPORT or "
                    "Netmon team for further assistance";
            }

            auto expression(context.GetExpressionStorage().Get(id));
            if (expression.Defined()) {
                if (expression->Owners.empty()) {
                    return true;
                } else if (userState.IsActive) {
                    return (
                        isSudoer
                        || std::find(expression->Owners.begin(), expression->Owners.end(), userState.Login) != expression->Owners.end()
                    );
                } else {
                    return false;
                }
            } else {
                return true;
            }
        }

        const TString& GetRequestExpressionId(const NJson::TJsonValue& request) {
            if (!request.Has("id") || !request["id"].IsString()) {
                ythrow TValidationError() << "no id given";
            }
            return request["id"].GetStringSafe();
        }
    }

    NThreading::TFuture<void> TExpressionHostsReply::Process() {
        if (!GetServerContext().GetExpressionStorage().IsReady()) {
            ythrow yexception() << "expressions aren't ready yet";
        }

        const auto& request(GetRequest());
        const auto& id(GetRequestExpressionId(request));

        const auto expression(GetServerContext().GetExpressionStorage().Get(id));
        if (!expression.Defined()) {
            ythrow TNotFoundError() << "no expression found";
        }

        GetResponse()
            .BeginObject()
            .WriteKey(TStringBuf("hosts"))
            .BeginList();

        const auto selector(GetServerContext().GetTopologyStorage().GetTopologySelector());
        for (const auto& host : selector->GetHosts(expression->ExpressionId)) {
            GetResponse().WriteString(host->GetName());
        }

        GetResponse()
            .EndList()
            .EndObject();

        return NThreading::MakeFuture();
    }

    NThreading::TFuture<void> TExpressionExpandReply::Process() {
        if (!GetServerContext().GetExpressionStorage().IsReady()) {
            ythrow yexception() << "expressions aren't ready yet";
        }
        if (!GetRequest().Has("expression")) {
            ythrow TValidationError() << "no expression given";
        }

        const auto expression(ParseExpression(GetRequest()["expression"].GetStringSafe()));
        const auto selector(GetServerContext().GetTopologyStorage().GetTopologySelector());

        bool partial(false);
        bool failed(false);
        const auto& groupStorage(GetServerContext().GetGroupStorage());
        for (const auto& group : ExtractGroups(expression)) {
            const auto state(groupStorage.FetchGroup(group));
            partial = partial || !state.Ready || selector->Created() < state.Created;
            failed = failed || !state.Error.empty();
        }

        GetResponse()
            .BeginObject()
            .WriteKey(TStringBuf("hosts"))
            .BeginList();

        for (const auto& host : selector->GetHosts(expression)) {
            GetResponse().WriteString(host->GetName());
        }

        GetResponse()
            .EndList()
            .WriteKey(TStringBuf("partial"))
            .WriteBool(partial)
            .WriteKey(TStringBuf("failed"))
            .WriteBool(failed)
            .EndObject();

        return NThreading::MakeFuture();
    }

    NThreading::TFuture<void> TExpressionListReply::Process() {
        if (!GetServerContext().GetSliceCollector().IsReady()) {
            ythrow yexception() << "slices aren't collected yet";
        }
        if (!GetServerContext().GetExpressionStorage().IsReady()) {
            ythrow yexception() << "expressions aren't ready yet";
        }

        GetResponse()
            .BeginObject()
            .WriteKey(TStringBuf("expressions"))
            .BeginList();

        for (const auto& expression : GetServerContext().GetExpressionStorage().List()) {
            ExpressionToJson(expression, GetResponse(), GetUserState(), GetServerContext());
        }

        GetResponse()
            .EndList()
            .EndObject();

        return NThreading::MakeFuture();
    }

    NThreading::TFuture<void> TExpressionGetReply::Process() {
        if (!GetServerContext().GetSliceCollector().IsReady()) {
            ythrow yexception() << "slices aren't collected yet";
        }
        if (!GetServerContext().GetExpressionStorage().IsReady()) {
            ythrow yexception() << "expressions aren't ready yet";
        }

        const auto& request(GetRequest());
        const auto& id(GetRequestExpressionId(request));

        const auto expression(GetServerContext().GetExpressionStorage().Get(id));
        if (!expression.Defined()) {
            ythrow TNotFoundError() << "no expression found";
        }

        GetResponse()
            .BeginObject()
            .WriteKey(TStringBuf("expression"));
        ExpressionToJson(*expression, GetResponse(), GetUserState(), GetServerContext());
        GetResponse()
            .EndObject();

        return NThreading::MakeFuture();
    }

    NThreading::TFuture<void> TExpressionUpsertReply::Process() {
        if (!GetServerContext().GetExpressionStorage().IsReady()) {
            ythrow yexception() << "expressions aren't ready yet";
        }

        const auto& sudoers(GetServerContext().GetSudoers());
        const bool isSudoer(sudoers.contains(GetUserState().Login));

        const auto& request(GetRequest());
        TExpressionStorage::TChangeSet changeSet;
        if (request.Has("new_id")) {
            changeSet.Id = request["new_id"].GetStringSafe();
            if (!ValidateId(changeSet.Id.GetRef())) {
                ythrow yexception() << "invalid new_id given";
            }
        }
        if (request.Has("expression")) {
            changeSet.Expression = request["expression"].GetStringSafe();
        }
        if (request.Has("owners")) {
            changeSet.Owners.ConstructInPlace();
            bool ownerFound = false;
            for (const auto& owner : request["owners"].GetArraySafe()) {
                const auto& login(owner.GetStringSafe());
                if (!sudoers.contains(login)) {
                    changeSet.Owners->emplace(login);
                    ownerFound = ownerFound || login == GetUserState().Login;
                }
            }
            if (!ownerFound && GetUserState().IsActive && !isSudoer) {
                changeSet.Owners->emplace(GetUserState().Login);
            }
        } else if (GetUserState().IsActive && !isSudoer) {
            changeSet.Owners.ConstructInPlace();
            changeSet.Owners->emplace(GetUserState().Login);
        }

        const auto id(request.Has("id") ? request["id"].GetStringSafe() : "");
        if (!ValidateId(id)) {
            ythrow yexception() << "invalid id given";
        }
        if (!id.empty() && !HasPermissions(id, GetUserState(), GetServerContext())) {
            ythrow TUnauthorizedError() << "no permission to modify expression";
        }

        return GetServerContext().GetExpressionStorage().Upsert(id, changeSet).Apply(
            [this, id](const NThreading::TFuture<TExpressionStorage::TResult>& future) {
                try {
                    const auto expression(future.GetValue());
                    GetResponse()
                        .BeginObject()
                        .WriteKey(TStringBuf("expression"));
                    ExpressionToJson(expression, GetResponse(), GetUserState(), GetServerContext());
                    GetResponse()
                        .EndObject();

                    // MoveExpressionData is a no-op if id == expression.Id
                    return GetServerContext().GetUserDataStorage().MoveExpressionData(id, expression.Id);
                } catch (...) {
                    return NThreading::MakeErrorFuture<void>(std::current_exception());
                }
            }
        );
    }

    NThreading::TFuture<void> TExpressionDeleteReply::Process() {
        if (!GetServerContext().GetExpressionStorage().IsReady()) {
            ythrow yexception() << "expressions aren't ready yet";
        }

        const auto& request(GetRequest());
        const auto& id(GetRequestExpressionId(request));

        if (!HasPermissions(id, GetUserState(), GetServerContext())) {
            ythrow TUnauthorizedError() << "no permission to delete expression";
        }

        GetResponse()
            .BeginObject()
            .EndObject();

        return GetServerContext().GetExpressionStorage().Delete(id).Apply(
            [this, id](const NThreading::TFuture<void>& future) {
                if (future.HasException()) {
                    return future;
                } else {
                    return GetServerContext().GetUserDataStorage().DeleteExpressionData(id);
                }
            }
        );
    }

    TVector<TString> TMetadataRequestOps::ExtractFieldList(const NJson::TJsonValue& request) {
        if (!request.Has("fields")) {
            ythrow TValidationError() << "no fields given";
        }
        try {
            const auto& fields = request["fields"].GetArraySafe();
            TVector<TString> result(fields.size());
            Transform(fields.cbegin(), fields.cend(), result.begin(), [](const NJson::TJsonValue& value) {
                return value.GetStringSafe();
            });
            return result;
        } catch (const NJson::TJsonException&) {
            ythrow TValidationError() << CurrentExceptionMessage();
        }
    }

    template<class TRequestOps>
    NThreading::TFuture<void> TExpressionMetadataBaseReply<TRequestOps>::Process() {
        const auto& request(GetRequest());
        const auto& id(GetRequestExpressionId(request));
        auto fields = TRequestOps::ExtractFieldList(request);

        auto data(GetServerContext().GetUserDataStorage().GetExpressionData(id, fields));
        data["id"] = id;
        GetResponse().WriteJsonValue(&data);

        return NThreading::MakeFuture();
    }

    template<class TRequestOps>
    NThreading::TFuture<void> TExpressionMetadataUpsertBaseReply<TRequestOps>::Process() {
        const auto& request(GetRequest());
        const auto& id(GetRequestExpressionId(request));

        auto data = request;
        data.EraseValue("id");
        data.EraseValue("_id"); // prevent accidental modification of primary key in db

        if (!HasPermissions(id, GetUserState(), GetServerContext(), true)) {
            ythrow TUnauthorizedError() << "no permission to modify expression metadata";
        }

        GetResponse()
            .BeginObject()
            .EndObject();

        return GetServerContext().GetUserDataStorage().UpsertExpressionData(id, data);
    }

    template<class TRequestOps>
    NThreading::TFuture<void> TExpressionMetadataDeleteBaseReply<TRequestOps>::Process() {
        const auto& request(GetRequest());
        const auto& id(GetRequestExpressionId(request));
        auto fields = TRequestOps::ExtractFieldList(request);

        if (!HasPermissions(id, GetUserState(), GetServerContext(), true)) {
            ythrow TUnauthorizedError() << "no permission to delete expression metadata";
        }

        GetResponse()
            .BeginObject()
            .EndObject();

        return GetServerContext().GetUserDataStorage().DeleteExpressionDataFields(id, fields);
    }

    template class TExpressionMetadataBaseReply<TMetadataRequestOps>;
    template class TExpressionMetadataUpsertBaseReply<TMetadataRequestOps>;
    template class TExpressionMetadataDeleteBaseReply<TMetadataRequestOps>;

    template class TExpressionMetadataBaseReply<TLimitsRequestOps>;
    template class TExpressionMetadataUpsertBaseReply<TLimitsRequestOps>;
    template class TExpressionMetadataDeleteBaseReply<TLimitsRequestOps>;
}
