#pragma once

#include <infra/netmon/library/api_handler_helpers.h>
#include <infra/netmon/probe_aggregator_maintainer.h>
#include <infra/netmon/user_data_storage.h>
#include <infra/netmon/topology/expression_storage.h>
#include <infra/netmon/topology/clients/groups.h>
#include <infra/netmon/topology/clients/staff.h>
#include <infra/netmon/settings.h>

namespace NNetmon {
    class TExpressionContextBase: public TNonCopyable {
    public:
        TExpressionContextBase(const TTopologyStorage& topologyStorage,
                               TExpressionStorage& expressionStorage,
                               const TGroupStorage& groupStorage)
            : TopologyStorage(topologyStorage)
            , ExpressionStorage(expressionStorage)
            , GroupStorage(groupStorage)
        {
        }

        inline const TTopologyStorage& GetTopologyStorage() const noexcept {
            return TopologyStorage;
        }
        inline TExpressionStorage& GetExpressionStorage() const noexcept {
            return ExpressionStorage;
        }
        inline const TGroupStorage& GetGroupStorage() const noexcept {
            return GroupStorage;
        }

    private:
        const TTopologyStorage& TopologyStorage;
        TExpressionStorage& ExpressionStorage;
        const TGroupStorage& GroupStorage;
    };

    class TExpressionContext: public TExpressionContextBase {
    public:
        TExpressionContext(const TTopologyStorage& topologyStorage,
                           TExpressionStorage& expressionStorage,
                           const TSliceCollector& sliceCollector,
                           const TProbeAggregatorMaintainer& aggregatorMaintainer,
                           const TGroupStorage& groupStorage,
                           const TStaffStorage& staffStorage,
                           const TUserDataStorage& userDataStorage)
            : TExpressionContextBase(topologyStorage, expressionStorage, groupStorage)
            , SliceCollector(sliceCollector)
            , AggregatorMaintainer(aggregatorMaintainer)
            , StaffStorage(staffStorage)
            , UserDataStorage(userDataStorage)
        {
        }

        inline const TSliceCollector& GetSliceCollector() const noexcept {
            return SliceCollector;
        }
        inline const TProbeAggregatorMaintainer& GetAggregatorMaintainer() const noexcept {
            return AggregatorMaintainer;
        }
        inline const TStaffStorage& GetStaffStorage() const noexcept {
            return StaffStorage;
        }
        inline const TUserDataStorage& GetUserDataStorage() const noexcept {
            return UserDataStorage;
        }

        inline TSet<TString> GetSudoers() const noexcept {
            return StaffStorage.ExpandLoginsAndGroups(TSettings::Get()->GetSudoers());
        }

    private:
        const TSliceCollector& SliceCollector;
        const TProbeAggregatorMaintainer& AggregatorMaintainer;
        const TStaffStorage& StaffStorage;
        const TUserDataStorage& UserDataStorage;
    };

    class TExpressionHostsReply: public TAuthorizedJsonHttpReply<TExpressionContext> {
    public:
        using TAuthorizedJsonHttpReply::TAuthorizedJsonHttpReply;

        NThreading::TFuture<void> Process();
    };

    class TExpressionExpandReply: public TJsonHttpReply<THttpDispatcher, TExpressionContextBase> {
    public:
        using TJsonHttpReply::TJsonHttpReply;

        NThreading::TFuture<void> Process();
    };

    class TExpressionListReply: public TAuthorizedJsonHttpReply<TExpressionContext> {
    public:
        using TAuthorizedJsonHttpReply::TAuthorizedJsonHttpReply;

        NThreading::TFuture<void> Process();
    };

    class TExpressionGetReply: public TAuthorizedJsonHttpReply<TExpressionContext> {
    public:
        using TAuthorizedJsonHttpReply::TAuthorizedJsonHttpReply;

        NThreading::TFuture<void> Process();
    };

    class TExpressionUpsertReply: public TAuthorizedJsonHttpReply<TExpressionContext> {
    public:
        using TAuthorizedJsonHttpReply::TAuthorizedJsonHttpReply;

        NThreading::TFuture<void> Process();
    };

    class TExpressionDeleteReply: public TAuthorizedJsonHttpReply<TExpressionContext> {
    public:
        using TAuthorizedJsonHttpReply::TAuthorizedJsonHttpReply;

        NThreading::TFuture<void> Process();
    };

    class TMetadataRequestOps {
    public:
        static TVector<TString> ExtractFieldList(const NJson::TJsonValue& request);
    };

    class TLimitsRequestOps {
    public:
        static TVector<TString> ExtractFieldList(const NJson::TJsonValue&) {
            return { "limits" };
        }
    };

    template<class TRequestOps>
    class TExpressionMetadataBaseReply: public TAuthorizedJsonHttpReply<TExpressionContext> {
    public:
        using TAuthorizedJsonHttpReply::TAuthorizedJsonHttpReply;

        NThreading::TFuture<void> Process();
    };

    template<class TRequestOps>
    class TExpressionMetadataUpsertBaseReply: public TAuthorizedJsonHttpReply<TExpressionContext> {
    public:
        using TAuthorizedJsonHttpReply::TAuthorizedJsonHttpReply;

        NThreading::TFuture<void> Process();
    };

    template<class TRequestOps>
    class TExpressionMetadataDeleteBaseReply: public TAuthorizedJsonHttpReply<TExpressionContext> {
    public:
        using TAuthorizedJsonHttpReply::TAuthorizedJsonHttpReply;

        NThreading::TFuture<void> Process();
    };

    using TExpressionMetadataReply = TExpressionMetadataBaseReply<TMetadataRequestOps>;
    using TExpressionMetadataUpsertReply = TExpressionMetadataUpsertBaseReply<TMetadataRequestOps>;
    using TExpressionMetadataDeleteReply = TExpressionMetadataDeleteBaseReply<TMetadataRequestOps>;

    using TExpressionLimitsReply = TExpressionMetadataBaseReply<TLimitsRequestOps>;
    using TExpressionLimitsUpsertReply = TExpressionMetadataUpsertBaseReply<TLimitsRequestOps>;
    using TExpressionLimitsDeleteReply = TExpressionMetadataDeleteBaseReply<TLimitsRequestOps>;

    using TExpressionHostsHandler = THttpHandler<TExpressionHostsReply>;
    using TExpressionExpandHandler = THttpHandler<TExpressionExpandReply>;
    using TExpressionListHandler = THttpHandler<TExpressionListReply>;
    using TExpressionGetHandler = THttpHandler<TExpressionGetReply>;
    using TExpressionUpsertHandler = THttpHandler<TExpressionUpsertReply>;
    using TExpressionDeleteHandler = THttpHandler<TExpressionDeleteReply>;
    using TExpressionLimitsHandler = THttpHandler<TExpressionLimitsReply>;
    using TExpressionLimitsUpsertHandler = THttpHandler<TExpressionLimitsUpsertReply>;
    using TExpressionLimitsDeleteHandler = THttpHandler<TExpressionLimitsDeleteReply>;
    using TExpressionMetadataHandler = THttpHandler<TExpressionMetadataReply>;
    using TExpressionMetadataUpsertHandler = THttpHandler<TExpressionMetadataUpsertReply>;
    using TExpressionMetadataDeleteHandler = THttpHandler<TExpressionMetadataDeleteReply>;
}
