#include <infra/netmon/api/history_api.h>
#include <infra/netmon/library/iterators.h>

namespace NNetmon {
    namespace {
        template <class TPairStateHistory>
        void StateToJson(NJsonWriter::TBuf& jsonWriter, const TPairStateHistory& state) {
            jsonWriter.BeginObject()
                .WriteKey(TStringBuf("history"))
                .BeginList()
                .BeginObject()
                .WriteKey(TStringBuf("source"))
                .WriteString(state.GetSource().GetName())
                .WriteKey(TStringBuf("target"))
                .WriteString(state.GetTarget().GetName())
                .WriteKey(TStringBuf("series"))
                .BeginList();

            const auto timestampSeries(state.GetTimestampSeries());
            const auto connectivitySeries(state.GetConnectivitySeries());
            const auto rttSeries(state.GetRttSeries());
            for (const auto& point : ZipIterator(timestampSeries, connectivitySeries, rttSeries)) {
                jsonWriter.BeginObject();

                jsonWriter.WriteKey(TStringBuf("connectivity"));
                std::get<1>(point).ToJson(jsonWriter);

                jsonWriter.WriteKey(TStringBuf("rtt"));
                std::get<2>(point).ToJson(jsonWriter);

                jsonWriter.WriteKey(TStringBuf("timestamp"));
                jsonWriter.WriteString(std::get<0>(point).ToString());

                jsonWriter.EndObject();
            }

            jsonWriter
                .EndList()
                .EndObject()
                .EndList()
                .EndObject();
        }
    }

    // api/v1/dc/history handler

    TThreadPool::TFuture TDcHistoryReply::Process() {
        return GetServerContext().GetHistoryRequester().CollectDatacenterHistorySeries(
            GetRequestData().GetSince(),
            GetRequestData().GetUntil(),
            GetRequestData().GetAggregatorKey(),
            GetRequestData().GetDatacenterPair()
        ).Apply([this] (const NThreading::TFuture<TDatacenterPairStateHistory::TRef>& future) {
            StateToJson(GetResponse(), *future.GetValueSync());
        });
    }

    // api/v1/queue/history handler

    TThreadPool::TFuture TLineHistoryReply::Process() {
        return GetServerContext().GetHistoryRequester().CollectLineHistorySeries(
            GetRequestData().GetSince(),
            GetRequestData().GetUntil(),
            GetRequestData().GetAggregatorKey(),
            GetRequestData().GetLinePair()
        ).Apply([this] (const NThreading::TFuture<TLinePairStateHistory::TRef>& future) {
            StateToJson(GetResponse(), *future.GetValueSync());
        });
    }

    // api/v1/switch/history handler

    TThreadPool::TFuture TSwitchHistoryReply::Process() {
        return GetServerContext().GetHistoryRequester().CollectSwitchHistorySeries(
            GetRequestData().GetSince(),
            GetRequestData().GetUntil(),
            GetRequestData().GetAggregatorKey(),
            GetRequestData().GetSwitchPair()
        ).Apply([this] (const NThreading::TFuture<TSwitchPairStateHistory::TRef>& future) {
            StateToJson(GetResponse(), *future.GetValueSync());
        });
    }
}
