#include <infra/netmon/api/network_availability_api.h>
#include <infra/netmon/settings.h>

namespace NNetmon {
    namespace {
        template <class TAccessor>
        class TAvailabilityView: public TNonCopyable {
        public:
            TAvailabilityView(NJsonWriter::TBuf& jsonWriter, const TRequestData& requestData, const TAggregatorContext& context)
                : JsonWriter(jsonWriter)
                , RequestData(requestData)
                , Accessor(RequestData)
            {
                if (!context.GetSliceCollector().IsReady()) {
                    ythrow yexception() << "slices aren't collected yet";
                }
                if (Accessor.GetUpdateTime() + Accessor.GetInterval() * 3 < TInstant::Now()) {
                    ythrow yexception() << "that state is too old";
                }

                const auto& availability(Accessor.GetIndex().GetAvailability());

                TNetworkAvailability::TSwitchMap switchStates;
                availability.FillMap(switchStates);

                TNetworkAvailability::TLineMap queueStates;
                availability.FillMap(queueStates);

                TNetworkAvailability::TDatacenterMap datacenterStates;
                availability.FillMap(datacenterStates);

                JsonWriter
                    .BeginObject()
                    .WriteKey(TStringBuf("generated"))
                    .WriteInt(Accessor.GetUpdateTime().Seconds())
                    .WriteKey(TStringBuf("switches"))
                    .BeginList();

                for (const auto& pair : switchStates) {
                    JsonWriter
                        .BeginObject()
                        .WriteKey(TStringBuf("name"))
                        .WriteString(pair.first->GetName())
                        .WriteKey(TStringBuf("queue"))
                        .WriteString(pair.first->GetLine().GetName())
                        .WriteKey(TStringBuf("dc"))
                        .WriteString(pair.first->GetDatacenter().GetName());
                    DumpMetrics(pair.second);
                    JsonWriter.EndObject();
                }

                JsonWriter
                    .EndList()
                    .WriteKey(TStringBuf("queues"))
                    .BeginList();

                for (const auto& pair : queueStates) {
                    JsonWriter
                        .BeginObject()
                        .WriteKey(TStringBuf("name"))
                        .WriteString(pair.first->GetName())
                        .WriteKey(TStringBuf("dc"))
                        .WriteString(pair.first->GetDatacenter().GetName());
                    DumpMetrics(pair.second);
                    JsonWriter.EndObject();
                }

                JsonWriter
                    .EndList()
                    .WriteKey(TStringBuf("datacenters"))
                    .BeginList();

                for (const auto& pair : datacenterStates) {
                    JsonWriter
                        .BeginObject()
                        .WriteKey(TStringBuf("name"))
                        .WriteString(pair.first->GetName());
                    DumpMetrics(pair.second);
                    JsonWriter.EndObject();
                }

                JsonWriter
                    .EndList()
                    .EndObject();
            }

        private:
            void DumpMetrics(const TNetworkAvailability::TAvailabilityState& state) {
                JsonWriter.WriteKey(TStringBuf("alive"));
                auto alive(state.Alive.GetValue());
                if (alive.Defined()) {
                    JsonWriter.WriteDouble(alive.GetRef());
                } else {
                    JsonWriter.WriteNull();
                }

                JsonWriter.WriteKey(TStringBuf("connectivity"));
                state.Connectivity.ToJson(JsonWriter);
            }

            NJsonWriter::TBuf& JsonWriter;
            const TRequestData& RequestData;
            TAccessor Accessor;
        };
    }

    // api/v1/dc/alive handler

    void TDcAliveReply::Process() {
        class TAccessor {
        public:
            TAccessor(const TRequestData& requestData)
            {
                const auto aggregator(requestData.GetProbeAggregator());
                Index = aggregator->GetDatacenterIndex();
            }

            const TDuration& GetInterval() const {
                return TSettings::Get()->GetDcAggregationInterval();
            }

            const TDatacenterPairIndex& GetIndex() const {
                if (!Index) {
                    ythrow yexception() << "index isn't collected yet";
                }
                return *Index;
            }

            TInstant GetUpdateTime() const {
                return Index ? Index->GetGenerated() : TInstant::Zero();
            }

        private:
            TDatacenterPairIndex::TRef Index;
        };

        TAvailabilityView<TAccessor> view(GetResponse(), GetRequestData(), GetServerContext());
    }

    // api/v1/queue/alive handler

    void TLineAliveReply::Process() {
        class TAccessor {
        public:
            TAccessor(const TRequestData& requestData)
            {
                const auto aggregator(requestData.GetProbeAggregator());
                Index = aggregator->GetLineIndex();
            }

            const TDuration& GetInterval() const {
                return TSettings::Get()->GetLineAggregationInterval();
            }

            const TLinePairIndex& GetIndex() const {
                if (!Index) {
                    ythrow yexception() << "index isn't collected yet";
                }
                return *Index;
            }

            TInstant GetUpdateTime() const {
                return Index ? Index->GetGenerated() : TInstant::Zero();
            }

        private:
            TLinePairIndex::TRef Index;
        };

        TAvailabilityView<TAccessor> view(GetResponse(), GetRequestData(), GetServerContext());
    }

    // api/v1/switch/alive handler

    void TSwitchAliveReply::Process() {
        class TAccessor {
        public:
            TAccessor(const TRequestData& requestData)
            {
                const auto aggregator(requestData.GetProbeAggregator());
                Index = aggregator->GetSwitchIndex();
            }

            const TDuration& GetInterval() const {
                return TSettings::Get()->GetSwitchAggregationInterval();
            }

            const TSwitchPairIndex& GetIndex() const {
                if (!Index) {
                    ythrow yexception() << "index isn't collected yet";
                }
                return *Index;
            }

            TInstant GetUpdateTime() const {
                return Index ? Index->GetGenerated() : TInstant::Zero();
            }

        private:
            TSwitchPairIndex::TRef Index;
        };

        TAvailabilityView<TAccessor> view(GetResponse(), GetRequestData(), GetServerContext());
    }
}
