#include <infra/netmon/api/probes_api.h>
#include <infra/netmon/settings.h>

#include <infra/netmon/library/clickhouse/translate.h>

#include <util/generic/ymath.h>

namespace NNetmon {
    namespace {
        const TDuration QUERY_TIMEOUT = TDuration::Seconds(15);

        const TString JSON_NULL = TString("null");

        inline TString RttToJson(double value) {
            if (!IsNan(value)) {
                if (value < 0.0) {
                    return JSON_NULL;
                } else {
                    return ::ToString(value * 1000.0);
                }
            } else {
                return JSON_NULL;
            }
        }

        class TProbesView: public TNonCopyable {
        public:
            static TThreadPool::TFuture Dispatch(
                    NJsonWriter::TBuf& jsonWriter, const TRequestData& requestData,
                    const TAggregatorContext& context, NApi::ELevel sliceLevel) {

                THolder<TProbesView> view(new TProbesView(
                    jsonWriter, requestData, context, sliceLevel
                ));

                flatbuffers::FlatBufferBuilder builder;
                view->FillRequest(builder);
                return context.GetSliceCollector()
                    .CollectProbes(builder)
                    .Apply([view_ = view.Release()] (const NThreading::TFuture<TSliceCollector::TProbesResponseList>& future_) {
                        THolder<TProbesView> view(view_);
                        view->WriteReply(future_.GetValue());
                    });
            }

        private:
            TProbesView(
                    NJsonWriter::TBuf& jsonWriter, const TRequestData& requestData,
                    const TAggregatorContext& context, NApi::ELevel sliceLevel)
                : JsonWriter(jsonWriter)
                , RequestData(requestData)
                , Context(context)
                , SliceLevel(sliceLevel)
            {
            }

            void FillRequest(flatbuffers::FlatBufferBuilder& builder) {
                const TProbeAggregatorKey aggregatorKey(RequestData.GetAggregatorKey());

                const TSwitchPairKey switchKey(RequestData.GetSwitchPair(false));
                const TLinePairKey queueKey(RequestData.GetLinePair(false));
                const TDatacenterPairKey datacenterKey(RequestData.GetDatacenterPair(false));

                builder.Finish(NApi::CreateTProbesRequest(
                    builder,
                    SliceLevel,
                    &aggregatorKey.ToProto(),
                    switchKey.ToOptionalProto(),
                    queueKey.ToOptionalProto(),
                    datacenterKey.ToOptionalProto(),
                    RequestData.GetSince().Seconds(),
                    RequestData.GetUntil().Seconds(),
                    RequestData.GetMinScore(),
                    RequestData.GetMaxScore(),
                    RequestData.HasMinRtt() ? RequestData.GetMinRtt() : 0,
                    RequestData.HasMaxRtt() ? RequestData.GetMaxRtt() : 0,
                    RequestData.GetLimit()
                ));
            }

            void WriteReply(const TSliceCollector::TProbesResponseList& responses) {
                const TTopologyStorage& topologyStorage(Context.GetTopologyStorage());

                ui64 totalProbes(0);
                ui64 filteredProbes(0);
                TVector<const NProbe::TProbe*> collectedProbes;
                for (const auto& response : responses) {
                    totalProbes += response->TotalProbes();
                    filteredProbes += response->FilteredProbes();
                    for (const auto& probe : *response->Probes()) {
                        collectedProbes.emplace_back(probe);
                    }
                }

                auto sortAscending = RequestData.GetSortOrder() == TRequestData::SORT_ASCENDING;
                if (RequestData.GetSortField() == TRequestData::SORT_RTT) {
                    Sort(collectedProbes, [&](const NProbe::TProbe* lhs, const NProbe::TProbe* rhs) {
                        return sortAscending
                                ? lhs->RoundTripTime() < rhs->RoundTripTime()
                                : lhs->RoundTripTime() > rhs->RoundTripTime();
                    });
                } else {
                    Sort(collectedProbes, [&](const NProbe::TProbe* lhs, const NProbe::TProbe* rhs) {
                        return sortAscending
                                ? lhs->Score() < rhs->Score()
                                : lhs->Score() > rhs->Score();
                    });
                }

                // start root object
                JsonWriter.BeginObject()
                    .WriteKey(TStringBuf("probes"))
                    .BeginList();

                std::size_t recordCount = 0;
                for (const auto& probe : collectedProbes) {
                    if (recordCount > RequestData.GetLimit()) {
                        break;
                    }

                    const auto sourceIface(topologyStorage.FindHostInterface(*probe->SourceIface()));
                    const auto targetIface(topologyStorage.FindHostInterface(*probe->TargetIface()));
                    if (!sourceIface || !targetIface) {
                        continue;
                    }

                    const TIpAddress sourceAddress(*probe->SourceAddress());
                    const TIpAddress targetAddress(*probe->TargetAddress());

                    JsonWriter.BeginObject()
                        .WriteKey(TStringBuf("source"))
                        .WriteString(sourceIface->GetName())
                        .WriteKey(TStringBuf("source_switch"))
                        .WriteString(sourceIface->GetSwitch().GetName())
                        .WriteKey(TStringBuf("source_queue"))
                        .WriteString(sourceIface->GetLine().GetName())
                        .WriteKey(TStringBuf("source_dc"))
                        .WriteString(sourceIface->GetDatacenter().GetName())
                        .WriteKey(TStringBuf("source_address"))
                        .WriteString(sourceAddress.ToString())
                        .WriteKey(TStringBuf("source_port"))
                        .WriteInt(probe->SourcePort())
                        .WriteKey(TStringBuf("target"))
                        .WriteString(targetIface->GetName())
                        .WriteKey(TStringBuf("target_switch"))
                        .WriteString(targetIface->GetSwitch().GetName())
                        .WriteKey(TStringBuf("target_queue"))
                        .WriteString(targetIface->GetLine().GetName())
                        .WriteKey(TStringBuf("target_dc"))
                        .WriteString(targetIface->GetDatacenter().GetName())
                        .WriteKey(TStringBuf("target_address"))
                        .WriteString(targetAddress.ToString())
                        .WriteKey(TStringBuf("target_port"))
                        .WriteInt(probe->TargetPort())
                        .WriteKey(TStringBuf("generated"))
                        .WriteString(TInstant::Seconds(probe->Generated()).ToString())
                        .WriteKey(TStringBuf("score"))
                        .WriteDouble(probe->Score());
                    if (probe->RoundTripTime() > 0.0) {
                        JsonWriter
                            .WriteKey(TStringBuf("rtt"))
                            .WriteDouble(probe->RoundTripTime());
                    } else {
                        JsonWriter
                            .WriteKey(TStringBuf("rtt"))
                            .UnsafeWriteValue(TStringBuf("null"));
                    }
                    JsonWriter.EndObject();
                    recordCount++;
                }

                // finish root object
                JsonWriter.EndList()
                    .WriteKey(TStringBuf("total_probes"))
                    .WriteULongLong(totalProbes)
                    .WriteKey(TStringBuf("filtered_probes"))
                    .WriteULongLong(filteredProbes)
                    .EndObject();
            }

            NJsonWriter::TBuf& JsonWriter;
            const TRequestData& RequestData;
            const TAggregatorContext& Context;
            const NApi::ELevel SliceLevel;
        };
    }

    // abstract replies

    NThreading::TFuture<void> TClickhouseAggregatorReply::Process() {
        TClickhouseClient::TQueryOptions options(GetQuery());
        options
            .SetTimeout(QUERY_TIMEOUT)
            .RandomShardIndex();
        OnStart();
        return TClickhouseClient::Get()->Select(options, [this](const NClickHouse::TBlock& block) {
            OnBlock(block);
        }).Apply([this] (const NThreading::TFuture<void>& future) {
            future.GetValue();
            OnEnd();
        });
    }

    // api/v1/probes/history handler

    TString TProbesHistoryReply::GetQuery() {
        ExpressionId = GetRequestData().GetExpressionId();

        TStringBuilder builder;
        builder
            << "SELECT "
            << "SourceFqdn, TargetFqdn, toUInt64(Generated), Score, Rtt, "
            << "SourceAddress, SourcePort, TargetAddress, TargetPort, "
            << "SourceDc, TargetDc, SourceQueue, TargetQueue "
            << "FROM distributed_probes "
            << "WHERE Score >= " << GetRequestData().GetMinScore() << " AND Score <= " << GetRequestData().GetMaxScore() << " "
            << "AND Inserted >= toDateTime('" << GetRequestData().GetSinceHuman() << "') "
            << "AND Inserted <= toDateTime('" << GetRequestData().GetUntilHuman() << "') "
            << "AND InsertedDate >= toDate('" << GetRequestData().GetSinceHumanDate() << "') "
            << "AND InsertedDate <= toDate('" << GetRequestData().GetUntilHumanDate() << "') "
            << "AND Network = " << ui64(GetRequestData().GetNetwork()) << " "
            << "AND Protocol = " << ui64(GetRequestData().GetProtocol()) << " ";

        if (GetRequestData().HasSourceDc()) {
            builder << "AND SourceDc = " << GetRequestData().GetSourceDatacenter()->GetReducedId() << " ";
        }
        if (GetRequestData().HasTargetDc()) {
            builder << "AND TargetDc = " << GetRequestData().GetTargetDatacenter()->GetReducedId() << " ";
        }
        if (GetRequestData().HasSourceQueue()) {
            builder << "AND SourceQueue = " << GetRequestData().GetSourceLine()->GetReducedId() << " ";
        }
        if (GetRequestData().HasTargetQueue()) {
            builder << "AND TargetQueue = " << GetRequestData().GetTargetLine()->GetReducedId() << " ";
        }
        if (GetRequestData().HasSourceSwitch()) {
            builder << "AND SourceSwitch = " << GetRequestData().GetSourceSwitch()->GetReducedId() << " ";
        }
        if (GetRequestData().HasTargetSwitch()) {
            builder << "AND TargetSwitch = " << GetRequestData().GetTargetSwitch()->GetReducedId() << " ";
        }
        if (GetRequestData().HasMinRtt()) {
            builder << "AND Rtt >= " << GetRequestData().GetMinRtt() << " ";
        }
        if (GetRequestData().HasMaxRtt()) {
            builder << "AND Rtt <= " << GetRequestData().GetMaxRtt() << " ";
        }

        TStringBuf sortOrder, sortField;
        if (GetRequestData().GetSortOrder() == TRequestData::SORT_ASCENDING) {
            sortOrder = TStringBuf("ASC");
        } else {
            Y_ASSERT(GetRequestData().GetSortOrder() == TRequestData::SORT_DESCENDING);
            sortOrder = TStringBuf("DESC");
        }
        if (GetRequestData().GetSortField() == TRequestData::SORT_RTT) {
            sortField = TStringBuf("Rtt");
        } else {
            Y_ASSERT(GetRequestData().GetSortField() == TRequestData::SORT_SCORE);
            sortField = TStringBuf("Score");
        }

        builder
            << "ORDER BY " << sortField << " " << sortOrder << ", " << "Generated DESC, SourceFqdn, TargetFqdn "
            << "LIMIT " << GetRequestData().GetLimit() * 100 << " BY SourceDc, SourceQueue, TargetDc, TargetQueue";

        return builder;
    }

    void TProbesHistoryReply::OnStart() {
        SeenHosts = GetServerContext().GetSeenHostsUpdater().GetHosts();
        DeadHosts = GetServerContext().GetWalleUpdater().GetHosts();

        GetResponse()
            .BeginObject()
            .WriteKey(TStringBuf("probes"))
            .BeginList();
    }

    void TProbesHistoryReply::OnBlock(const NClickHouse::TBlock& block) {
        if (Selected >= GetRequestData().GetLimit()) {
            return;
        }

        using TRow = std::tuple<ui64, ui64, ui64, double, double,
                                TString, ui16, TString, ui16,
                                ui64, ui64, ui64, ui64>;
        const auto rows(TranslateClickhouseBlock<TRow>(block));

        for (const auto& row : rows) {
            const auto& topology(GetServerContext().GetTopologyStorage());
            const auto selector(topology.GetTopologySelector());

            const auto sourceIface(topology.FindHostInterface(std::get<0>(row)));
            const auto targetIface(topology.FindHostInterface(std::get<1>(row)));
            if (!sourceIface || !targetIface) {
                continue;
            }

            // TODO: don't use hosts from future!
            if (!SeenHosts->empty() && SeenHosts->find(targetIface->GetHost()).IsEnd()) {
                continue;
            } else if (!DeadHosts->find(targetIface->GetHost()).IsEnd()) {
                continue;
            } else if (!DeadHosts->find(sourceIface->GetHost()).IsEnd()) {
                continue;
            }

            const auto* sourceExpressions(selector->FindSourceExpressions(sourceIface));
            const auto* targetExpressions(selector->FindTargetExpressions(targetIface));
            if (!sourceExpressions || !targetExpressions) {
                continue;
            }

            if (
                !BinarySearch(sourceExpressions->begin(), sourceExpressions->end(), ExpressionId)
                || !BinarySearch(targetExpressions->begin(), targetExpressions->end(), ExpressionId)
            ) {
                continue;
            }

            const TIpAddress sourceAddress(std::get<5>(row));
            const TIpAddress targetAddress(std::get<7>(row));

            GetResponse()
                .BeginObject()
                .WriteKey(TStringBuf("source_dc"))
                .WriteString(sourceIface->GetDatacenter().GetName())
                .WriteKey(TStringBuf("target_dc"))
                .WriteString(targetIface->GetDatacenter().GetName())
                .WriteKey(TStringBuf("source_queue"))
                .WriteString(sourceIface->GetLine().GetName())
                .WriteKey(TStringBuf("target_queue"))
                .WriteString(targetIface->GetLine().GetName())
                .WriteKey(TStringBuf("source_switch"))
                .WriteString(sourceIface->GetSwitch().GetName())
                .WriteKey(TStringBuf("target_switch"))
                .WriteString(targetIface->GetSwitch().GetName())
                .WriteKey(TStringBuf("source_address"))
                .WriteString(sourceAddress.ToString())
                .WriteKey(TStringBuf("target_address"))
                .WriteString(targetAddress.ToString())
                .WriteKey(TStringBuf("source_port"))
                .WriteInt(std::get<6>(row))
                .WriteKey(TStringBuf("target_port"))
                .WriteInt(std::get<8>(row))
                .WriteKey(TStringBuf("source"))
                .WriteString(sourceIface->GetName())
                .WriteKey(TStringBuf("target"))
                .WriteString(targetIface->GetName())
                .WriteKey(TStringBuf("generated"))
                .WriteString(TInstant::Seconds(std::get<2>(row)).ToString())
                .WriteKey(TStringBuf("score"))
                .WriteDouble(std::get<3>(row))
                .WriteKey(TStringBuf("rtt"))
                .UnsafeWriteValue(RttToJson(std::get<4>(row)))
                .EndObject();

            Selected++;
            if (Selected >= GetRequestData().GetLimit()) {
                return;
            }
        }
    }

    void TProbesHistoryReply::OnEnd() {
        GetResponse()
            .EndList()
            .EndObject();
    }

    // api/v1/dc/probes handler

    TThreadPool::TFuture TDcProbesReply::Process() {
        return TProbesView::Dispatch(
            GetResponse(), GetRequestData(), GetServerContext(), NApi::ELevel_LEVEL_DATACENTER
        );
    }

    // api/v1/queue/probes handler

    TThreadPool::TFuture TLineProbesReply::Process() {
        return TProbesView::Dispatch(
            GetResponse(), GetRequestData(), GetServerContext(), NApi::ELevel_LEVEL_QUEUE
        );
    }

    // api/v1/switch/probes handler

    TThreadPool::TFuture TSwitchProbesReply::Process() {
        return TProbesView::Dispatch(
            GetResponse(), GetRequestData(), GetServerContext(), NApi::ELevel_LEVEL_SWITCH
        );
    }
}
