#include <infra/netmon/api/slicer_api.h>
#include <infra/netmon/metrics.h>

#include <util/generic/xrange.h>

namespace NNetmon {
    namespace {
        template <class T, class R>
        struct TBaseVisitor {
            using TReturnType = R;

            template <typename... Args>
            static inline R Visit(NApi::ELevel sliceLevel, Args&&... args) {
                T visitor;
                switch (sliceLevel) {
                    case NApi::ELevel_LEVEL_SWITCH:
                        return visitor.OnSwitch(std::forward<Args>(args)...);
                    case NApi::ELevel_LEVEL_QUEUE:
                        return visitor.OnQueue(std::forward<Args>(args)...);
                    case NApi::ELevel_LEVEL_DATACENTER:
                        return visitor.OnDatacenter(std::forward<Args>(args)...);
                    default:
                        Y_VERIFY(false);
                }
            }
        };

        struct TFindProbes : public TBaseVisitor<TFindProbes, NThreading::TFuture<TProbe::TRefVector>> {
            TReturnType OnSwitch(TProbeGatherer& gatherer, TProbeGatherer::TQuery& query) {
                return gatherer.FindProbes<TProbeStorage::TSwitchIndexAccessor>(query);
            }

            TReturnType OnQueue(TProbeGatherer& gatherer, TProbeGatherer::TQuery& query) {
                return gatherer.FindProbes<TProbeStorage::TLineIndexAccessor>(query);
            }

            TReturnType OnDatacenter(TProbeGatherer& gatherer, TProbeGatherer::TQuery& query) {
                return gatherer.FindProbes<TProbeStorage::TDatacenterIndexAccessor>(query);
            }
        };

        struct TGetStateKeys : public TBaseVisitor<TGetStateKeys, TProbeAggregatorKeySet> {
            TReturnType OnSwitch(const TProbeSliceMerger& merger) {
                return merger.GetSwitchLevelKeys();
            }

            TReturnType OnQueue(const TProbeSliceMerger& merger) {
                return merger.GetLineLevelKeys();
            }

            TReturnType OnDatacenter(const TProbeSliceMerger& merger) {
                return merger.GetDatacenterLevelKeys();
            }
        };

        TProbeGatherer::TRef CreateProbeGatherer(const TSlicerContext& context, const NCommon::TAggregatorKey& key) {
            const TProbeAggregatorKey restoredKey(key);
            return context.GetSliceMaintainer().CreateProbeGatherer(restoredKey);
        }

        NThreading::TFuture<TProbe::TRefVector> GetProbes(const TSlicerContext& context, const NApi::TProbesRequest& request) {
            const TTopologyStorage& topologyStorage(context.GetTopologyStorage());

            TProbeGatherer::TQuery query;
            if (request.SwitchPairKey()) {
                query.SwitchKey.ConstructInPlace(topologyStorage, *request.SwitchPairKey());
            }
            if (request.QueuePairKey()) {
                query.QueueKey.ConstructInPlace(topologyStorage, *request.QueuePairKey());
            }
            if (request.DatacenterPairKey()) {
                query.DatacenterKey.ConstructInPlace(topologyStorage, *request.DatacenterPairKey());
            }

            const auto gatherer(CreateProbeGatherer(context, *request.AggregatorKey()));
            return TFindProbes::Visit(request.SliceLevel(), *gatherer, query);
        }

        template <class T>
        using TFlatIndexVector = std::vector<flatbuffers::Offset<T>>;

        template <class TIndex>
        TFlatIndexVector<typename TIndex::TFlat> GetFlatStates(
                const TSlicerContext& context, const NApi::TStateRequest& request, flatbuffers::FlatBufferBuilder& builder) {

            if (!context.GetWalleUpdater().IsReady()) {
                ythrow yexception() << "wall-e cache isn't ready yet";
            }
            if (!context.GetSeenHostsUpdater().IsReady()) {
                ythrow yexception() << "seen hosts cache isn't ready yet";
            }
            if (!context.GetSliceMaintainer().IsReady()) {
                ythrow yexception() << "slices aren't ready yet";
            }

            TVector<TProbeAggregatorKey> aggregatorKeys;
            aggregatorKeys.reserve(request.Keys()->size());
            for (const auto& key : *request.Keys()) {
                aggregatorKeys.emplace_back(*key);
            }

            TFlatIndexVector<typename TIndex::TFlat> states;
            states.reserve(aggregatorKeys.size());
            for (const auto& key : aggregatorKeys) {
                auto index(GetMergerIndex<TIndex>(context.GetSliceMerger(), key));
                states.emplace_back(index->ToProto(builder));
            }

            return states;
        }

        template <class TIndex, class TRequest, typename... Args>
        TFlatIndexVector<typename TIndex::TFlat> GetFlatHistoryStates(
                const TProbeSliceDumper& dumper, const TRequest& request,
                flatbuffers::FlatBufferBuilder& builder, Args&&... args) {

            if (!dumper.IsReady()) {
                ythrow yexception() << "slice dumper isn't ready yet";
            }

            const auto index(dumper.Read(
                TInstant::MicroSeconds(request.Generated()),
                TProbeAggregatorKey(*request.AggregatorKey()),
                std::forward<Args>(args)...
            ));

            TFlatIndexVector<typename TIndex::TFlat> states;
            if (index) {
                states.emplace_back(index->ToProto(builder));
            }
            return states;
        }

        template <class TFlat, class TRequest>
        TFlatIndexVector<TFlat> GetFlatHistorySeries(
                const TProbeSliceHistory& history, const TTopologyStorage& topologyStorage,
                const TRequest& request, flatbuffers::FlatBufferBuilder& builder) {

            const TProbeAggregatorKey key(*request.AggregatorKey());
            const auto pairKey(ToPairKey(topologyStorage, *request.PairKey()));

            const auto since(TInstant::MicroSeconds(request.Since()));
            const auto until(TInstant::MicroSeconds(request.Until()));

            TFlatIndexVector<TFlat> result;
            if (pairKey.IsValid()) {
                for (const auto& state : history.Read(since, until, key, pairKey)) {
                    result.emplace_back(state->ToProto(builder));
                }
            }

            return result;
        }
    }

    void TDatacenterStateReply::Process() {
        TUnistatTimer timer{TUnistat::Instance(), ENetmonSignals::SlicerStateReplyTime};
        Builder.Finish(NApi::CreateTDatacenterStateResponse(
            Builder,
            Builder.CreateVector(GetFlatStates<TDatacenterPairIndex>(GetServerContext(), *Request, Builder))
        ));
    }

    void TLineStateReply::Process() {
        TUnistatTimer timer{TUnistat::Instance(), ENetmonSignals::SlicerStateReplyTime};
        Builder.Finish(NApi::CreateTLineStateResponse(
            Builder,
            Builder.CreateVector(GetFlatStates<TLinePairIndex>(GetServerContext(), *Request, Builder))
        ));
    }

    void TSwitchStateReply::Process() {
        TUnistatTimer timer{TUnistat::Instance(), ENetmonSignals::SlicerStateReplyTime};
        Builder.Finish(NApi::CreateTSwitchStateResponse(
            Builder,
            Builder.CreateVector(GetFlatStates<TSwitchPairIndex>(GetServerContext(), *Request, Builder))
        ));
    }

    void TDatacenterHistoryStateReply::Process() {
        Builder.Finish(NApi::CreateTDatacenterStateResponse(
            Builder,
            Builder.CreateVector(GetFlatHistoryStates<TDatacenterPairIndex>(
                GetServerContext().GetSliceDumper(), *Request, Builder
            ))
        ));
    }

    void TLineHistoryStateReply::Process() {
        Builder.Finish(NApi::CreateTLineStateResponse(
            Builder,
            Builder.CreateVector(GetFlatHistoryStates<TLinePairIndex>(
                GetServerContext().GetSliceDumper(), *Request, Builder,
                ToPairKey(GetServerContext().GetTopologyStorage(), *Request->PairKey())
            ))
        ));
    }

    void TSwitchHistoryStateReply::Process() {
        Builder.Finish(NApi::CreateTSwitchStateResponse(
            Builder,
            Builder.CreateVector(GetFlatHistoryStates<TSwitchPairIndex>(
                GetServerContext().GetSliceDumper(), *Request, Builder,
                ToPairKey(GetServerContext().GetTopologyStorage(), *Request->PairKey())
            ))
        ));
    }

    void TDatacenterHistorySeriesReply::Process() {
        if (!GetServerContext().GetSliceHistory().IsDatacenterReady()) {
            ythrow yexception() << "slice history isn't ready yet";
        }
        Builder.Finish(NApi::CreateTDatacenterHistorySeriesResponse(
            Builder,
            Builder.CreateVector(GetFlatHistorySeries<NHistory::TDatacenterPairStateHistory>(
                GetServerContext().GetSliceHistory(),
                GetServerContext().GetTopologyStorage(),
                *Request,
                Builder
            ))
        ));
    }

    void TLineHistorySeriesReply::Process() {
        if (!GetServerContext().GetSliceHistory().IsLineReady()) {
            ythrow yexception() << "slice history isn't ready yet";
        }
        Builder.Finish(NApi::CreateTLineHistorySeriesResponse(
            Builder,
            Builder.CreateVector(GetFlatHistorySeries<NHistory::TLinePairStateHistory>(
                GetServerContext().GetSliceHistory(),
                GetServerContext().GetTopologyStorage(),
                *Request,
                Builder
            ))
        ));
    }

    void TSwitchHistorySeriesReply::Process() {
        if (!GetServerContext().GetSliceHistory().IsSwitchReady()) {
            ythrow yexception() << "slice history isn't ready yet";
        }
        Builder.Finish(NApi::CreateTSwitchHistorySeriesResponse(
            Builder,
            Builder.CreateVector(GetFlatHistorySeries<NHistory::TSwitchPairStateHistory>(
                GetServerContext().GetSliceHistory(),
                GetServerContext().GetTopologyStorage(),
                *Request,
                Builder
            ))
        ));
    }

    void TAggregatorKeysReply::Process() {
        TProbeAggregatorKeySet keySet(TGetStateKeys::Visit(Request->SliceLevel(), GetServerContext().GetSliceMerger()));
        std::vector<NCommon::TAggregatorKey> keys;
        keys.reserve(keySet.size());
        for (const auto& key : keySet) {
            keys.push_back(key.ToProto());
        }

        Builder.Finish(NApi::CreateTAggregatorKeysResponse(
            Builder,
            Builder.CreateVectorOfStructs(keys)
        ));
    }

    void TProbesReply::Process() {
        TProbe::TRefVector probeList(GetProbes(GetServerContext(), *Request).GetValue(TDuration::Max()));

        // group probes by switch before filtering
        THashMap<TSwitchPairKey, TProbe::TRefVector> groupedProbes;

        std::vector<flatbuffers::Offset<NProbe::TProbe>> resultProbes;
        resultProbes.reserve(Request->Limit() ? Min(probeList.size(), (size_t)Request->Limit()) : probeList.size());

        size_t found(0);
        const auto seenHosts = GetServerContext().GetSeenHostsUpdater().GetHosts();
        const auto terminatedHosts = GetServerContext().GetTerminatedHostsMaintainer().GetHosts();
        const auto deadHosts = GetServerContext().GetWalleUpdater().GetHosts();
        for (auto& probe : probeList) {
            if (Request->GeneratedSince() && probe->GetGenerated().Seconds() < Request->GeneratedSince()) {
                continue;
            } else if (Request->GeneratedUntil() && Request->GeneratedUntil() < probe->GetGenerated().Seconds()) {
                continue;
            } else if (Request->MinScore() && probe->GetScore() < Request->MinScore()) {
                continue;
            } else if (Request->MaxScore() && Request->MaxScore() < probe->GetScore()) {
                continue;
            } else if (Request->MinRtt() && Request->MinRtt() > probe->GetRoundTripTime()) {
                continue;
            } else if (Request->MaxRtt() && Request->MaxRtt() < probe->GetRoundTripTime()) {
                continue;
            } else if (probe->IsDead(*seenHosts, *terminatedHosts, *deadHosts)) {
                continue;
            }
            TSwitchPairKey key(probe->GetTargetIface().GetSwitch(), probe->GetSourceIface().GetSwitch());
            groupedProbes[key].emplace_back(std::move(probe));
            found++;
        }

        size_t probeExists(found);
        while (probeExists && (!Request->Limit() || resultProbes.size() < Request->Limit())) {
            for (auto& pair : groupedProbes) {
                if (!pair.second.empty()) {
                    resultProbes.emplace_back(pair.second.back()->ToProto(Builder));
                    pair.second.pop_back();
                    probeExists--;
                }
            }
        }

        Builder.Finish(NApi::CreateTProbesResponse(
            Builder,
            Builder.CreateVector(resultProbes),
            probeList.size(),
            found
        ));
    }

    void TSlicerSeenHostsReply::Process() {
        const auto seenHosts(GetServerContext().GetSliceMaintainer().GetPartialSeenHosts());

        std::vector<NCommon::THost> convertedHosts;
        convertedHosts.reserve(seenHosts.size());
        for (auto hostId : seenHosts) {
            convertedHosts.push_back(hostId);
        }

        Builder.Finish(NApi::CreateTSeenHostsResponse(
            Builder,
            Builder.CreateVectorOfStructs(convertedHosts)
        ));
    }

    void TSlicerTerminatedHostsReply::Process() {
        std::vector<NCommon::THost> convertedHosts;

        const auto terminatedHosts(GetServerContext().GetTerminatedHostsMaintainer().GetHosts());
        if (terminatedHosts) {
            convertedHosts.reserve(terminatedHosts->size());
            for (auto host : *terminatedHosts) {
                convertedHosts.push_back(host->GetReducedId());
            }
        }

        Builder.Finish(NApi::CreateTTerminatedHostsResponse(
            Builder,
            Builder.CreateVectorOfStructs(convertedHosts)
        ));
    }

    void TUpdateTerminatedHostsReply::Process() {
        if (Request->Hosts()) {
            GetServerContext().GetTerminatedHostsMaintainer().UpdateHosts(*Request->Hosts());
        }

        Builder.Finish(NApi::CreateTUpdateTerminatedHostsResponse(Builder));
    }

    void TAppendProbesReply::Process() {
        if (!GetServerContext().GetProbeDumper().IsReady()) {
            ythrow yexception() << "too many probes aren't dumped";
        }

        auto& queue_ = GetServerContext().GetProbeQueue();
        const auto& topologyStorage = GetServerContext().GetTopologyStorage();

        TDumperProbeBatch::TRecordVector dumperProbes;
        for (const auto& probe : *Request->Probes()) {
            if (!probe->Network() || !probe->Protocol()) {
                continue;
            }

            const auto sourceIface = topologyStorage.FindHostInterface(*probe->SourceIface());
            const auto targetIface = topologyStorage.FindHostInterface(*probe->TargetIface());
            if (!sourceIface || !targetIface) {
                continue;
            }

            const TIpAddress sourceAddress(*probe->SourceAddress());
            const TIpAddress targetAddress(*probe->TargetAddress());

            TDumperProbe dumperProbe(
                sourceIface,
                targetIface,
                sourceAddress,
                targetAddress,
                probe->SourcePort(),
                probe->TargetPort(),
                static_cast<ENetworkType>(probe->Network()),
                static_cast<EProtocolType>(probe->Protocol()),
                probe->Score(),
                probe->RoundTripTime(),
                probe->Generated()
            );
            dumperProbes.PushBack(TDumperProbeBatch::TRecord::Make(std::move(dumperProbe), dumperProbe.GetGenerated()));
        }
        queue_.EnqueueAll(dumperProbes);

        Builder.Finish(NApi::CreateTAppendProbesResponse(
            Builder,
            dumperProbes.size()
        ));

        INFO_LOG << dumperProbes.size() << " new probes received" << Endl;
    }

    void TSlicerPingReply::Process() {
        if (!GetServerContext().GetExpressionStorage().IsReady()) {
            ythrow yexception() << "expressions aren't ready yet";
        }
        if (!GetServerContext().GetWalleUpdater().IsReady()) {
            ythrow yexception() << "wall-e cache isn't ready yet";
        }
        if (!GetServerContext().GetSeenHostsUpdater().IsReady()) {
            ythrow yexception() << "seen hosts cache isn't ready yet";
        }
        if (!GetServerContext().GetClickhouseMonitor().IsReady()) {
            ythrow yexception() << "clickhouse is broken";
        }
        if (!GetServerContext().GetProbeDumper().IsReady()) {
            ythrow yexception() << "too many probes aren't dumped";
        }
        if (!GetServerContext().GetSliceMaintainer().IsReady()) {
            ythrow yexception() << "slices aren't ready yet";
        }
        if (!GetServerContext().GetSliceDumper().IsReady()) {
            ythrow yexception() << "slice dumper isn't ready yet";
        }
        if (!GetServerContext().GetSliceHistory().IsReady()) {
            ythrow yexception() << "slice history isn't ready yet";
        }

        GetResponse().BeginObject()
            .WriteKey(TStringBuf("status"))
            .WriteString("OK")
            .EndObject();
    }
}
