#include <infra/netmon/api/state_api.h>
#include <infra/netmon/settings.h>

namespace NNetmon {
    namespace {
        template <class TState>
        class TStateFilter {
        public:
            using TRef = THolder<TStateFilter<TState>>;
            using TKey = typename TState::TKey;
            using TTree = typename TState::TTree;
            using TIterator = typename TTree::TConstIterator;

            TStateFilter(const TTree& tree)
                : Tree(tree)
            {
            }

            virtual ~TStateFilter() = default;

            virtual TIterator Begin() {
                return Tree.Begin();
            }

            virtual bool ShouldSkip(const TKey&) {
                return false;
            }

            virtual bool ShouldSkip(const TIterator&) {
                return false;
            }

            virtual bool IsEnd(const TIterator& it) {
                return it == Tree.End();
            }

        protected:
            const TTree& Tree;
        };

        template <class TState, class TKey>
        class TContainsFilter: public TStateFilter<TState> {
        public:
            TContainsFilter(const typename TStateFilter<TState>::TTree& tree, const TKey& key)
                : TStateFilter<TState>(tree)
                , Key(key)
            {
            }

            typename TStateFilter<TState>::TIterator Begin() override {
                return TStateFilter<TState>::Tree.LowerBound(Key);
            }

            bool ShouldSkip(const typename TStateFilter<TState>::TKey& key) override {
                return !Key.Contains(key);
            }

            bool ShouldSkip(const typename TStateFilter<TState>::TIterator& it) override {
                return ShouldSkip(it->GetKey());
            }

            bool IsEnd(const typename TStateFilter<TState>::TIterator& it) override {
                return it == TStateFilter<TState>::Tree.End() || !Key.SameTarget(it->GetKey());
            }

        private:
            const TKey Key;
        };

        template <class TState, class TKey>
        class TEqualFilter: public TStateFilter<TState> {
        public:
            TEqualFilter(const typename TStateFilter<TState>::TTree& tree, const TKey& key)
                : TStateFilter<TState>(tree)
                , Key(key)
            {
            }

            typename TStateFilter<TState>::TIterator Begin() override {
                return TStateFilter<TState>::Tree.LowerBound(Key);
            }

            bool IsEnd(const typename TStateFilter<TState>::TIterator& it) override {
                return it == TStateFilter<TState>::Tree.End() || it->GetKey() != Key;
            }

        private:
            const TKey Key;
        };

        template <class TObjectRef,
                  class TObjectPairState,
                  class TObjectPairIndex,
                  class TAvailabilityMap>
        class TReplyConfig: public TNonCopyable {
        public:
            using TPairState = TObjectPairState;
            using TPairKey = typename TPairState::TKey;
            using TFilter = TStateFilter<TPairState>;

            TReplyConfig(const TRequestData& requestData, typename TObjectPairIndex::TRef index)
                : RequestData(requestData)
                , Index(index)
            {
                if (!Exists()) {
                    ythrow TNotFoundError() << "such aggregate does not exist yet";
                }
            }

            virtual ~TReplyConfig() = default;

            virtual typename TFilter::TRef CreateFilter() const = 0;
            virtual THashSet<TObjectRef> GetPossibleObjects() const = 0;

            virtual bool FillMissedPairs() const {
                return true;
            }

            std::tuple<ui64, ui64> GetProbeCount() const {
                return Index->GetProbeCount();
            }

            TInstant GetUpdateTime() const {
                return Index->GetGenerated();
            }

            TAvailabilityMap GetAvailability() const {
                TAvailabilityMap states;
                Index->GetAvailability().FillMap(states);
                return states;
            }

            TExpressionId GetExpressionId() const {
                return RequestData.GetExpressionId();
            }

            static bool IsPossibleSourceObject(const TObjectRef& object) {
#ifdef NOC_SLA_BUILD
                return AnyOf(TSettings::Get()->GetProbeScheduleDcs(), [&object](const TString& dcName) -> bool {
                    if constexpr (std::is_same_v<TObjectRef, TTopology::TDatacenterRef>) {
                        return object->GetName() == dcName;
                    } else {
                        return object->GetDatacenter().GetName() == dcName;
                    }
                });
#else
                Y_UNUSED(object);
                return true;
#endif
            }

            bool Exists() const {
                return !!Index;
            }

        protected:
            const TRequestData& RequestData;
            const typename TObjectPairIndex::TRef Index;
        };

        class TDatacenterConfig: public TReplyConfig<TTopology::TDatacenterRef,
                                                     TDatacenterPairState,
                                                     TDatacenterPairIndex,
                                                     TNetworkAvailability::TDatacenterMap> {
        public:
            using TReplyConfig<TTopology::TDatacenterRef,
                               TDatacenterPairState,
                               TDatacenterPairIndex,
                               TNetworkAvailability::TDatacenterMap>::TReplyConfig;

            virtual ~TDatacenterConfig() = default;

            TFilter::TRef CreateFilter() const override {
                using TDatacenterFilter = TEqualFilter<TPairState, TDatacenterPairState::TKey>;

                if (RequestData.HasSourceDc() && RequestData.HasTargetDc()) {
                    TDatacenterPairState::TKey key(RequestData.GetDatacenterPair());
                    return MakeHolder<TDatacenterFilter>(Index->GetTree(), key);
                } else {
                    return MakeHolder<TFilter>(Index->GetTree());
                }
            }

            static const TDuration& GetInterval() {
                return TSettings::Get()->GetDcAggregationInterval();
            }

            static const TDuration& GetWindow() {
                return TSettings::Get()->GetDcAggregationWindow();
            }

            TTopologySelector::TDatacenterSet GetPossibleObjects() const override {
                return RequestData.GetTopologySelector()->GetDatacenters(RequestData.GetExpressionId());
            }
        };

        class TLatestDatacenterConfig: public TDatacenterConfig {
        public:
            TLatestDatacenterConfig(const TRequestData& requestData)
                : TDatacenterConfig(requestData, requestData.GetProbeAggregator()->GetDatacenterIndex())
            {
            }
        };

        class TLineConfig: public TReplyConfig<TTopology::TLineRef,
                                               TLinePairState,
                                               TLinePairIndex,
                                               TNetworkAvailability::TLineMap> {
        public:
            using TReplyConfig<TTopology::TLineRef,
                               TLinePairState,
                               TLinePairIndex,
                               TNetworkAvailability::TLineMap>::TReplyConfig;

            virtual ~TLineConfig() = default;

            TFilter::TRef CreateFilter() const override {
                using TDatacenterFilter = TContainsFilter<TPairState, TDatacenterPairState::TKey>;
                using TLineFilter = TEqualFilter<TPairState, TLinePairState::TKey>;

                if (RequestData.HasSourceQueue() && RequestData.HasTargetQueue()) {
                    TLinePairState::TKey key(RequestData.GetLinePair());
                    return MakeHolder<TLineFilter>(Index->GetTree(), key);
                } else if (RequestData.HasSourceDc() && RequestData.HasTargetDc()) {
                    TDatacenterPairState::TKey key(RequestData.GetDatacenterPair());
                    return MakeHolder<TDatacenterFilter>(Index->GetTree(), key);
                } else {
                    return MakeHolder<TFilter>(Index->GetTree());
                }
            }

            static const TDuration& GetInterval() {
                return TSettings::Get()->GetLineAggregationInterval();
            }

            static const TDuration& GetWindow() {
                return TSettings::Get()->GetLineAggregationWindow();
            }

            TTopologySelector::TLineSet GetPossibleObjects() const override {
                return RequestData.GetTopologySelector()->GetLines(RequestData.GetExpressionId());
            }
        };

        class TLatestQueueConfig: public TLineConfig {
        public:
            TLatestQueueConfig(const TRequestData& requestData)
                : TLineConfig(requestData, requestData.GetProbeAggregator()->GetLineIndex())
            {
            }
        };

        class TSwitchConfig: public TReplyConfig<TTopology::TSwitchRef,
                                                 TSwitchPairState,
                                                 TSwitchPairIndex,
                                                 TNetworkAvailability::TSwitchMap> {
        public:
            using TReplyConfig<TTopology::TSwitchRef,
                               TSwitchPairState,
                               TSwitchPairIndex,
                               TNetworkAvailability::TSwitchMap>::TReplyConfig;

            virtual ~TSwitchConfig() = default;

            TFilter::TRef CreateFilter() const override {
                using TDatacenterFilter = TContainsFilter<TPairState, TDatacenterPairState::TKey>;
                using TLineFilter = TContainsFilter<TPairState, TLinePairState::TKey>;
                using TSwitchFilter = TEqualFilter<TPairState, TSwitchPairState::TKey>;

                if (RequestData.HasSourceSwitch() && RequestData.HasTargetSwitch()) {
                    TSwitchPairState::TKey key(RequestData.GetSwitchPair());
                    return MakeHolder<TSwitchFilter>(Index->GetTree(), key);
                } else if (RequestData.HasSourceQueue() && RequestData.HasTargetQueue()) {
                    TLinePairState::TKey key(RequestData.GetLinePair());
                    return MakeHolder<TLineFilter>(Index->GetTree(), key);
                } else if (RequestData.HasSourceDc() && RequestData.HasTargetDc()) {
                    TDatacenterPairState::TKey key(RequestData.GetDatacenterPair());
                    return MakeHolder<TDatacenterFilter>(Index->GetTree(), key);
                } else {
                    return MakeHolder<TFilter>(Index->GetTree());
                }
            }

            bool FillMissedPairs() const override {
                return false;
            }

            static const TDuration& GetInterval() {
                return TSettings::Get()->GetSwitchAggregationInterval();
            }

            static const TDuration& GetWindow() {
                return TSettings::Get()->GetSwitchAggregationWindow();
            }

            TTopologySelector::TSwitchSet GetPossibleObjects() const override {
                return RequestData.GetTopologySelector()->GetSwitches(GetExpressionId());
            }
        };

        class TLatestSwitchConfig: public TSwitchConfig {
        public:
            TLatestSwitchConfig(const TRequestData& requestData)
                : TSwitchConfig(requestData, requestData.GetProbeAggregator()->GetSwitchIndex())
            {
            }
        };

        template <class TState>
        void StateToJson(NJsonWriter::TBuf& jsonWriter, const TState& pairState) {
            jsonWriter
                .BeginObject()
                .WriteKey(TStringBuf("source"))
                .WriteString(pairState.GetSource().GetName())
                .WriteKey(TStringBuf("target"))
                .WriteString(pairState.GetTarget().GetName());

            jsonWriter.WriteKey(TStringBuf("connectivity"));
            pairState.GetConnectivityHistogram().ToJson(jsonWriter);

            jsonWriter.WriteKey(TStringBuf("rtt"));
            pairState.GetRttHistogram().ToJson(jsonWriter);

            jsonWriter
                .WriteKey(TStringBuf("probes")).WriteULongLong(pairState.GetProbeCount())
                .WriteKey(TStringBuf("dead_probes")).WriteULongLong(pairState.GetDeadProbeCount())
                .EndObject();
        }

        template <>
        void StateToJson<TLinePairState>(NJsonWriter::TBuf& jsonWriter, const TLinePairState& pairState) {
            jsonWriter
                .BeginObject()
                .WriteKey(TStringBuf("source"))
                .WriteString(pairState.GetSource().GetName())
                .WriteKey(TStringBuf("source_dc"))
                .WriteString(pairState.GetSource().GetDatacenter().GetName())
                .WriteKey(TStringBuf("target"))
                .WriteString(pairState.GetTarget().GetName())
                .WriteKey(TStringBuf("target_dc"))
                .WriteString(pairState.GetTarget().GetDatacenter().GetName());

            jsonWriter.WriteKey(TStringBuf("connectivity"));
            pairState.GetConnectivityHistogram().ToJson(jsonWriter);

            jsonWriter.WriteKey(TStringBuf("rtt"));
            pairState.GetRttHistogram().ToJson(jsonWriter);

            jsonWriter
                .WriteKey(TStringBuf("probes")).WriteULongLong(pairState.GetProbeCount())
                .WriteKey(TStringBuf("dead_probes")).WriteULongLong(pairState.GetDeadProbeCount())
                .EndObject();
        }

        template <class TConfig>
        void IndexToJson(NJsonWriter::TBuf& jsonWriter, const TConfig& config) {
            jsonWriter
                .BeginObject()
                .WriteKey(TStringBuf("state"))
                .BeginList();

            THashSet<typename TConfig::TPairKey> pairs;

            auto filter(config.CreateFilter());
            for (auto it(filter->Begin()); !filter->IsEnd(it); ++it) {
                if (!filter->ShouldSkip(it)) {
                    StateToJson(jsonWriter, *it);

                    if (config.FillMissedPairs()) {
                        pairs.emplace(it->GetKey());
                    }
                }
            }

            if (config.FillMissedPairs()) {
                const auto expressionId(config.GetExpressionId());
                const auto possibleObjects(config.GetPossibleObjects());

                for (const auto source : possibleObjects) {
                    if (!config.IsPossibleSourceObject(source)) {
                        continue;
                    }
                    for (const auto target : possibleObjects) {
                        typename TConfig::TPairKey key(target, source);
                        if (!filter->ShouldSkip(key) && !pairs.contains(key)) {
                            const auto emptyState(TConfig::TPairState::Make(target, source, expressionId));
                            StateToJson(jsonWriter, *emptyState);
                        }
                    }
                }
            }

            ui64 totalProbes, deadProbes;
            std::tie(totalProbes, deadProbes) = config.GetProbeCount();
            jsonWriter
                .EndList()
                .WriteKey(TStringBuf("interval"))
                .WriteULongLong(config.GetInterval().Seconds())
                .WriteKey(TStringBuf("window"))
                .WriteULongLong(config.GetWindow().Seconds())
                .WriteKey(TStringBuf("generated"))
                .WriteULongLong(config.GetUpdateTime().Seconds())
                .WriteKey(TStringBuf("probes"))
                .WriteULongLong(totalProbes)
                .WriteKey(TStringBuf("dead_probes"))
                .WriteULongLong(deadProbes)
                .WriteKey(TStringBuf("availability"))
                .BeginList();

            for (const auto& pair : config.GetAvailability()) {
                auto alive(pair.second.Alive.GetValue());
                if (alive.Defined()) {
                    jsonWriter
                        .BeginObject()
                        .WriteKey(TStringBuf("name"))
                        .WriteString(pair.first->GetName())
                        .WriteKey(TStringBuf("metric"))
                        .WriteDouble(alive.GetRef())
                        .EndObject();
                }
            }

            jsonWriter
                .EndList()
                .EndObject();
        }

        template <class TConfig>
        void LatestStateToJson(NJsonWriter::TBuf& jsonWriter, const TRequestData& requestData, const TAggregatorContext& context) {
            if (!context.GetSliceCollector().IsReady()) {
                ythrow yexception() << "slices aren't collected yet";
            }
            if (!context.GetProbeDumper().IsReady()) {
                ythrow yexception() << "too many probes aren't dumped";
            }

            TConfig config(requestData);
            if (config.GetUpdateTime() + config.GetInterval() * 3 < TInstant::Now()) {
                ythrow yexception() << "that aggregate is so old";
            }

            IndexToJson(jsonWriter, config);
        }

    } // end namespace

    // api/v1/dc/state/latest handler

    void TDcLatestStateReply::Process() {
        LatestStateToJson<TLatestDatacenterConfig>(GetResponse(), GetRequestData(), GetServerContext());
    }

    // api/v1/dc/state/history handler

    TThreadPool::TFuture TDcStateHistoryReply::Process() {
        return GetServerContext().GetHistoryRequester().CollectDatacenterHistoryState(
            GetRequestData().GetGenerated(),
            GetRequestData().GetAggregatorKey()
        ).Apply(
            [this] (const NThreading::TFuture<TDatacenterPairIndex::TRef>& future) {
                TDatacenterConfig config(GetRequestData(), future.GetValue());
                IndexToJson(GetResponse(), config);
            }
        );
    }

    // api/v1/queue/state/latest handler

    void TLineLatestStateReply::Process() {
        LatestStateToJson<TLatestQueueConfig>(GetResponse(), GetRequestData(), GetServerContext());
    }

    // api/v1/queue/state/history handler

    TThreadPool::TFuture TLineStateHistoryReply::Process() {
        return GetServerContext().GetHistoryRequester().CollectLineHistoryState(
            GetRequestData().GetGenerated(),
            GetRequestData().GetAggregatorKey(),
            GetRequestData().GetDatacenterPair(false)
        ).Apply(
            [this] (const NThreading::TFuture<TLinePairIndex::TRef>& future) {
                TLineConfig config(GetRequestData(), future.GetValue());
                IndexToJson(GetResponse(), config);
            }
        );
    }

    // api/v1/switch/state/latest handler

    void TSwitchLatestStateReply::Process() {
        LatestStateToJson<TLatestSwitchConfig>(GetResponse(), GetRequestData(), GetServerContext());
    }

    // api/v1/switch/state/history handler

    TThreadPool::TFuture TSwitchStateHistoryReply::Process() {
        return GetServerContext().GetHistoryRequester().CollectSwitchHistoryState(
            GetRequestData().GetGenerated(),
            GetRequestData().GetAggregatorKey(),
            GetRequestData().GetLinePair()
        ).Apply(
            [this] (const NThreading::TFuture<TSwitchPairIndex::TRef>& future) {
                TSwitchConfig config(GetRequestData(), future.GetValue());
                IndexToJson(GetResponse(), config);
            }
        );
    }
}
