#include <infra/netmon/api/tasks_api.h>
#include <infra/netmon/settings.h>

#include <util/string/join.h>

namespace NNetmon {
    namespace {
        const ui64 MICROSECONDS_IN_SECOND = 1000000UL;
        const size_t MAX_TASKS_ON_HOST = 3;

        template <class T>
        inline TString Dump(const google::protobuf::io::TAsJSON<T>& obj) {
            TString buf;
            TStringOutput stream(buf);
            stream << obj;
            stream.Finish();
            return buf;
        }

        bool HasPermissions(const THost& host, const TUserState& userState, const TStaffStorage& staff) {
            if (staff.ExpandLoginsAndGroups(TSettings::Get()->GetSudoers()).contains(userState.Login)) {
                return true;
            } else if (staff.ExpandLoginsAndGroups(TSettings::Get()->GetTasksUsers()).contains(userState.Login)) {
                return true;
            } else if (host.GetOwners().contains(userState.Login)) {
                return true;
            } else {
                return false;
            }
        }
    }

    NClient::TEnqueuedTask TEnqueueTaskReply::CreateTask(const NJson::TJsonValue& request) {
        auto now(TInstant::Now());

        if (!request.Has("host")) {
            ythrow TValidationError() << "no host given";
        }
        if (!request.Has("type")) {
            ythrow TValidationError() << "no type given";
        }
        if (!request.Has("arguments")) {
            ythrow TValidationError() << "no arguments given";
        }

        auto iface(GetServerContext().GetTopologyStorage().FindHostInterface(
            request["host"].GetStringSafe()
        ));
        if (!iface) {
            ythrow TNotFoundError() << "such host or interface not exists";
        }

        auto host(iface.GetHost());
        if (!HasPermissions(*host, GetUserState(), GetServerContext().GetStaffStorage())) {
            ythrow TForbiddenError() << "user can't create tasks on " << host->GetName()
                                     << ", ask: " << JoinSeq(TStringBuf(", "), host->GetOwners());
        }

        auto seenHosts(GetServerContext().GetSeenHostsUpdater().GetHosts());
        if (seenHosts->find(host).IsEnd()) {
            ythrow TNotFoundError() << "given host is dead";
        }

        if (GetServerContext().GetEnqueuedTaskIndex().FindByHost(host->GetName()).size() > MAX_TASKS_ON_HOST) {
            ythrow TTooManyRequestsError() << "too many tasks on host";
        }

        NClient::TEnqueuedTask task;
        task.SetHost(host->GetName());
        task.SetGenerated(now.MicroSeconds());
        task.SetDeadline((now + TDuration::Hours(24)).MicroSeconds());

        NJsonWriter::TBuf buffer;
        buffer.WriteJsonValue(&request["arguments"]);

        if (request["type"].GetStringSafe() == TStringBuf("diagnostic")) {
            auto status(google::protobuf::util::JsonStringToMessage(buffer.Str(), task.MutableDiagnostic()));
            if (!status.ok()) {
                ythrow TValidationError() << status.message().ToString();
            }
        } else {
            ythrow TValidationError() << "unknown type given";
        }

        return task;
    }

    NThreading::TFuture<void> TEnqueueTaskReply::SpawnTask(const NClient::TEnqueuedTask& task) {
        TEnqueuedTask agentTask(task);

        GetResponse()
            .BeginObject()
            .WriteKey("key")
            .WriteString(GetGuidAsString(agentTask.Key()))
            .WriteKey("task")
            .UnsafeWriteValue(Dump(task.AsJSON()))
            .EndObject();

        return GetServerContext().GetEnqueuedTaskStorage().Put(agentTask).Apply(
            [] (const NThreading::TFuture<TAgentTaskStorage::EResultCode>& future) {
                switch (future.GetValue()) {
                    case TAgentTaskStorage::OK: {
                        return;
                    }
                    case TAgentTaskStorage::FAILED: {
                        ythrow yexception() << "failed to create task";
                    }
                    case TAgentTaskStorage::TOO_MANY_REQUESTS: {
                        ythrow TTooManyRequestsError() << "too many tasks enqueued";
                    }
                }
            }
        );
    }

    NThreading::TFuture<void> TEnqueueTaskReply::Process() {
        if (!GetUserState().IsActive) {
            ythrow TUnauthorizedError() << "unknown user can't create tasks";
        }

        if (!GetRequest().Has("tasks")) {
            ythrow TValidationError() << "no tasks given";
        }

        TVector<NClient::TEnqueuedTask> tasks;
        for (const auto& task : GetRequest()["tasks"].GetArraySafe()) {
            tasks.emplace_back(CreateTask(task));
        }

        GetResponse()
            .BeginObject()
            .WriteKey("tasks")
            .BeginList();

        TVector<NThreading::TFuture<void>> futures;
        for (const auto& task : tasks) {
            futures.emplace_back(SpawnTask(task));
        }

        GetResponse()
            .EndList()
            .EndObject();

        return NThreading::WaitExceptionOrAll(futures);
    }

    bool TFetchTaskReply::FetchTask(const NJson::TJsonValue& request) {
        const auto& index(GetServerContext().GetFinishedTaskIndex());

        auto parentKey(GetGuid(request["parent_key"].GetStringSafe()));
        NClient::TTaskResult result;
        if (!index.FindByParentKey(parentKey, result)) {
            GetResponse()
                .BeginObject()
                .WriteKey("error")
                .WriteString("task with such parent key not found")
                .EndObject();
            return false;
        }

        GetResponse()
            .BeginObject()
            .WriteKey("generated")
            .WriteULongLong(result.GetGenerated() / MICROSECONDS_IN_SECOND)
            .WriteKey("finished")
            .WriteBool(result.GetFinished());

        if (result.GetError().empty()) {
            GetResponse()
                .WriteKey("error")
                .WriteNull();
        } else {
            GetResponse()
                .WriteKey("error")
                .WriteString(result.GetError());
        }

        switch (result.GetResponseCase()) {
            case NClient::TTaskResult::kVersion: {
                GetResponse()
                    .WriteKey("type")
                    .WriteString(TStringBuf("version"))
                    .WriteKey("result")
                    .UnsafeWriteValue(Dump(result.GetVersion().AsJSON()));
                break;
            }
            case NClient::TTaskResult::kDiagnostic: {
                GetResponse()
                    .WriteKey("type")
                    .WriteString(TStringBuf("diagnostic"))
                    .WriteKey("result")
                    .UnsafeWriteValue(Dump(result.GetDiagnostic().AsJSON()));
                break;
            }
            default: {
                break;
            }
        }

        GetResponse()
            .EndObject();

        return true;
    }

    NThreading::TFuture<void> TFetchTaskReply::Process() {
        if (!GetRequest().Has("tasks")) {
            ythrow TValidationError() << "no tasks given";
        }

        GetResponse()
            .BeginObject()
            .WriteKey("tasks")
            .BeginList();

        bool ready = true;
        for (const auto& task : GetRequest()["tasks"].GetArraySafe()) {
            ready &= FetchTask(task);
        }

        GetResponse()
            .EndList()
            .WriteKey("ready")
            .WriteBool(ready)
            .EndObject();

        return NThreading::MakeFuture();
    }
}
