# -*- coding: utf-8 -*-

from __future__ import absolute_import

import atexit
import contextlib
import os
import socket
import threading
import weakref

import tornado.httpclient
from tornado.platform.select import SelectIOLoop


def join(url, *args):
    for part in args:
        url = "{0}/{1}".format(url.rstrip("/"), part.lstrip("/"))
    return url


class HttpClientStorage(object):
    """Store a loop for each thread, don't create them every time."""

    _client_key = "_http_client"

    class Client(object):

        __slots__ = ("io_loop", "http_client", "used", "_current_pid", "__weakref__")

        def __init__(self):
            self.io_loop = None
            self.http_client = None
            self.used = None
            self._current_pid = os.getpid()
            self.reset()

        def destroy(self):
            if self.http_client is not None:
                self.http_client.close()
                self.http_client = None
            if self.io_loop is not None:
                self.io_loop.close(all_fds=True)
                self.io_loop = None
            self.used = False

        def reset(self):
            self.destroy()
            self.io_loop = SelectIOLoop(make_current=False)
            self.http_client = tornado.httpclient.AsyncHTTPClient(
                self.io_loop, force_instance=True, max_body_size=500000000
            )

        def get(self):
            assert not self.used and self.io_loop is not None, "client already used or deleted"
            current_pid = os.getpid()
            if self._current_pid != current_pid:
                self._current_pid = current_pid
                self.reset()
            return self

        def __del__(self):
            self.destroy()

    def __init__(self):
        self._local_storage = threading.local()
        self._clients = weakref.WeakSet()
        atexit.register(self._destroy)

    def _destroy(self):
        for client in list(self._clients):
            client.destroy()

    @property
    def _current_client(self):
        d = self._local_storage.__dict__
        if self._client_key in d:
            return d[self._client_key]
        else:
            # client will be collected by GC when threads, owning it, terminates
            client = d[self._client_key] = self.Client()
            self._clients.add(client)
            return client

    @contextlib.contextmanager
    def acquire(self):
        client = self._current_client.get()
        client.used = True
        try:
            yield (client.io_loop, client.http_client)
        finally:
            client.used = False


_http_client_storage = HttpClientStorage()


def acquireHttpClient():
    return _http_client_storage.acquire()


def prepareRequestKwargs(timeout=None, validate_cert=False, **kwargs):
    timeout = socket.getdefaulttimeout() if timeout is None else timeout
    kwargs.setdefault("user_agent", "netmon_build_topology")
    kwargs.setdefault("decompress_response", True)
    kwargs.setdefault("connect_timeout", timeout)
    kwargs.setdefault("request_timeout", timeout)
    kwargs.setdefault("validate_cert", validate_cert)
    kwargs.setdefault("allow_nonstandard_methods", True)
    if kwargs.get("body") is not None and kwargs.get("method") is None:
        kwargs["method"] = "POST"
    return kwargs


def makeRequest(url, timeout=None, validate_cert=False, **kwargs):
    kwargs = prepareRequestKwargs(timeout, validate_cert, **kwargs)
    with acquireHttpClient() as (io_loop, http_client):
        return io_loop.run_sync(lambda: http_client.fetch(url, **kwargs))


def httpDate(dt):
    return dt.strftime('%a, %d %b %Y %H:%M:%S GMT')
