#pragma once

#include <infra/netmon/probe_slice_merger.h>
#include <infra/netmon/probe.h>
#include <infra/netmon/seen_hosts.h>
#include <infra/netmon/terminated_hosts.h>
#include <infra/netmon/topology/clients/walle.h>
#include <infra/netmon/topology/common_ut.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NNetmon;

template <class T>
inline void WaitForFuture(T future) {
    future.GetValue(TDuration::Max());
}

inline const TIpAddress& GetEmptyIpAddress() {
    return Default<TIpAddress>();
}

inline TTopologyStorage& GetTopologyStorage() {
    return TGlobalTopology::GetTopologyStorage();
}

inline TTopologySelector::TBox::TConstValueRef GetTopologySelector() {
    return GetTopologyStorage().GetTopologySelector();
}

inline TTopology::THostRef FindHost(const TString& name) {
    auto host = GetTopologyStorage().FindHost(name);
    UNIT_ASSERT_C(host, name);
    return host;
}

inline TTopology::TDatacenterRef FindDatacenter(const TString& name, TTopologyStorage* topologyStorage=nullptr) {
    if (!topologyStorage) {
        topologyStorage = &GetTopologyStorage();
    }

    auto dc = topologyStorage->FindDatacenter(name);
    UNIT_ASSERT_C(dc, name);
    return dc;
}

inline TProbe::TRef CreateProbe(const TString& source, const TString& target, TInstant created, TTopologyStorage* topologyStorage=nullptr) {
    if (!topologyStorage) {
        topologyStorage = &GetTopologyStorage();
    }

    auto sourceIface = topologyStorage->FindHostInterface(source);
    auto targetIface = topologyStorage->FindHostInterface(target);
    UNIT_ASSERT_C(sourceIface, source);
    UNIT_ASSERT_C(targetIface, source);
    return TProbe::Make(
        sourceIface, targetIface,
        GetEmptyIpAddress(), GetEmptyIpAddress(), 0, 0,
        created, 0.0, -1.0
    );
}

inline TProbe::TRef CreateProbe(const TString& source, const TString& target) {
    return CreateProbe(source, target, TInstant::Now());
}

class TFakeSeenHostsUpdater: public IHostsMaintainer {
public:
    TFakeSeenHostsUpdater()
    {
        auto hosts(Hosts.Own());
        hosts->emplace(FindHost("jmon-test.search.yandex.net"));
        hosts->emplace(FindHost("man1-0015.search.yandex.net"));
        hosts->emplace(FindHost("sas1-1234.search.yandex.net"));
    }

    TTopologyStorage::THostSetBox::TConstValueRef GetHosts() const override {
        return Hosts.Get();
    }

private:
    TTopologyStorage::THostSetBox Hosts;
};

class TFakeTerminatedHostsMaintainer: public IHostsMaintainer {
public:
    TTopologyStorage::THostSetBox::TConstValueRef GetHosts() const override {
        return Hosts.Get();
    }

private:
    TTopologyStorage::THostSetBox Hosts;
};

class TFakeWalleUpdater: public IHostsMaintainer {
public:
    TTopologyStorage::THostSetBox::TConstValueRef GetHosts() const override {
        return Hosts.Get();
    }

private:
    TTopologyStorage::THostSetBox Hosts;
};

inline const TFakeSeenHostsUpdater& GetSeenHostsUpdater() {
    return Default<TFakeSeenHostsUpdater>();
}

inline const TFakeTerminatedHostsMaintainer& GetTerminatedHostsMaintainer() {
    return Default<TFakeTerminatedHostsMaintainer>();
}

inline const TFakeWalleUpdater& GetWalleUpdater() {
    return Default<TFakeWalleUpdater>();
}

class TFakeProbeSliceMerger: public IProbeSliceMerger, public TNonCopyable {
public:
    TFakeProbeSliceMerger()
        : Key(
            GetTopologyStorage().DefaultExpressionId(),
            BACKBONE6,
            UDP
        )
        , SwitchEventHub(TEventHub<TSwitchPairIndex::TRef>::Make())
        , LineEventHub(TEventHub<TLinePairIndex::TRef>::Make())
        , DatacenterEventHub(TEventHub<TDatacenterPairIndex::TRef>::Make())
        , Timestamp(TInstant::Zero())
    {
    }

    TSwitchPairIndex::TRef GetSwitchLevelIndex(const TProbeAggregatorKey& key) const override {
        Y_VERIFY(Key == key);
        auto probe(CreateProbe("jmon-test.search.yandex.net", "man1-0015.search.yandex.net"));
        auto index(TSwitchPairIndex::Make(Key, Timestamp));
        index->AppendDeadProbe(*probe);
        index->AppendAliveProbe(*probe, Timestamp);
        return index;
    }

    TLinePairIndex::TRef GetLineLevelIndex(const TProbeAggregatorKey& key) const override {
        auto selector(GetTopologySelector());
        auto index(GetSwitchLevelIndex(key));
        return TLinePairIndex::Make(*selector, *index, Key);
    }

    TDatacenterPairIndex::TRef GetDatacenterLevelIndex(const TProbeAggregatorKey& key) const override {
        auto selector(GetTopologySelector());
        auto index(GetSwitchLevelIndex(key));
        return TDatacenterPairIndex::Make(*selector, *index, Key);
    }

    TProbeAggregatorKeySet GetSwitchLevelKeys() const override {
        return {Key};
    }

    TProbeAggregatorKeySet GetLineLevelKeys() const override {
        return {Key};
    }

    TProbeAggregatorKeySet GetDatacenterLevelKeys() const override {
        return {Key};
    }

    const TEventHub<TSwitchPairIndex::TRef>& OnSwitchUpdated() const override {
        return *SwitchEventHub;
    }

    const TEventHub<TLinePairIndex::TRef>& OnLineUpdated() const override {
        return *LineEventHub;
    }

    const TEventHub<TDatacenterPairIndex::TRef>& OnDatacenterUpdated() const override {
        return *DatacenterEventHub;
    }

    void NotifySwitch() const {
        SwitchEventHub->Notify(GetSwitchLevelIndex(Key));
    }

    void NotifyLine() const {
        LineEventHub->Notify(GetLineLevelIndex(Key));
    }

    void NotifyDatacenter() const {
        DatacenterEventHub->Notify(GetDatacenterLevelIndex(Key));
    }

    void SetTimestamp(const TInstant& timestamp) {
        Timestamp = timestamp;
    }

    const TProbeAggregatorKey& GetKey() const {
        return Key;
    }

private:
    const TProbeAggregatorKey Key;
    TEventHub<TSwitchPairIndex::TRef>::TRef SwitchEventHub;
    TEventHub<TLinePairIndex::TRef>::TRef LineEventHub;
    TEventHub<TDatacenterPairIndex::TRef>::TRef DatacenterEventHub;
    TInstant Timestamp;
};

