#pragma once

#include <library/cpp/deprecated/atomic/atomic.h>

#include <util/generic/string.h>
#include <util/generic/maybe.h>
#include <util/generic/vector.h>
#include <util/generic/singleton.h>
#include <util/digest/multi.h>

namespace NNetmon {
    namespace {
        template <typename TOne>
        inline size_t VectorHash(size_t acc, const TVector<TOne>& one) noexcept {
            if (one.empty()) {
                return 0;
            }
            acc = 1UL << acc;
            for (const auto& element : one) {
                acc = CombineHashes(acc, THash<TOne>()(element));
            }
            return acc;
        }

        template <typename TOne>
        inline size_t VectorHash(size_t acc, const TMaybe<TOne>& one) noexcept {
            if (one.Empty()) {
                return 0;
            }
            return CombineHashes(1UL << acc, THash<TOne>()(one.GetRef()));
        }

        template <typename THead, typename... TTail>
        size_t VectorHash(size_t acc, const THead& head, TTail&&... tail) noexcept {
            size_t first = VectorHash(acc + 1, tail...);
            size_t second = VectorHash(acc, head);
            if (first && second) {
                return CombineHashes(first, second);
            } else if (first) {
                return first;
            } else {
                return second;
            }
        }
    }

    class TExpressionClause {
    public:
        enum EDirection {
            Undefined = 0,
            Source,
            Target,
            Both
        };

#define DECLARE_GETTER(name, type) \
        inline const type& name() const noexcept { \
            return name##_; \
        } \
        inline type& name() noexcept { \
            return name##_; \
        }

#define DECLARE_TWO_GETTERS(name, type) \
        DECLARE_GETTER(Positive##name, type) \
        DECLARE_GETTER(Negative##name, type);

        using TVectorType = TVector<TExpressionClause>;

        using TOptionalBooleans = TVector<bool>;
        using TOptionalIntergers = TVector<int>;
        using TOptionalStrings = TVector<TString>;
        using TOptionalDirection = TMaybe<EDirection>;

        DECLARE_TWO_GETTERS(VirtualFilter, TOptionalBooleans);
        DECLARE_TWO_GETTERS(VlanFilter, TOptionalIntergers);
        DECLARE_TWO_GETTERS(VrfFilter, TOptionalStrings);
        DECLARE_TWO_GETTERS(GroupFilter, TOptionalStrings);
        DECLARE_TWO_GETTERS(DatacenterFilter, TOptionalStrings);
        DECLARE_TWO_GETTERS(QueueFilter, TOptionalStrings);
        DECLARE_TWO_GETTERS(SwitchFilter, TOptionalStrings);
        DECLARE_TWO_GETTERS(NannyFilter, TOptionalStrings);
        DECLARE_TWO_GETTERS(GencfgFilter, TOptionalStrings);
        DECLARE_TWO_GETTERS(WalleProjectFilter, TOptionalStrings);
        DECLARE_TWO_GETTERS(WalleTagFilter, TOptionalStrings);
        DECLARE_GETTER(DirectionFilter, TOptionalDirection);

#undef DECLARE_TWO_GETTERS
#undef DECLARE_GETTER

        inline std::size_t hash() const noexcept {
            return VectorHash(
                0,

                PositiveVirtualFilter_,
                NegativeVirtualFilter_,

                PositiveVlanFilter_,
                NegativeVlanFilter_,

                PositiveVrfFilter_,
                NegativeVrfFilter_,

                PositiveGroupFilter_,
                NegativeGroupFilter_,

                PositiveDatacenterFilter_,
                NegativeDatacenterFilter_,

                PositiveQueueFilter_,
                NegativeQueueFilter_,

                PositiveSwitchFilter_,
                NegativeSwitchFilter_,

                PositiveNannyFilter_,
                NegativeNannyFilter_,

                PositiveGencfgFilter_,
                NegativeGencfgFilter_,

                PositiveWalleProjectFilter_,
                NegativeWalleProjectFilter_,

                PositiveWalleTagFilter_,
                NegativeWalleTagFilter_,

                DirectionFilter_
            );
        }

    private:
        TOptionalBooleans PositiveVirtualFilter_;
        TOptionalBooleans NegativeVirtualFilter_;

        TOptionalIntergers PositiveVlanFilter_;
        TOptionalIntergers NegativeVlanFilter_;

        TOptionalStrings PositiveVrfFilter_;
        TOptionalStrings NegativeVrfFilter_;

        TOptionalStrings PositiveGroupFilter_;
        TOptionalStrings NegativeGroupFilter_;

        TOptionalStrings PositiveDatacenterFilter_;
        TOptionalStrings NegativeDatacenterFilter_;

        TOptionalStrings PositiveQueueFilter_;
        TOptionalStrings NegativeQueueFilter_;

        TOptionalStrings PositiveSwitchFilter_;
        TOptionalStrings NegativeSwitchFilter_;

        TOptionalStrings PositiveNannyFilter_;
        TOptionalStrings NegativeNannyFilter_;

        TOptionalStrings PositiveGencfgFilter_;
        TOptionalStrings NegativeGencfgFilter_;

        TOptionalStrings PositiveWalleProjectFilter_;
        TOptionalStrings NegativeWalleProjectFilter_;

        TOptionalStrings PositiveWalleTagFilter_;
        TOptionalStrings NegativeWalleTagFilter_;

        TOptionalDirection DirectionFilter_;
    };

    class TExpressionDnf {
    public:
        TExpressionDnf()
        {
        }

        TExpressionDnf(const TExpressionDnf& other)
            : Clauses(other.Clauses)
            , CachedHash(other.CachedHash)
        {
        }

        TExpressionDnf(std::initializer_list<TExpressionClause> clauses)
            : Clauses(clauses)
        {
        }

        inline TExpressionClause::TVectorType::const_iterator begin() const noexcept {
            return Clauses.begin();
        }
        inline TExpressionClause::TVectorType::const_iterator end() const noexcept {
            return Clauses.end();
        }

        inline void PushBack(const TExpressionClause& clause) noexcept {
            Clauses.push_back(clause);
            AtomicSet(CachedHash, 0);
        }

        inline std::size_t hash() const noexcept {
            std::size_t result = AtomicGet(CachedHash);
            if (result) {
                return result;
            } else {
                for (const auto& clause : Clauses) {
                    result = CombineHashes(result, clause.hash());
                }
                AtomicSet(CachedHash, result);
                return result;
            }
        }

        inline bool operator==(const TExpressionDnf& rhs) const {
            return hash() == rhs.hash();
        }

    private:
        TExpressionClause::TVectorType Clauses;
        mutable TAtomic CachedHash = 0;
    };
}

template <>
class THash<NNetmon::TExpressionDnf> {
public:
    inline size_t operator()(const NNetmon::TExpressionDnf& value) const {
        return value.hash();
    }
};

template <>
class TEqualTo<NNetmon::TExpressionDnf> {
public:
    inline bool operator()(const NNetmon::TExpressionDnf& lhs, const NNetmon::TExpressionDnf& rhs) const {
        return lhs == rhs;
    }
};
