#pragma once

#include "clause.h"

#include <library/cpp/any/any.h>

#include <library/cpp/containers/intrusive_avl_tree/avltree.h>
#include <util/memory/pool.h>

namespace NNetmon {
    namespace {
        template <typename T>
        class TMakeMixin {
        public:
            using TRef = THolder<T>;

            template <typename... Args>
            static inline TRef Make(TMemoryPool& pool, Args&&... args) {
                return TRef(new (pool) T(std::forward<Args>(args)...));
            }
        };

        struct TLessCompare {
            template <class T>
            static inline bool Compare(const T& l, const T& r) {
                return l < r;
            }
        };

        template <class T, class C>
        class TAvlTreeWithAutoDelete : public TAvlTree<T, C> {
        public:
            using TItem = T;

            ~TAvlTreeWithAutoDelete() {
                for (auto it = this->Begin(); it != this->End();) {
                    typename T::TRef node(&(*it++));
                }
            }

            inline T* InsertHolder(typename T::TRef holder) noexcept {
                return this->Insert(holder.Release());
            }

            inline T* AddHolder(typename T::TRef holder) noexcept {
                if (this->Find(holder.Get()) == nullptr) {
                    return this->Insert(holder.Release());
                } else {
                    return nullptr;
                }
            }
        };

        template <class TTree>
        inline void MergeTrees(TTree& output, TTree& left, TTree& right) {
            using TItem = typename TTree::TItem;
            for (auto it = left.Begin(); it != left.End();) {
                typename TItem::TRef item(&(*(it++)));
                output.InsertHolder(std::move(item));
            }
            for (auto it = right.Begin(); it != right.End();) {
                typename TItem::TRef item(&(*(it++)));
                output.AddHolder(std::move(item));
            }
        }

        template <class TTree>
        inline void CopyTree(TMemoryPool& pool, TTree& output, const TTree& other) {
            for (auto it = other.Begin(); it != other.End(); ++it) {
                output.InsertHolder(it->Copy(pool));
            }
        }
    }

    class TLiteral : public TMakeMixin<TLiteral>, public TAvlTreeItem<TLiteral, TLessCompare>, public TPoolable {
    public:
        using TTree = TAvlTreeWithAutoDelete<TLiteral, TLessCompare>;

        enum ECondition {
            Undefined = 0,
            Group,
            Nanny,
            Gencfg,
            WalleTag,
            WalleProject,
            VLAN,
            VRF,
            Virtual,
            Datacenter,
            Queue,
            Switch,
            Direction
        };

        template <class T>
        inline explicit TLiteral(ECondition type_, const T& value)
            : Type_(type_)
            , Value_(value)
        {
        }

        inline TRef Copy(TMemoryPool& pool) const noexcept {
            return Make(pool, Type_, Value_);
        }

        inline void FillExpressionClause(TExpressionClause& clause, bool positive) const {
            switch (Type_) {
                case ECondition::Virtual: {
                    return Append(positive ? clause.PositiveVirtualFilter() : clause.NegativeVirtualFilter());
                }
                case ECondition::VLAN: {
                    return Append(positive ? clause.PositiveVlanFilter() : clause.NegativeVlanFilter());
                }
                case ECondition::VRF: {
                    return Append(positive ? clause.PositiveVrfFilter() : clause.NegativeVrfFilter());
                }
                case ECondition::Group: {
                    return Append(positive ? clause.PositiveGroupFilter() : clause.NegativeGroupFilter());
                }
                case ECondition::Nanny: {
                    return Append(positive ? clause.PositiveNannyFilter() : clause.NegativeNannyFilter());
                }
                case ECondition::Gencfg: {
                    return Append(positive ? clause.PositiveGencfgFilter() : clause.NegativeGencfgFilter());
                }
                case ECondition::WalleProject: {
                    return Append(positive ? clause.PositiveWalleProjectFilter() : clause.NegativeWalleProjectFilter());
                }
                case ECondition::WalleTag: {
                    return Append(positive ? clause.PositiveWalleTagFilter() : clause.NegativeWalleTagFilter());
                }
                case ECondition::Datacenter: {
                    return Append(positive ? clause.PositiveDatacenterFilter() : clause.NegativeDatacenterFilter());
                }
                case ECondition::Queue: {
                    return Append(positive ? clause.PositiveQueueFilter() : clause.NegativeQueueFilter());
                }
                case ECondition::Switch: {
                    return Append(positive ? clause.PositiveSwitchFilter() : clause.NegativeSwitchFilter());
                }
                case ECondition::Direction: {
                    if (!positive) {
                        ythrow yexception() << "direction can't be negative";
                    }
                    const auto direction(Value_.Cast<TExpressionClause::EDirection>());
                    if (!clause.DirectionFilter().Empty() && clause.DirectionFilter().GetRef() != direction) {
                        ythrow yexception() << "directions can't be mixed";
                    }
                    clause.DirectionFilter() = direction;
                    return;
                }
                case ECondition::Undefined: {
                    ythrow yexception() << "can't convert literal to clause";
                    return;
                }
            }
        }

        void Out(IOutputStream& stream) const noexcept {
            switch (Type_) {
                case ECondition::Virtual: {
                    stream << "(virtual " << Value_.Cast<bool>() << ")";
                    break;
                }
                case ECondition::VLAN: {
                    stream << "(vlan " << Value_.Cast<int>() << ")";
                    break;
                }
                case ECondition::VRF: {
                    stream << "(vrf " << Value_.Cast<TString>() << ")";
                    break;
                }
                case ECondition::Group: {
                    stream << "(group " << Value_.Cast<TString>() << ")";
                    break;
                }
                case ECondition::Nanny: {
                    stream << "(nanny " << Value_.Cast<TString>() << ")";
                    break;
                }
                case ECondition::Gencfg: {
                    stream << "(gencfg " << Value_.Cast<TString>() << ")";
                    break;
                }
                case ECondition::WalleProject: {
                    stream << "(walle-project " << Value_.Cast<TString>() << ")";
                    break;
                }
                case ECondition::WalleTag: {
                    stream << "(walle-tag " << Value_.Cast<TString>() << ")";
                    break;
                }
                case ECondition::Datacenter: {
                    stream << "(datacenter " << Value_.Cast<TString>() << ")";
                    break;
                }
                case ECondition::Queue: {
                    stream << "(queue " << Value_.Cast<TString>() << ")";
                    break;
                }
                case ECondition::Switch: {
                    stream << "(switch " << Value_.Cast<TString>() << ")";
                    break;
                }
                case ECondition::Direction: {
                    stream << "(direction " << Value_.Cast<TExpressionClause::EDirection>() << ")";
                    break;
                }
                case ECondition::Undefined: {
                    stream << "(undefined)";
                    break;
                }
            }
        }

        inline bool operator<(const TLiteral& other) const noexcept {
            if (Type_ < other.Type_) {
                return true;
            } else if (Type_ == other.Type_) {
                switch (Type_) {
                    case ECondition::Virtual:
                        return Value_.Cast<bool>() < other.Value_.Cast<bool>();
                    case ECondition::VLAN:
                        return Value_.Cast<int>() < other.Value_.Cast<int>();
                    case ECondition::VRF:
                    case ECondition::Group:
                    case ECondition::Nanny:
                    case ECondition::Gencfg:
                    case ECondition::WalleProject:
                    case ECondition::WalleTag:
                    case ECondition::Datacenter:
                    case ECondition::Queue:
                    case ECondition::Switch:
                        return Value_.Cast<TString>() < other.Value_.Cast<TString>();
                    case ECondition::Direction:
                        return Value_.Cast<TExpressionClause::EDirection>() < other.Value_.Cast<TExpressionClause::EDirection>();
                    case ECondition::Undefined:
                        return false;
                }
            } else {
                return false;
            }
        }

    private:
        template <class T>
        inline void Append(TVector<T>& container) const {
            container.emplace_back(Value_.Cast<T>());
        }

        const ECondition Type_;
        const NAny::TAny Value_;
    };

    class TClause : public TMakeMixin<TClause>, public TAvlTreeItem<TClause, TLessCompare>, public TPoolable {
    public:
        using TTree = TAvlTreeWithAutoDelete<TClause, TLessCompare>;

        inline TClause() {
        }

        inline TClause(TLiteral::TRef literal) {
            PositiveLiterals_.InsertHolder(std::move(literal));
        }

        inline TRef Copy(TMemoryPool& pool) const noexcept {
            TRef result(Make(pool));
            CopyTree(pool, result->PositiveLiterals_, PositiveLiterals_);
            CopyTree(pool, result->NegativeLiterals_, NegativeLiterals_);
            return result;
        }

        inline TRef Intersect(TMemoryPool& pool, TClause& other) noexcept {
            TRef result(Make(pool));
            MergeTrees(result->PositiveLiterals_, PositiveLiterals_, other.PositiveLiterals_);
            MergeTrees(result->NegativeLiterals_, NegativeLiterals_, other.NegativeLiterals_);
            return result;
        }

        inline void Invert(TMemoryPool& pool, TClause::TTree& clauses) noexcept {
            for (auto it = PositiveLiterals_.Begin(); it != PositiveLiterals_.End();) {
                TLiteral::TRef literal(&(*(it++)));
                auto item(Make(pool));
                item->NegativeLiterals_.AddHolder(std::move(literal));
                clauses.AddHolder(std::move(item));
            }

            for (auto it = NegativeLiterals_.Begin(); it != NegativeLiterals_.End();) {
                TLiteral::TRef literal(&(*(it++)));
                clauses.AddHolder(Make(pool, std::move(literal)));
            }
        }

        inline TExpressionClause ToExpressionClause() const {
            TExpressionClause clause;
            for (auto it = PositiveLiterals_.Begin(); it != PositiveLiterals_.End(); ++it) {
                it->FillExpressionClause(clause, true);
            }
            for (auto it = NegativeLiterals_.Begin(); it != NegativeLiterals_.End(); ++it) {
                it->FillExpressionClause(clause, false);
            }
            return clause;
        }

        inline void Out(IOutputStream& stream) const noexcept {
            stream << "(and";
            for (auto it = PositiveLiterals_.Begin(); it != PositiveLiterals_.End(); ++it) {
                stream << " ";
                it->Out(stream);
            }
            for (auto it = NegativeLiterals_.Begin(); it != NegativeLiterals_.End(); ++it) {
                stream << " (not ";
                it->Out(stream);
                stream << ")";
            }
            stream << ")";
        }

        inline bool operator<(const TClause& other) const noexcept {
            auto positiveResult(CompareTree(PositiveLiterals_, other.PositiveLiterals_));
            auto negativeResult(CompareTree(NegativeLiterals_, other.NegativeLiterals_));
            return positiveResult == -1 || negativeResult == -1;
        }

    private:
        template <typename T>
        static inline int CompareTree(const T& lhs, const T& rhs) noexcept {
            // -1 if lhs < rhs
            //  0 if lhs = rhs
            //  1 if lhs > rhs
            for (auto leftIt(lhs.cbegin()), rightIt(rhs.cbegin()); ; ++leftIt, ++rightIt)
            {
                if (leftIt == lhs.cend() && rightIt == rhs.cend()) {
                    return 0;
                } else if (leftIt == lhs.cend()) {
                    return -1;
                } else if (rightIt == rhs.cend()) {
                    return 1;
                }
                if (*leftIt < *rightIt) {
                    return -1;
                } else if (*rightIt < *leftIt) {
                    return 1;
                }
            }
            return 0;
        }

        TLiteral::TTree PositiveLiterals_;
        TLiteral::TTree NegativeLiterals_;
    };

    class TNormalForm : public TMakeMixin<TNormalForm>, public TPoolable {
    public:
        inline TNormalForm() {
        }

        inline TNormalForm(TClause::TRef clause) {
            Clauses_.InsertHolder(std::move(clause));
        }

        inline TRef Union(TMemoryPool& pool, TNormalForm& other) noexcept {
            TRef result(Make(pool));
            MergeTrees(result->Clauses_, Clauses_, other.Clauses_);
            return result;
        }

        inline TRef Intersect(TMemoryPool& pool, TNormalForm& other) const noexcept {
            TRef result(Make(pool));
            for (auto leftIt = Clauses_.Begin(); leftIt != Clauses_.End(); ++leftIt) {
                for (auto rightIt = other.Clauses_.Begin(); rightIt != other.Clauses_.End(); ++rightIt) {
                    auto leftCopy(leftIt->Copy(pool));
                    auto rightCopy(rightIt->Copy(pool));

                    TClause::TRef clause(leftCopy->Intersect(pool, *rightCopy));
                    result->Clauses_.AddHolder(std::move(clause));
                }
            }
            return result;
        }

        inline TRef Invert(TMemoryPool& pool) noexcept {
            TRef first;
            for (auto it = Clauses_.Begin(); it != Clauses_.End();) {
                TRef second(Make(pool));

                TClause::TRef clause(&(*(it++)));
                clause->Invert(pool, second->Clauses_);

                if (first) {
                    first = first->Intersect(pool, *second);
                } else {
                    second.Swap(first);
                }
            }
            return first;
        }

        inline TRef Substract(TMemoryPool& pool, TNormalForm& other) noexcept {
            TRef inverted(other.Invert(pool));
            return Intersect(pool, *inverted);
        }

        inline TExpressionDnf ToExpressionDnf() const {
            TExpressionDnf result;
            for (auto it = Clauses_.Begin(); it != Clauses_.End(); ++it) {
                result.PushBack(it->ToExpressionClause());
            }
            return result;
        }

        inline void Out(IOutputStream& stream) const noexcept {
            stream << "(or";
            for (auto it = Clauses_.Begin(); it != Clauses_.End(); ++it) {
                stream << " ";
                it->Out(stream);
            }
            stream << ")";
        }

    private:
        TClause::TTree Clauses_;
    };
}
