#include "parser.h"
#include "lexer.h"
#include "normal_form.h"

#include <util/generic/maybe.h>
#include <util/stream/output.h>
#include <util/stream/str.h>
#include <util/string/cast.h>

namespace {
    const std::size_t MAX_DEPTH = 64;
}

#define NULL_IF_TRUE(expr)   \
    if (expr) {              \
        return nullptr;      \
    }

namespace NNetmon {
    class TParser {
    public:
        TParser(const TString& text)
            : Lexer_(text)
            , Pool_(0)
            , CurrentToken_(MakeToken(Lexer_.GetNextToken()))
            , Depth_(0)
        {
        }

    private:
        class TTypeToken;
        class TValueToken;
        class TNormalFormToken;

        class TTokenVisitor {
        public:
            virtual ~TTokenVisitor() = default;

            virtual void VisitType(TTypeToken&) noexcept {}
            virtual void VisitValue(TValueToken&) noexcept {}
            virtual void VisitNormalForm(TNormalFormToken&) noexcept {}
        };

        class TToken : public TPoolable {
        public:
            using TRef = THolder<TToken>;

            TToken(TParser* parent)
                : Parent_(parent)
            {
            }

            virtual ~TToken() = default;

            virtual std::size_t Priority() const noexcept {
                return 0;
            }

            virtual bool IsEnd() const noexcept {
                return false;
            }

            virtual TLexer::EToken TokenType() const noexcept {
                return TLexer::EToken::Undefined;
            }

            virtual TRef NullDenotation() noexcept {
                return nullptr;
            }

            virtual TRef LeftDenotation(TRef) noexcept {
                return nullptr;
            }

            // access to subclass methods should be done through visitor
            virtual void Accept(TTokenVisitor&) noexcept {
            }

        protected:
            inline TRef DrillDown() const noexcept {
                return Parent_->DrillDown(Priority());
            }

            inline TRef DrillDown(std::size_t priority) const noexcept {
                return Parent_->DrillDown(priority);
            }

            inline bool Advance(TLexer::EToken tokenType) const noexcept {
                return Parent_->Advance(tokenType);
            }

            TParser* Parent_;
        };

        class TDepthCounter {
        public:
            inline TDepthCounter(TParser* parser)
                : Parent_(parser)
            {
                Parent_->Depth_++;
            }

            ~TDepthCounter() {
                Parent_->Depth_--;
            }

        private:
            TParser* Parent_;
        };

        inline TToken::TRef DrillDown(std::size_t priority=0) noexcept {
            auto guard(TDepthCounter(this));
            NULL_IF_TRUE(Depth_ > MAX_DEPTH);

            TToken::TRef previousToken(CurrentToken_.Release());
            CurrentToken_ = MakeToken(Lexer_.GetNextToken());
            NULL_IF_TRUE(!previousToken || !CurrentToken_);

            TToken::TRef left(previousToken->NullDenotation());
            NULL_IF_TRUE(!left);

            while (priority < CurrentToken_->Priority()) {
                previousToken.Swap(CurrentToken_);
                CurrentToken_ = MakeToken(Lexer_.GetNextToken());
                NULL_IF_TRUE(!CurrentToken_);

                left = previousToken->LeftDenotation(std::move(left));
                NULL_IF_TRUE(!left);
            }

            return left;
        }

        inline bool Advance(TLexer::EToken tokenType) noexcept {
            if (CurrentToken_->TokenType() != tokenType) {
                return false;
            } else {
                CurrentToken_ = MakeToken(Lexer_.GetNextToken());
                return true;
            }
        }

        template <typename T>
        class TFactoryMixin {
        public:
            template <typename... Args>
            static inline TToken::TRef Make(TParser* parser, Args&&... args) {
                return TToken::TRef(new (parser->Pool_) T(parser, std::forward<Args>(args)...));
            }
        };

        class TTypeToken : public TFactoryMixin<TTypeToken>, public TToken {
        public:
            TTypeToken(TParser* parent, TLiteral::ECondition type)
                : TToken(parent)
                , Type_(type)
            {
            }

            TRef NullDenotation() noexcept override {
                return TTypeToken::Make(Parent_, Type_);
            }

            void Accept(TTokenVisitor& visitor) noexcept override {
                visitor.VisitType(*this);
            }

            inline TLiteral::ECondition Type() const noexcept {
                return Type_;
            }

        private:
            const TLiteral::ECondition Type_;
        };

        class TValueToken : public TFactoryMixin<TValueToken>, public TToken {
        public:
            explicit TValueToken(TParser* parent, const NAny::TAny& value)
                : TToken(parent)
                , Value_(value)
            {
            }

            TRef NullDenotation() noexcept override {
                return TValueToken::Make(Parent_, Value_);
            }

            void Accept(TTokenVisitor& visitor) noexcept override {
                visitor.VisitValue(*this);
            }

            template <typename T>
            inline TMaybe<T> Cast() const noexcept {
                if (Value_.Compatible<T>()) {
                    return Value_.Cast<T>();
                } else {
                    return Nothing();
                }
            }

        private:
            const NAny::TAny Value_;
        };

        class TNormalFormToken : public TFactoryMixin<TNormalFormToken>, public TToken {
        public:
            TNormalFormToken(TParser* parent, TNormalForm::TRef normalForm)
                : TToken(parent)
                , NormalForm_(std::move(normalForm))
            {
            }

            void Accept(TTokenVisitor& visitor) noexcept override {
                visitor.VisitNormalForm(*this);
            }

            inline TNormalForm::TRef NormalForm() noexcept {
                return std::move(NormalForm_);
            }

        private:
            TNormalForm::TRef NormalForm_;
        };

        class TAssignToken : public TFactoryMixin<TAssignToken>, public TToken, public TTokenVisitor {
        public:
            using TToken::TToken;

            inline std::size_t Priority() const noexcept override {
                return 50;
            }

            TRef LeftDenotation(TRef left) noexcept override {
                TRef right(DrillDown());
                NULL_IF_TRUE(!right);

                left->Accept(*this);
                right->Accept(*this);
                NULL_IF_TRUE(!Literal_);

                auto clause(TClause::Make(Parent_->Pool_, std::move(Literal_)));
                auto normalForm(TNormalForm::Make(Parent_->Pool_, std::move(clause)));
                return TNormalFormToken::Make(Parent_, std::move(normalForm));
            }

            void VisitType(TTypeToken& token) noexcept override {
                Type_ = token.Type();
            }

            void VisitValue(TValueToken& token) noexcept override {
                switch (Type_) {
                    case TLiteral::ECondition::VLAN: {
                        return SetLiteral<int>(token);
                    }
                    case TLiteral::ECondition::Virtual: {
                        return SetLiteral<bool>(token);
                    }
                    case TLiteral::ECondition::VRF:
                    case TLiteral::ECondition::Group:
                    case TLiteral::ECondition::Nanny:
                    case TLiteral::ECondition::Gencfg:
                    case TLiteral::ECondition::WalleProject:
                    case TLiteral::ECondition::WalleTag:
                    case TLiteral::ECondition::Datacenter:
                    case TLiteral::ECondition::Queue:
                    case TLiteral::ECondition::Switch: {
                        return SetLiteral<TString>(token);
                    }
                    case TLiteral::ECondition::Direction: {
                        return SetDirectionLiteral(token);
                    }
                    case TLiteral::Undefined: {
                        return;
                    }
                }
            }

        private:
            template <class T>
            inline void SetLiteral(const TValueToken& token) noexcept {
                auto value(token.Cast<T>());
                if (value.Defined()) {
                    Literal_ = TLiteral::Make(Parent_->Pool_, Type_, value.GetRef());
                }
            }

            inline void SetDirectionLiteral(const TValueToken& token) noexcept {
                using EDirection = TExpressionClause::EDirection;
                auto value(token.Cast<TString>());
                if (!value.Defined()) {
                    return;
                }

                EDirection direction(EDirection::Undefined);
                if (value.GetRef() == TStringBuf("both")) {
                    direction = EDirection::Both;
                } else if (value.GetRef() == TStringBuf("target")) {
                    direction = EDirection::Target;
                } else if (value.GetRef() == TStringBuf("source")) {
                    direction = EDirection::Source;
                } else {
                    return;
                }

                Literal_ = TLiteral::Make(Parent_->Pool_, Type_, direction);
            }

            TLiteral::ECondition Type_ = TLiteral::ECondition::Undefined;
            TLiteral::TRef Literal_;
        };

        class TLeftBracketToken : public TFactoryMixin<TLeftBracketToken>, public TToken {
        public:
            using TToken::TToken;

            std::size_t Priority() const noexcept override {
                return 60;
            }

            TRef NullDenotation() noexcept override {
                auto token(DrillDown(0));
                if (Advance(TLexer::EToken::RightBracket)) {
                    return token;
                } else {
                    return nullptr;
                }
            }
        };

        class TRightBracketToken : public TFactoryMixin<TRightBracketToken>, public TToken {
        public:
            using TToken::TToken;

            TLexer::EToken TokenType() const noexcept override {
                return TLexer::EToken::RightBracket;
            }
        };

        template <class T>
        class TBinaryToken : public TFactoryMixin<T>, public TToken, public TTokenVisitor {
        public:
            using TToken::TToken;

            TRef LeftDenotation(TRef left) noexcept override {
                left->Accept(*this);
                NULL_IF_TRUE(!NormalForm_);

                auto right(DrillDown());
                NULL_IF_TRUE(!right);

                right->Accept(*this);
                NULL_IF_TRUE(!NormalForm_ || !Ready_);

                return TNormalFormToken::Make(Parent_, std::move(NormalForm_));
            }

            void VisitNormalForm(TNormalFormToken& token) noexcept override {
                auto normalForm(token.NormalForm());
                if (NormalForm_) {
                    NormalForm_ = Operation(*NormalForm_, *normalForm);
                    Ready_ = true;
                } else {
                    normalForm.Swap(NormalForm_);
                }
            }

            virtual TNormalForm::TRef Operation(TNormalForm& lhs, TNormalForm& rhs) const noexcept = 0;

        private:
            TNormalForm::TRef NormalForm_;
            bool Ready_ = false;
        };

        class TUnionToken : public TBinaryToken<TUnionToken> {
        public:
            using TBinaryToken::TBinaryToken;

            inline std::size_t Priority() const noexcept override {
                return 10;
            }

            TNormalForm::TRef Operation(TNormalForm& lhs, TNormalForm& rhs) const noexcept override {
                return lhs.Union(Parent_->Pool_, rhs);
            }
        };

        class TIntersectToken : public TBinaryToken<TIntersectToken> {
        public:
            using TBinaryToken::TBinaryToken;

            inline std::size_t Priority() const noexcept override {
                return 20;
            }

            TNormalForm::TRef Operation(TNormalForm& lhs, TNormalForm& rhs) const noexcept override {
                return lhs.Intersect(Parent_->Pool_, rhs);
            }
        };

        class TSubstractToken : public TBinaryToken<TSubstractToken> {
        public:
            using TBinaryToken::TBinaryToken;

            inline std::size_t Priority() const noexcept override {
                return 30;
            }

            TNormalForm::TRef Operation(TNormalForm& lhs, TNormalForm& rhs) const noexcept override {
                return lhs.Substract(Parent_->Pool_, rhs);
            }
        };

        class TNegateToken : public TFactoryMixin<TNegateToken>, public TToken, public TTokenVisitor {
        public:
            using TToken::TToken;

            inline std::size_t Priority() const noexcept override {
                return 40;
            }

            TRef NullDenotation() noexcept override {
                auto token(DrillDown());
                NULL_IF_TRUE(!token);

                token->Accept(*this);
                NULL_IF_TRUE(!NormalForm_);

                return TNormalFormToken::Make(Parent_, std::move(NormalForm_));
            }

            void VisitNormalForm(TNormalFormToken& token) noexcept override {
                auto normalForm(token.NormalForm());
                NormalForm_ = normalForm->Invert(Parent_->Pool_);
            }

        private:
            TNormalForm::TRef NormalForm_;
        };

        class TEndToken : public TFactoryMixin<TEndToken>, public TToken {
        public:
            using TToken::TToken;

            bool IsEnd() const noexcept override {
                return true;
            }
        };

        TToken::TRef MakeToken(TLexer::TTokenValue value) {
            switch (value.TokenType) {
                case TLexer::EToken::Integer:
                    return TValueToken::Make(this, FromString<int>(value.Value));
                case TLexer::EToken::String:
                case TLexer::EToken::GroupValue:
                    return TValueToken::Make(this, value.Value);
                case TLexer::EToken::False:
                    return TValueToken::Make(this, false);
                case TLexer::EToken::True:
                    return TValueToken::Make(this, true);

                case TLexer::EToken::Direction:
                    return TTypeToken::Make(this, TLiteral::ECondition::Direction);
                case TLexer::EToken::VLAN:
                    return TTypeToken::Make(this, TLiteral::ECondition::VLAN);
                case TLexer::EToken::VRF:
                    return TTypeToken::Make(this, TLiteral::ECondition::VRF);
                case TLexer::EToken::Group:
                    return TTypeToken::Make(this, TLiteral::ECondition::Group);
                case TLexer::EToken::Nanny:
                    return TTypeToken::Make(this, TLiteral::ECondition::Nanny);
                case TLexer::EToken::Gencfg:
                    return TTypeToken::Make(this, TLiteral::ECondition::Gencfg);
                case TLexer::EToken::WalleProject:
                    return TTypeToken::Make(this, TLiteral::ECondition::WalleProject);
                case TLexer::EToken::WalleTag:
                    return TTypeToken::Make(this, TLiteral::ECondition::WalleTag);
                case TLexer::EToken::Virtual:
                    return TTypeToken::Make(this, TLiteral::ECondition::Virtual);
                case TLexer::EToken::Datacenter:
                    return TTypeToken::Make(this, TLiteral::ECondition::Datacenter);
                case TLexer::EToken::Queue:
                    return TTypeToken::Make(this, TLiteral::ECondition::Queue);
                case TLexer::EToken::Switch:
                    return TTypeToken::Make(this, TLiteral::ECondition::Switch);

                case TLexer::EToken::Assign:
                    return TAssignToken::Make(this);
                case TLexer::EToken::Invert:
                    return TNegateToken::Make(this);
                case TLexer::EToken::And:
                    return TIntersectToken::Make(this);
                case TLexer::EToken::Or:
                    return TUnionToken::Make(this);
                case TLexer::EToken::Substract:
                    return TSubstractToken::Make(this);
                case TLexer::EToken::LeftBracket:
                    return TLeftBracketToken::Make(this);
                case TLexer::EToken::RightBracket:
                    return TRightBracketToken::Make(this);

                case TLexer::EToken::End:
                    return TEndToken::Make(this);

                case TLexer::EToken::Undefined:
                    return nullptr;
            };
        }

        class TNormalFormExtractor : public TTokenVisitor {
        public:
            void VisitNormalForm(TNormalFormToken& token) noexcept override {
                NormalForm_ = token.NormalForm();
            }

            inline TNormalForm::TRef NormalForm() noexcept {
                return std::move(NormalForm_);
            }

        private:
            TNormalForm::TRef NormalForm_;
        };

    public:
        inline TNormalForm::TRef ToNormalForm() noexcept {
            auto token(DrillDown());
            NULL_IF_TRUE(!token);

            if (CurrentToken_ && !CurrentToken_->IsEnd()) {
                return nullptr;
            }

            TNormalFormExtractor extractor;
            token->Accept(extractor);

            return extractor.NormalForm();
        }

    private:
        TLexer Lexer_;
        TMemoryPool Pool_;
        TToken::TRef CurrentToken_;
        std::size_t Depth_;
    };

    TMaybe<TString> StringifyNormalForm(const TString& input) noexcept {
        TParser parser(input);
        TNormalForm::TRef normalForm(parser.ToNormalForm());
        if (normalForm) {
            TString result;
            TStringOutput output(result);
            normalForm->Out(output);
            output.Finish();
            return result;
        } else {
            return Nothing();
        }
    }

    TExpressionDnf ParseExpression(const TString& input) {
        TParser parser(input);
        TNormalForm::TRef normalForm(parser.ToNormalForm());
        if (!normalForm) {
            ythrow yexception() << "can't parse given expression";
        }
        return normalForm->ToExpressionDnf();
    }
}
