#include "parser.h"

#include <util/generic/xrange.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NNetmon;

class TParserTest: public TTestBase {
    UNIT_TEST_SUITE(TParserTest);
    UNIT_TEST(TestFilters)
    UNIT_TEST(TestOperators)
    UNIT_TEST(TestNewLine)
    UNIT_TEST(TestBrackets)
    UNIT_TEST(TestMerge)
    UNIT_TEST(TestInvalid)
    UNIT_TEST(TestExpressionComparator)
    UNIT_TEST(TestValidExpression)
    UNIT_TEST(TestDuplicateLiterals)
    UNIT_TEST(TestDepth)
    UNIT_TEST(TestOrder)
    UNIT_TEST(TestDirection)
    UNIT_TEST_SUITE_END();

private:
    inline void TestFilters() {
        AssertEqual("vlan = 678", "(or (and (vlan 678)))");
        AssertEqual("vrf = Search", "(or (and (vrf Search)))");
        AssertEqual("group = G@ALL_SEARCH", "(or (and (group G@ALL_SEARCH)))");
        AssertEqual("virtual = true", "(or (and (virtual 1)))");
        AssertEqual("virtual = false", "(or (and (virtual 0)))");
        // spaces aren't needed
        AssertEqual("vlan=678", "(or (and (vlan 678)))");

        AssertEqual("datacenter = vla", "(or (and (datacenter vla)))");
        AssertEqual("dc = man", "(or (and (datacenter man)))");
        AssertEqual("queue = sas-1.2.3", "(or (and (queue sas-1.2.3)))");
        AssertEqual("line = man#4", "(or (and (queue man#4)))");
        AssertEqual("switch = sas1-i45", "(or (and (switch sas1-i45)))");

        AssertEqual("nanny = production_yasm_front", "(or (and (nanny production_yasm_front)))");
        AssertEqual("gencfg = ALL_JUGGLER_BALANCERS", "(or (and (gencfg ALL_JUGGLER_BALANCERS)))");
        AssertEqual("walle_project = rtc-yabs", "(or (and (walle-project rtc-yabs)))");
        AssertEqual("walle_tag = rtc", "(or (and (walle-tag rtc)))");

        AssertEqual("direction = both", "(or (and (direction Both)))");
        AssertEqual("direction = source", "(or (and (direction Source)))");
        AssertEqual("direction = target", "(or (and (direction Target)))");
    }

    inline void TestOperators() {
        AssertEqual("vlan = 678 or vlan = 789 or vrf = Search",
                    "(or (and (vlan 678)) (and (vlan 789)) (and (vrf Search)))");
        AssertEqual("vlan = 678 and vrf = Search and group = G@ALL_SEARCH",
                    "(or (and (group G@ALL_SEARCH) (vlan 678) (vrf Search)))");
        AssertEqual("not vlan = 678 and vrf = Search",
                    "(or (and (vrf Search) (not (vlan 678))))");
        // intersect operator has higher priority than union
        AssertEqual("vlan = 678 or vrf = Search and group = G@ALL_SEARCH",
                    "(or (and (group G@ALL_SEARCH) (vrf Search)) (and (vlan 678)))");
        // TODO: maybe change substract priority?
        AssertEqual("vlan = 678 or vrf = Search - group = G@ALL_SEARCH",
                    "(or (and (vlan 678)) (and (vrf Search) (not (group G@ALL_SEARCH))))");
    }

    inline void TestNewLine() {
        AssertEqual("vlan = 678 or vlan = 679 or \n vlan = 680",
                    "(or (and (vlan 678)) (and (vlan 679)) (and (vlan 680)))");
        AssertEqual("vlan = 678 or vlan = 679 \n or vlan = 680",
                    "(or (and (vlan 678)) (and (vlan 679)) (and (vlan 680)))");
    }

    inline void TestBrackets() {
        AssertEqual("(vlan = 678 or vrf = Search) and group = G@ALL_SEARCH",
                    "(or (and (group G@ALL_SEARCH) (vlan 678)) (and (group G@ALL_SEARCH) (vrf Search)))");
        AssertEqual("not (vlan = 678 or vrf = Search) and group = G@ALL_SEARCH",
                    "(or (and (group G@ALL_SEARCH) (not (vlan 678)) (not (vrf Search))))");
        AssertEqual("(vlan = 678 or vrf = Search) - (virtual = true or vlan = 789)",
                    "(or (and (vlan 678) (not (vlan 789)) (not (virtual 1))) (and (vrf Search) (not (vlan 789)) (not (virtual 1))))");
        AssertEqual("(vlan = 604 or vrf = Search) - dc = sas - dc = man",
                    "(or (and (vlan 604) (not (datacenter man)) (not (datacenter sas))) (and (vrf Search) (not (datacenter man)) (not (datacenter sas))))");
        AssertEqual("(vlan = 604 and vlan = 688) - dc = sas - dc = man",
                    "(or (and (vlan 604) (vlan 688) (not (datacenter man)) (not (datacenter sas))))");
    }

    inline void TestMerge() {
        AssertEqual("vlan = 678 or vlan = 678 and vlan = 678", "(or (and (vlan 678)))");
        AssertEqual("vlan = 678 - vlan = 678", "(or (and (vlan 678) (not (vlan 678))))");
    }

    inline void TestInvalid() {
        AssertIsInvalid("");
        AssertIsInvalid(" ");
        AssertIsInvalid("()");
        AssertIsInvalid("((())");

        AssertIsInvalid("678");
        AssertIsInvalid("vlan");
        AssertIsInvalid("vlan 678");
        AssertIsInvalid("vlan == 678");
        AssertIsInvalid("vlan not 678");
        AssertIsInvalid("vlan = 678 or vlan = 789 - vlan673");

        AssertIsInvalid("vlan = Search");
        AssertIsInvalid("vlan = true");
        AssertIsInvalid("vlan = false");

        AssertIsInvalid("vrf = 678");
        AssertIsInvalid("vrf = true");
        AssertIsInvalid("vrf = false");

        AssertIsInvalid("group = 678");
        AssertIsInvalid("group = true");
        AssertIsInvalid("group = false");

        AssertIsInvalid("virtual = Search");
        AssertIsInvalid("virtual = 678");
        AssertIsInvalid("virtual = G@ALL_SEARCH");

        AssertIsInvalid("group = G@ALL group = G@SEARCH");

        AssertIsInvalid("direction = not_valid");

    }

    inline void TestExpressionComparator() {
        {
            TExpressionDnf lhs;
            TExpressionDnf rhs;
            UNIT_ASSERT_EQUAL(lhs, rhs);
        }
        {
            TExpressionClause clause;
            TExpressionDnf lhs({clause});
            TExpressionDnf rhs({clause});
            UNIT_ASSERT_EQUAL(lhs, rhs);
        }
        {
            TExpressionClause clause;
            clause.PositiveVlanFilter() = {678};
            TExpressionDnf lhs({clause});
            TExpressionDnf rhs({clause});
            UNIT_ASSERT_EQUAL(lhs, rhs);
        }
        {
            TExpressionClause leftClause;
            leftClause.PositiveVlanFilter() = {678};
            TExpressionClause rightClause;
            rightClause.NegativeVlanFilter() = {678};
            TExpressionDnf lhs({leftClause});
            TExpressionDnf rhs({rightClause});
            UNIT_ASSERT_UNEQUAL(lhs, rhs);
        }
    }

    inline void TestValidExpression() {
        {
            TExpressionClause clause;
            clause.PositiveVlanFilter() = {678};
            AssertEqual("vlan = 678", TExpressionDnf({clause}));
            // because of clause merging
            AssertEqual("vlan = 678 and vlan = 678", TExpressionDnf({clause}));
        }
        {
            TExpressionClause clause;
            clause.NegativeVlanFilter() = {678};
            AssertEqual("not vlan = 678", TExpressionDnf({clause}));
        }
        {
            TExpressionClause firstClause;
            firstClause.PositiveVrfFilter() = {"search"};
            firstClause.PositiveVlanFilter() = {678};
            TExpressionClause secondClause;
            secondClause.PositiveVirtualFilter() = {true};
            secondClause.PositiveGroupFilter() = {"G@ALL_RUNTIME"};
            AssertEqual("(vrf = search and vlan = 678) or (virtual = true and group = G@ALL_RUNTIME)", TExpressionDnf({
                // because of sorting
                secondClause,
                firstClause
            }));
        }
    }

    inline void TestDuplicateLiterals() {
        {
            TExpressionClause clause;
            clause.PositiveVlanFilter() = {604, 688};
            AssertEqual("vlan = 604 and vlan = 688", TExpressionDnf({clause}));
        }
        {
            TExpressionClause clause;
            clause.PositiveVlanFilter() = {604, 688};
            AssertEqual("vlan = 688 and vlan = 604", TExpressionDnf({clause}));
        }
    }

    inline void TestDepth() {
        TStringBuilder builder;

        for (const auto& idx : xrange(62)) {
            Y_UNUSED(idx);
            builder << "(";
        }

        builder << "vlan = 678";

        for (const auto& idx : xrange(62)) {
            Y_UNUSED(idx);
            builder << ")";
        }

        AssertValid(builder);
        AssertIsInvalid(TStringBuilder() << "(" << builder << ")");
    }

    inline void TestOrder() {
        // try to protect hash values from accidental change
        UNIT_ASSERT(ParseExpression("vlan=768").hash() < ParseExpression("virtual=false").hash());
        UNIT_ASSERT(ParseExpression("virtual=false").hash() < ParseExpression("virtual=true").hash());
        UNIT_ASSERT(ParseExpression("virtual=true").hash() < ParseExpression("nanny=something").hash());
    }

    inline void TestDirection() {
        UNIT_ASSERT_EXCEPTION(ParseExpression("direction = target and direction = source"), yexception);
        UNIT_ASSERT_EXCEPTION(ParseExpression("not direction = source"), yexception);

        {
            TExpressionClause first, second;

            first.DirectionFilter() = TExpressionClause::EDirection::Source;
            first.PositiveVrfFilter() = {"Search"};

            second.DirectionFilter() = TExpressionClause::EDirection::Target;
            second.PositiveVrfFilter() = {"hbf"};

            AssertEqual("(direction = source and vrf = Search) or (direction = target and vrf = hbf)", TExpressionDnf({first, second}));
        }

        {
            TExpressionClause first, second;

            first.DirectionFilter() = TExpressionClause::EDirection::Target;
            first.PositiveVrfFilter() = {"Search"};

            second.DirectionFilter() = TExpressionClause::EDirection::Target;
            second.PositiveVrfFilter() = {"hbf"};

            AssertEqual("(vrf = Search or vrf = hbf) and direction = target ", TExpressionDnf({first, second}));
        }
    }

    inline void AssertEqual(const TString& expression, const TString& reference) {
        auto result(StringifyNormalForm(expression));
        UNIT_ASSERT(result.Defined());
        UNIT_ASSERT_STRINGS_EQUAL(result.GetRef(), reference);
    }

    inline void AssertEqual(const TString& expression, const TExpressionDnf& reference) {
        auto result(ParseExpression(expression));
        UNIT_ASSERT_EQUAL(result, reference);
    }

    inline void AssertValid(const TString& expression) {
        auto result(StringifyNormalForm(expression));
        UNIT_ASSERT(result.Defined());
    }

    inline void AssertIsInvalid(const TString& expression) {
        auto result(StringifyNormalForm(expression));
        UNIT_ASSERT(!result.Defined());
    }
};

UNIT_TEST_SUITE_REGISTRATION(TParserTest);
