#pragma once

#include <library/cpp/on_disk/mms/unordered_map.h>
#include <library/cpp/on_disk/mms/string.h>
#include <library/cpp/on_disk/mms/writer.h>
#include <library/cpp/on_disk/mms/mapping.h>
#include <library/cpp/on_disk/mms/declare_fields.h>
#include <library/cpp/on_disk/coded_blob/coded_blob_simple_builder.h>
#include <library/cpp/on_disk/coded_blob/coded_blob_array.h>

#include <util/generic/maybe.h>

namespace NNetmon {
    namespace {
        template <class P, typename TKey, typename... TKeys>
        struct TOffsetHierarchyPod {
            using MmappedType = TOffsetHierarchyPod<NMms::TMmapped, TKey, TKeys...>;
            using TChild = TOffsetHierarchyPod<P, TKeys...>;

            template <class F, typename... Args>
            inline void Iterate(F&& func, Args... args) const {
                for (const auto& pair : Children) {
                    pair.second.Iterate(func, args..., pair.first);
                }
            }

            NMms::TFastUnorderedMap<P, TKey, TChild> Children;

            MMS_DECLARE_FIELDS(Children);
        };

        template <class P, typename TKey>
        struct TOffsetHierarchyPod<P, TKey> {
            using MmappedType = TOffsetHierarchyPod<NMms::TMmapped, TKey>;

            template <class F, typename... Args>
            inline void Iterate(F&& func, Args... args) const {
                for (const auto& pair : Children) {
                    func(std::make_tuple(args..., pair.first), pair.second);
                }
            }

            NMms::TFastUnorderedMap<P, TKey, size_t> Children;

            MMS_DECLARE_FIELDS(Children);
        };

        template <typename TKey, typename... TKeys>
        struct TOffsetHierarchy {
        public:
            using TChild = TOffsetHierarchy<TKeys...>;

            using TTuple = std::tuple<TKey, TKeys...>;
            using TTupleIndices = std::make_index_sequence<std::tuple_size<TTuple>::value>;

            using TWritablePod = TOffsetHierarchyPod<NMms::TStandalone, TKey, TKeys...>;
            using TReadablePod = TOffsetHierarchyPod<NMms::TMmapped, TKey, TKeys...>;

            static inline TMaybe<size_t> Find(const TReadablePod& pod, const TKey& key, const TKeys& ...keys) {
                const auto it(pod.Children.find(key));
                if (it == pod.Children.end()) {
                    return Nothing();
                } else {
                    return TChild::Find(it->second, keys...);
                }
            }

            static inline TMaybe<size_t> FindTuple(const TReadablePod& pod, const TTuple& key) {
                return FindImpl(pod, key, TTupleIndices{});
            }

            static inline void Insert(TWritablePod& pod, size_t offset, const TKey& key, const TKeys& ...keys) {
                TChild::Insert(pod.Children[key], offset, keys...);
            }

            static inline void InsertTuple(TWritablePod& pod, size_t offset, const TTuple& key) {
                InsertImpl(pod, offset, key, TTupleIndices{});
            }

        private:
            template <std::size_t ...I>
            static inline TMaybe<size_t> FindImpl(const TReadablePod& pod, const TTuple& key, std::index_sequence<I...>) {
                return Find(pod, std::get<I>(key)...);
            }

            template <std::size_t ...I>
            static inline void InsertImpl(TWritablePod& pod, size_t offset, const TTuple& key, std::index_sequence<I...>) {
                Insert(pod, offset, std::get<I>(key)...);
            }
        };

        template <typename TKey>
        struct TOffsetHierarchy<TKey> {
            using TWritablePod = TOffsetHierarchyPod<NMms::TStandalone, TKey>;
            using TReadablePod = TOffsetHierarchyPod<NMms::TMmapped, TKey>;

            static inline TMaybe<size_t> Find(const TReadablePod& pod, const TKey& key) {
                const auto it(pod.Children.find(key));
                if (it == pod.Children.end()) {
                    return Nothing();
                } else {
                    return it->second;
                }
            }

            static inline void Insert(TWritablePod& pod, size_t offset, const TKey& key) {
                pod.Children.insert(std::make_pair(key, offset));
            }
        };
    }

    using THistoryHierarchy = TOffsetHierarchy<
        // expression id, network type, protocol type
        ui64, ui8, ui8,
        // target, source
        ui64, ui64
    >;
    using THistoryTuple = THistoryHierarchy::TTuple;

    class THistoryChunkBuilder: public TNonCopyable {
    public:
        THistoryChunkBuilder(IOutputStream& keyStream, IOutputStream& dataStream, TStringBuf codec)
            : KeyStream(keyStream)
            , DataStream(dataStream)
            , DataBuilder(1 << 20, "snappy")
            , Position(0)
        {
            DataBuilder.Init(codec);
        }

        void Add(const THistoryTuple& key, const TStringBuf& value) {
            THistoryHierarchy::InsertTuple(OffsetMap, Position, key);
            DataBuilder.Add(value);
            Position++;
        }

        void Finish() {
            NMms::SafeWrite(KeyStream, OffsetMap);
            KeyStream.Finish();

            DataBuilder.Finish(&DataStream);
            DataStream.Finish();
        }

    private:
        IOutputStream& KeyStream;
        IOutputStream& DataStream;

        THistoryHierarchy::TWritablePod OffsetMap;
        NCodedBlob::TCodedBlobArraySimpleBuilder DataBuilder;

        std::size_t Position;
    };

    class THistoryChunkReader: public TNonCopyable {
    public:
        using TRef = TAtomicSharedPtr<THistoryChunkReader>;

        THistoryChunkReader(const TBlob& keyBlob, const TBlob& dataBlob)
            : OffsetMap(keyBlob)
            , DataArray(dataBlob)
        {
        }

        TStringBuf Find(const THistoryTuple& key, TBuffer& buffer) const {
            const auto offset(THistoryHierarchy::FindTuple(*OffsetMap, key));
            if (offset.Defined()) {
                Y_VERIFY(*offset < DataArray.Count());
                return DataArray.GetByIndex(*offset, buffer);
            } else {
                return {};
            }
        }

        template <class F>
        void Iterate(F&& func) const {
            TBuffer buffer;
            OffsetMap->Iterate([this, &buffer, &func] (THistoryTuple key, size_t offset) {
                buffer.Clear();
                func(std::move(key), DataArray.GetByIndex(offset, buffer));
            });
        }

    private:
        const NMms::TMapping<THistoryHierarchy::TReadablePod> OffsetMap;
        const NCodedBlob::TCodedBlobArray DataArray;
    };
}
