#pragma once

#include <infra/netmon/state_index_history.h>

namespace NNetmon {
    template <class TPairStateHistory>
    class THistorySeriesMerger : public TNonCopyable {
    public:
        void Append(TVector<typename TPairStateHistory::TRef>&& states) {
            for (auto& state : states) {
                Append(std::move(state));
            }
        }

        void Append(typename TPairStateHistory::TRef state) {
            TKeeper keeper(std::move(state));
            if (!keeper.IsEnd()) {
                Keepers.emplace_back(std::move(keeper));
                PushHeap(Keepers.begin(), Keepers.end());
            }
        }

        void Merge(TPairStateHistory& resultingState, TInstant since, TInstant until) {
            TVector<TKeeper> peers;
            peers.reserve(Keepers.size());

            while (!Keepers.empty()) {
                peers.clear();

                while (!Keepers.empty() && (peers.empty() || peers.back() == Keepers.front())) {
                    peers.emplace_back(std::move(Keepers.front()));
                    PopHeap(Keepers.begin(), Keepers.end());
                    Keepers.pop_back();
                }

                TInstant timestamp;
                typename TPairStateHistory::TConnectivitySeries::THistogram connectivity;
                TSampleHistogram rtt;

                for (auto& keeper : peers) {
                    timestamp = *keeper.TimestampIterator;
                    connectivity.Merge(*keeper.ConnectivityIterator);
                    rtt.Merge(*keeper.RttIterator);

                    keeper.Inc();
                    if (!keeper.IsEnd()) {
                        Keepers.emplace_back(std::move(keeper));
                        PushHeap(Keepers.begin(), Keepers.end());
                    }
                }

                if (since <= timestamp && timestamp <= until) {
                    resultingState.MutableTimestampSeries().Append(timestamp);
                    resultingState.MutableConnectivitySeries().Append(connectivity);
                    resultingState.MutableRttSeries().Append(rtt);
                }
            }
        }

        void Merge(TPairStateHistory& resultingState) {
            Merge(resultingState, TInstant::Zero(), TInstant::Max());
        }

    private:
        struct TKeeper : public TMoveOnly {
            TKeeper(typename TPairStateHistory::TRef state)
                : State(std::move(state))
                , TimestampIterator(State->MutableTimestampSeries().begin())
                , TimestampEnding(State->MutableTimestampSeries().end())
                , ConnectivityIterator(State->MutableConnectivitySeries().begin())
                , RttIterator(State->MutableRttSeries().begin())
            {
            }

            inline bool IsEnd() const {
                return TimestampIterator == TimestampEnding;
            }

            inline void Inc() {
                ++TimestampIterator;
                ++ConnectivityIterator;
                ++RttIterator;
            }

            inline bool operator==(const TKeeper& other) const noexcept {
                Y_ASSERT(!IsEnd() && !other.IsEnd());
                return *TimestampIterator == *other.TimestampIterator;
            }

            inline bool operator<(const TKeeper& other) const noexcept {
                Y_ASSERT(!IsEnd() && !other.IsEnd());
                return *TimestampIterator > *other.TimestampIterator;
            }

            typename TPairStateHistory::TRef State;
            TTimestampSeries::TIterator TimestampIterator;
            TTimestampSeries::TIterator TimestampEnding;
            typename TPairStateHistory::TConnectivitySeries::TIterator ConnectivityIterator;
            TSampleHistogramSeries::TIterator RttIterator;
        };

        TVector<TKeeper> Keepers;
    };

    using TSwitchHistorySeriesMerger = THistorySeriesMerger<TSwitchPairStateHistory>;
    using TLineHistorySeriesMerger = THistorySeriesMerger<TLinePairStateHistory>;
    using TDatacenterHistorySeriesMerger = THistorySeriesMerger<TDatacenterPairStateHistory>;
}
