#include <infra/netmon/history_merger.h>
#include <infra/netmon/common_ut.h>
#include <infra/netmon/library/iterators.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NNetmon;

class THistoryMergerTest: public TTestBase {
    UNIT_TEST_SUITE(THistoryMergerTest);
    UNIT_TEST(TestSwitchMerge)
    UNIT_TEST(TestLineMerge)
    UNIT_TEST(TestDatacenterMerge)
    UNIT_TEST_SUITE_END();

public:
    THistoryMergerTest()
        : SourceHost(FindHost("jmon-test.search.yandex.net"))
        , TargetHost(FindHost("man1-0015.search.yandex.net"))
    {
        SampleHistogram.Append(1.0);
        ConnectivityHistogram.Append(1.0);
        AverageHistogram.Merge(ConnectivityHistogram);
    }

private:
    struct TSwitchConfig {
        using TState = TSwitchPairStateHistory;

        TSwitchConfig(THistoryMergerTest* parent)
            : Parent(parent)
            , Result(CreateState())
        {
        }

        TSwitchPairStateHistory::TRef CreateState() const {
            return CreateEmptyStateHistory(Parent->TargetHost.GetSwitch(), Parent->SourceHost.GetSwitch());
        }

        void AppendToState(TSwitchPairStateHistory& result, const TInstant& timestamp) const {
            result.MutableTimestampSeries().Append(timestamp);
            result.MutableConnectivitySeries().Append(GetConnectivityHistogram());
            result.MutableRttSeries().Append(Parent->SampleHistogram);
        }

        const TConnectivityHistogram& GetConnectivityHistogram() const {
            return Parent->ConnectivityHistogram;
        }

        THistoryMergerTest* Parent;
        TSwitchHistorySeriesMerger Merger;
        TSwitchPairStateHistory::TRef Result;
    };

    struct TLineConfig {
        using TState = TLinePairStateHistory;

        TLineConfig(THistoryMergerTest* parent)
            : Parent(parent)
            , Result(CreateState())
        {
        }

        TLinePairStateHistory::TRef CreateState() const {
            return CreateEmptyStateHistory(Parent->TargetHost.GetLine(), Parent->SourceHost.GetLine());
        }

        void AppendToState(TLinePairStateHistory& result, const TInstant& timestamp) const {
            result.MutableTimestampSeries().Append(timestamp);
            result.MutableConnectivitySeries().Append(GetConnectivityHistogram());
            result.MutableRttSeries().Append(Parent->SampleHistogram);
        }

        const TAverageHistogram& GetConnectivityHistogram() const {
            return Parent->AverageHistogram;
        }

        THistoryMergerTest* Parent;
        TLineHistorySeriesMerger Merger;
        TLinePairStateHistory::TRef Result;
    };

    struct TDatacenterConfig {
        using TState = TDatacenterPairStateHistory;

        TDatacenterConfig(THistoryMergerTest* parent)
            : Parent(parent)
            , Result(CreateState())
        {
        }

        inline TDatacenterPairStateHistory::TRef CreateState() const {
            return CreateEmptyStateHistory(Parent->TargetHost.GetDatacenter(), Parent->SourceHost.GetDatacenter());
        }

        inline void AppendToState(TDatacenterPairStateHistory& result, const TInstant& timestamp) const {
            result.MutableTimestampSeries().Append(timestamp);
            result.MutableConnectivitySeries().Append(GetConnectivityHistogram());
            result.MutableRttSeries().Append(Parent->SampleHistogram);
        }

        const TAverageHistogram& GetConnectivityHistogram() const {
            return Parent->AverageHistogram;
        }

        THistoryMergerTest* Parent;
        TDatacenterHistorySeriesMerger Merger;
        TDatacenterPairStateHistory::TRef Result;
    };

    inline TVector<TInstant> GenerateTimestamps(const TDuration& interval) {
        const auto now(RoundInstant(TInstant::Now(), interval));
        return {now, now + interval, now + interval * 2};
    }

    template <class TConfig>
    inline void AssertMergerWorks(TConfig& config) {
        auto timestamps(GenerateTimestamps(config.Result->GetInterval()));

        // check empty one
        config.Merger.Append(config.CreateState());

        // fill states
        auto first(config.CreateState());
        config.AppendToState(*first, timestamps.at(0));
        config.AppendToState(*first, timestamps.at(1));

        auto second(config.CreateState());
        config.AppendToState(*second, timestamps.at(1));
        config.AppendToState(*second, timestamps.at(2));

        auto third(config.CreateState());
        config.AppendToState(*third, timestamps.at(0));

        // and append them to merger
        config.Merger.Append(std::move(first));
        config.Merger.Append(std::move(second));
        config.Merger.Append(std::move(third));
        config.Merger.Merge(*config.Result);

        // results
        TVector<size_t> indices{0, 0, 1, 1, 2};

        TVector<TSampleHistogram> rtts;
        rtts.resize(timestamps.size());
        for (auto idx : indices) {
            rtts.at(idx).Merge(SampleHistogram);
        }

        TVector<typename TConfig::TState::TConnectivitySeries::THistogram> connectivities;
        connectivities.resize(timestamps.size());
        for (auto idx : indices) {
            connectivities.at(idx).Merge(config.GetConnectivityHistogram());
        }

        const auto timestampSeries(config.Result->GetTimestampSeries());
        const auto rttSeries(config.Result->GetRttSeries());
        const auto connectivitySeries(config.Result->GetConnectivitySeries());
        for (const auto& elements : ZipIterator(timestampSeries, timestamps,
                                                rttSeries, rtts,
                                                connectivitySeries, connectivities)) {
            UNIT_ASSERT_EQUAL(std::get<0>(elements), std::get<1>(elements));
            UNIT_ASSERT_EQUAL(std::get<2>(elements), std::get<3>(elements));
            UNIT_ASSERT_EQUAL(std::get<4>(elements), std::get<5>(elements));
        }
    }

    inline void TestSwitchMerge() {
        TSwitchConfig config(this);
        AssertMergerWorks(config);
    }

    inline void TestLineMerge() {
        TLineConfig config(this);
        AssertMergerWorks(config);
    }

    inline void TestDatacenterMerge() {
        TDatacenterConfig config(this);
        AssertMergerWorks(config);
    }

    const TTopology::THostRef SourceHost;
    const TTopology::THostRef TargetHost;

    TSampleHistogram SampleHistogram;
    TConnectivityHistogram ConnectivityHistogram;
    TAverageHistogram AverageHistogram;
};

UNIT_TEST_SUITE_REGISTRATION(THistoryMergerTest);
