#pragma once

#include <infra/netmon/metrics.h>
#include <infra/netmon/history_chunk.h>
#include <infra/netmon/library/scheduler.h>

#include <library/cpp/cache/cache.h>

#include <util/digest/numeric.h>
#include <util/generic/guid.h>
#include <util/random/easy.h>
#include <util/stream/direct_io.h>
#include <util/system/direct_io.h>

namespace NNetmon {
    namespace {
        const TDuration DAY_LENGTH = TDuration::Days(1);
        const TDuration HOUR_LENGTH = TDuration::Hours(1);

        const TFlags<EOpenModeFlag> OPEN_FLAGS = EOpenModeFlag::CreateNew | EOpenModeFlag::Seq | EOpenModeFlag::WrOnly | EOpenModeFlag::Direct;
        const std::size_t BUFFER_SIZE = 1 << 27;

        template <class TConfig>
        class THistoryOneShotWriter : public TNonCopyable {
        public:
            THistoryOneShotWriter(const TConfig& config, const TInstant& timestamp,
                        IOutputStream& keyStream, IOutputStream& dataStream)
                : Config_(config)
                , Timestamp_(timestamp)
                , Builder_(keyStream, dataStream, config.GetCodec())
                , Successful_(false)
            {
            }

            void Dump() {
                // builder should be filled with sorted keys
                std::tie(Successful_, Timestamp_) = Config_.FillBuilder(Builder_, Timestamp_);
                if (Successful_) {
                    Builder_.Finish();
                }
            }

            inline bool Successful() const noexcept {
                return Successful_;
            }
            inline TInstant Timestamp() const noexcept {
                return Timestamp_;
            }

        private:
            const TConfig& Config_;
            TInstant Timestamp_;
            THistoryChunkBuilder Builder_;
            bool Successful_;
        };
    }

    template <class TConfigType>
    class THistoryPlacement : public TNonCopyable {
    public:
        using TConfig = TConfigType;
        using THistoryKey = typename TConfig::THistoryKey;

        THistoryPlacement(const TConfig& config)
            : Config(config)
        {
        }

        const TConfig& GetConfig() const noexcept {
            return Config;
        }

        inline TInstant RoundToHour(const TInstant& ts) const {
            return RoundInstant(ts, HOUR_LENGTH);
        }

        inline TInstant RoundToDay(const TInstant& ts) const {
            return RoundInstant(ts, DAY_LENGTH);
        }

        TFsPath GetPathToChunk(const TInstant& now) const {
            const auto ts(Config.RoundToInterval(now));
            return Config.GetRoot()
                .Child(::ToString(RoundToDay(ts).Seconds()))
                .Child(::ToString(RoundToHour(ts).Seconds()))
                .Child(::ToString(ts.Seconds()));
        }

        TFsPath ToDataPath(const TFsPath& path) const {
            return path.Parent().Child(path.GetName() + ".data");
        }

        TFsPath ToKeyPath(const TFsPath& path) const {
            return path.Parent().Child(path.GetName() + ".key");
        }

        TFsPath GetTempDir() const {
            return Config.GetRoot().Child("tmp");
        }

        TInstant GetLastDumpTime() const {
            return *LastDumpTime.Own();
        }

        void SetLastDumpTime(const TInstant& now) {
            *LastDumpTime.Own() = now;
        }

    private:
        const TConfig& Config;
        TPlainLockedBox<TInstant> LastDumpTime;
    };

    template <class TConfigType>
    class THistoryDumper : public TScheduledTask {
    public:
        using TPlacement = THistoryPlacement<TConfigType>;

        THistoryDumper(TPlacement& placement)
            : TScheduledTask(placement.GetConfig().GetDumpInterval())
            , Placement(placement)
            , Executor("HistoryDumper", 1)
        {
        }

        TThreadPool::TFuture Run() override {
            return Executor.Add([this]() {
                TUnistatTimer timer{TUnistat::Instance(), Placement.GetConfig().DumpMetric()};
                DumpData();
                *LastIterationTime.Own() = TInstant::Now();
            });
        }

        bool IsReady() const noexcept {
            const auto now(TInstant::Now());
            const auto lastDumpTime(Placement.GetLastDumpTime());
            const auto lastIterationTime(*LastIterationTime.Own());
            if (lastDumpTime) {
                return lastDumpTime + Placement.GetConfig().GetChunkDuration() * 3 + GetInterval() > now;
            } else {
                return lastIterationTime + GetInterval() * 3 > now;
            }
        }

    private:
        void DumpData() {
            if (!Placement.GetConfig().HasAnythingToDump()) {
                return;
            }

            const auto tempDir(Placement.GetTempDir());
            tempDir.MkDirs();

            const auto tempPath(tempDir.Child(CreateGuidAsString()));
            const auto keyPath(Placement.ToKeyPath(tempPath));
            const auto dataPath(Placement.ToDataPath(tempPath));

            auto successful(false);
            auto timestamp(Placement.GetLastDumpTime());
            try {
                TDirectIOBufferedFile keyFile(keyPath, OPEN_FLAGS, BUFFER_SIZE);
                TRandomAccessFileOutput keyStream(keyFile);

                TDirectIOBufferedFile dataFile(dataPath, OPEN_FLAGS, BUFFER_SIZE);
                TRandomAccessFileOutput dataStream(dataFile);

                THistoryOneShotWriter<typename TPlacement::TConfig> writer(Placement.GetConfig(), timestamp, keyStream, dataStream);
                writer.Dump();

                successful = writer.Successful();
                timestamp = writer.Timestamp();
            } catch (...) {
                tempPath.ForceDelete();
                throw;
            }

            if (successful) {
                const auto path(Placement.GetPathToChunk(timestamp));
                path.Parent().MkDirs();

                keyPath.RenameTo(Placement.ToKeyPath(path));
                dataPath.RenameTo(Placement.ToDataPath(path));

                Placement.SetLastDumpTime(timestamp);
            }

            tempDir.ForceDelete();
        }

        TPlacement& Placement;
        TPlainLockedBox<TInstant> LastIterationTime;
        TBaseThreadExecutor Executor;
    };

    template <class TConfigType>
    class THistoryReader : public TScheduledTask {
    public:
        using TPlacement = THistoryPlacement<TConfigType>;

        THistoryReader(const TPlacement& placement)
            : TScheduledTask(TDuration::Minutes(5))
            , Placement(placement)
            , BlobCache(Placement.GetConfig().GetBlobCacheSize())
        {
        }

        TThreadPool::TFuture Run() override {
            return TThreadPool::Get()->Add([this]() {
                CleanupCache();
            });
        }

        bool Read(const THistoryTuple& key, const TInstant& requestedTs, TStringBuf& payload, TBuffer& buffer) const noexcept {
            TUnistatTimer timer{TUnistat::Instance(), Placement.GetConfig().ReadMetric()};

            const auto chunk(ReadChunk(requestedTs));
            if (chunk) {
                payload = chunk->Find(key, buffer);
                return !!payload;
            } else {
                // nothing can be found
                return false;
            }
        }

        THistoryChunkReader::TRef ReadChunk(const TInstant& requestedTs, bool precharge=false) const noexcept {
            // define timestamp to work with
            const auto& config(Placement.GetConfig());
            const auto now(TInstant::Now());
            const auto currentTs(config.RoundToInterval(now));
            const auto startTs(config.RoundToInterval(currentTs - config.GetTimeToLive()));
            const auto needleTs(config.RoundToInterval(requestedTs));
            const auto lastDumpTime(Placement.GetLastDumpTime());
            const auto endTs(lastDumpTime ? config.RoundToInterval(lastDumpTime) : currentTs);

            if (startTs <= needleTs && needleTs <= endTs) {
                // let's load proper chunk
                return LoadChunk(needleTs, precharge);
            } else {
                return nullptr;
            }
        }

    private:
        THistoryChunkReader::TRef LoadChunk(const TInstant& ts, bool precharge=false) const noexcept {
            // yes, read one chunk at a time
            auto cache(BlobCache.Own());
            auto it(cache->Find(ts));
            if (!precharge && it != cache->End()) {
                return *it;
            }

            THistoryChunkReader::TRef result;
            try {
                const auto path(Placement.GetPathToChunk(ts));
                if (precharge) {
                    result = MakeAtomicShared<THistoryChunkReader>(
                        TBlob::PrechargedFromFile(Placement.ToKeyPath(path)),
                        TBlob::PrechargedFromFile(Placement.ToDataPath(path))
                    );
                } else {
                    result = MakeAtomicShared<THistoryChunkReader>(
                        TBlob::FromFile(Placement.ToKeyPath(path)),
                        TBlob::FromFile(Placement.ToDataPath(path))
                    );
                }
            } catch (...) {
                if (errno != ENOENT) {
                    ERROR_LOG << "Can't load chunk for " << TypeName<typename TPlacement::TConfig>() << ", "
                              << ts << ": " << CurrentExceptionMessage() << Endl;
                }
            }

            // don't push precharged chunks into cache, they can be too big
            if (!precharge && result) {
                cache->Insert(ts, result);
            }
            return result;
        }

        void CleanupCache() {
            auto cache(BlobCache.Own());
            const auto deadline(TInstant::Now() - Placement.GetConfig().GetTimeToLive());
            TVector<TInstant> toDelete;
            for (auto it(cache->Begin()); it != cache->End(); ++it) {
                if (it.Key() < deadline) {
                    toDelete.emplace_back(it.Key());
                }
            }
            for (const auto& key : toDelete) {
                cache->Erase(cache->Find(key));
            }
        }

        const TPlacement& Placement;
        TPlainLockedBox<TLRUCache<TInstant, THistoryChunkReader::TRef>> BlobCache;
    };

    template <class TConfigType>
    class THistoryCleaner : public TScheduledTask {
    public:
        using TPlacement = THistoryPlacement<TConfigType>;

        THistoryCleaner(const TPlacement& placement)
            : TScheduledTask(TDuration::Minutes(5))
            , Placement(placement)
        {
        }

        TThreadPool::TFuture Run() override {
            return TThreadPool::Get()->Add([this]() {
                CleanupOldData();
            });
        }

    private:
        void CleanupOldData() {
            auto root(Placement.GetConfig().GetRoot());
            root.MkDirs();
            const auto deadline(Placement.RoundToDay(TInstant::Now()) - Placement.GetConfig().GetTimeToLive());
            FindFolders(root, deadline, [this](const TFsPath& dayRoot) {
                const auto jitter(TDuration::MicroSeconds(RandomNumber<ui64>(HOUR_LENGTH.MicroSeconds())));
                const auto deadline(Placement.RoundToHour(TInstant::Now()) - Placement.GetConfig().GetTimeToLive() - jitter);
                const auto empty = FindFolders(dayRoot, deadline, [](const TFsPath& hourRoot) {
                    hourRoot.ForceDelete();
                });
                if (empty) {
                    dayRoot.ForceDelete();
                }
            });
        }

        template <class F>
        static bool FindFolders(const TFsPath& path, const TInstant& deadline, F&& func) {
            TVector<TFsPath> children;
            path.List(children);
            SortBy(children.begin(), children.end(), [](const auto& x) {
                return TInstant::Seconds(FromStringWithDefault(x.Basename(), 0UL));
            });

            bool empty = true;
            for (const auto& child : children) {
                const auto ts(TInstant::Seconds(FromStringWithDefault(child.Basename(), 0UL)));
                if (ts && ts < deadline) {
                    empty = false;
                    func(child);
                    break;
                }
            }

            return empty;
        }

        const TPlacement& Placement;
    };

    template <class TConfigType>
    class THistoryMaintainer : public TNonCopyable {
    public:
        using TPlacement = THistoryPlacement<TConfigType>;
        using TDumper = THistoryDumper<TConfigType>;
        using TReader = THistoryReader<TConfigType>;
        using TCleaner = THistoryCleaner<TConfigType>;

        THistoryMaintainer(const TConfigType& config)
            : Placement(config)
            , Dumper(Placement)
            , Reader(Placement)
            , Cleaner(Placement)
            , DumperGuard(Dumper.Schedule())
            , ReaderGuard(Reader.Schedule())
            , CleanerGuard(Cleaner.Schedule())
        {
        }

        const TConfigType& GetConfig() const noexcept {
            return Placement.GetConfig();
        }

        bool Read(const THistoryTuple& key, const TInstant& requestedTs, TStringBuf& payload, TBuffer& buffer) const noexcept {
            return Reader.Read(key, requestedTs, payload, buffer);
        }

        template <class F>
        void Iterate(const TInstant& requestedTs, F&& func) const noexcept {
            const auto chunk(Reader.ReadChunk(requestedTs, true));
            if (chunk) {
                chunk->Iterate(std::move(func));
            }
        }

        bool IsReady() const noexcept {
            return Dumper.IsReady();
        }

        TThreadPool::TFuture WaitDumper() {
            return Dumper.SpinAndWait();
        }

        void SpinDumper() {
            Dumper.Spin();
        }

    private:
        TPlacement Placement;

        // TODO: why this is intrusive pointer?
        TDumper Dumper;
        TReader Reader;
        TCleaner Cleaner;

        TScheduledTask::TTaskGuard DumperGuard;
        TScheduledTask::TTaskGuard ReaderGuard;
        TScheduledTask::TTaskGuard CleanerGuard;
    };
}
