#include <infra/netmon/history_requester.h>
#include <infra/netmon/history_merger.h>
#include <infra/netmon/response_gatherer.h>
#include <infra/netmon/metrics.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/library/api_client_helpers.h>
#include <infra/netmon/idl/api.fbs.h>

namespace NNetmon {
    namespace {
        class THistoryExecutor: public TCustomThreadExecutor<THistoryExecutor, 1> {
        public:
            THistoryExecutor()
                : TCustomThreadExecutor("HistoryExecutor")
            {
            }
        };

        class TSwitchConfig {
        public:
            using TIndex = TSwitchPairIndex;
            using TSeries = TSwitchPairStateHistory;
            using TSeriesMerger = TSwitchHistorySeriesMerger;
            using TStateGatherer = TGatherer<NApi::TSwitchStateResponse, THistoryExecutor>;
            using TSeriesGatherer = TGatherer<NApi::TSwitchHistorySeriesResponse, THistoryExecutor>;

            TString GetStatePath() const {
                return "/slicer/v1/switch_history_state";
            }

            TString GetSeriesPath() const {
                return "/slicer/v1/switch_history_series";
            }
        };

        class TLineConfig {
        public:
            using TIndex = TLinePairIndex;
            using TSeries = TLinePairStateHistory;
            using TSeriesMerger = TLineHistorySeriesMerger;
            using TStateGatherer = TGatherer<NApi::TLineStateResponse, THistoryExecutor>;
            using TSeriesGatherer = TGatherer<NApi::TLineHistorySeriesResponse, THistoryExecutor>;

            TString GetStatePath() const {
                return "/slicer/v1/queue_history_state";
            }

            TString GetSeriesPath() const {
                return "/slicer/v1/queue_history_series";
            }
        };

        class TDatacenterConfig {
        public:
            using TIndex = TDatacenterPairIndex;
            using TSeries = TDatacenterPairStateHistory;
            using TSeriesMerger = TDatacenterHistorySeriesMerger;
            using TStateGatherer = TGatherer<NApi::TDatacenterStateResponse, THistoryExecutor>;
            using TSeriesGatherer = TGatherer<NApi::TDatacenterHistorySeriesResponse, THistoryExecutor>;

            TString GetStatePath() const {
                return "/slicer/v1/datacenter_history_state";
            }

            TString GetSeriesPath() const {
                return "/slicer/v1/datacenter_history_series";
            }
        };
    }

    class THistoryRequester::TImpl {
    public:
        TImpl(const TTopologyStorage& topologyStorage)
            : TopologyStorage(topologyStorage)
        {
        }

        NThreading::TFuture<TSwitchPairIndex::TRef> CollectSwitchHistoryState(
                const TInstant& requestedTs, const TProbeAggregatorKey& key, const TLinePairKey& pairKey) const {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NApi::CreateTSwitchHistoryStateRequest(
                builder,
                &key.ToProto(),
                &pairKey.ToProto(),
                requestedTs.MicroSeconds()
            ));
            return CollectHistoryState(SwitchConfig, builder);
        }

        NThreading::TFuture<TLinePairIndex::TRef> CollectLineHistoryState(
                const TInstant& requestedTs, const TProbeAggregatorKey& key, const TDatacenterPairKey& pairKey) const {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NApi::CreateTLineHistoryStateRequest(
                builder,
                &key.ToProto(),
                &pairKey.ToProto(),
                requestedTs.MicroSeconds()
            ));
            return CollectHistoryState(LineConfig, builder);
        }

        NThreading::TFuture<TDatacenterPairIndex::TRef> CollectDatacenterHistoryState(
                const TInstant& requestedTs, const TProbeAggregatorKey& key) const {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NApi::CreateTDatacenterHistoryStateRequest(
                builder,
                &key.ToProto(),
                requestedTs.MicroSeconds()
            ));
            return CollectHistoryState(DatacenterConfig, builder);
        }

        NThreading::TFuture<TSwitchPairStateHistory::TRef> CollectSwitchHistorySeries(
                const TInstant& since, const TInstant& until, const TProbeAggregatorKey& key, const TSwitchPairKey& pairKey) const {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NApi::CreateTSwitchHistorySeriesRequest(
                builder,
                &key.ToProto(),
                &pairKey.ToProto(),
                since.MicroSeconds(),
                until.MicroSeconds()
            ));
            return CollectHistorySeries(SwitchConfig, since, until, pairKey, builder);
        }

        NThreading::TFuture<TLinePairStateHistory::TRef> CollectLineHistorySeries(
                const TInstant& since, const TInstant& until, const TProbeAggregatorKey& key, const TLinePairKey& pairKey) const {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NApi::CreateTLineHistorySeriesRequest(
                builder,
                &key.ToProto(),
                &pairKey.ToProto(),
                since.MicroSeconds(),
                until.MicroSeconds()
            ));
            return CollectHistorySeries(LineConfig, since, until, pairKey, builder);
        }

        NThreading::TFuture<TDatacenterPairStateHistory::TRef> CollectDatacenterHistorySeries(
                const TInstant& since, const TInstant& until, const TProbeAggregatorKey& key, const TDatacenterPairKey& pairKey) const {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NApi::CreateTDatacenterHistorySeriesRequest(
                builder,
                &key.ToProto(),
                &pairKey.ToProto(),
                since.MicroSeconds(),
                until.MicroSeconds()
            ));
            return CollectHistorySeries(DatacenterConfig, since, until, pairKey, builder);
        }

    private:
        template <class TConfig>
        NThreading::TFuture<typename TConfig::TIndex::TRef> CollectHistoryState(const TConfig& config, const flatbuffers::FlatBufferBuilder& builder) const {
            using TIndex = typename TConfig::TIndex;
            using TStateGatherer = typename TConfig::TStateGatherer;
            return TStateGatherer::Collect(config.GetStatePath(), builder).Apply([this] (const typename TStateGatherer::TFuture& future_) {
                typename TIndex::TRef result;
                for (const auto& response : future_.GetValue()) {
                    for (const auto& index : *response->Indexes()) {
                        auto partialIndex(TIndex::Make(TopologyStorage, *index));
                        if (result) {
                            result->MergeFrom(*partialIndex);
                        } else {
                            result = partialIndex;
                        }
                    }
                }
                return result;
            });
        }

        template <class TConfig>
        NThreading::TFuture<typename TConfig::TSeries::TRef> CollectHistorySeries(
                const TConfig& config, const TInstant& since, const TInstant& until,
                const typename TConfig::TSeries::TKey& pairKey, const flatbuffers::FlatBufferBuilder& builder) const {
            using TState = typename TConfig::TSeries;
            using TSeriesGatherer = typename TConfig::TSeriesGatherer;
            return TSeriesGatherer::Collect(config.GetSeriesPath(), builder).Apply([since, until, pairKey] (const typename TSeriesGatherer::TFuture& future_) {
                typename TConfig::TSeriesMerger merger;
                for (const auto& response : future_.GetValue()) {
                    for (const auto& state : *response->States()) {
                        merger.Append(TState::Make(pairKey.GetTarget(), pairKey.GetSource(), *state));
                    }
                }
                typename TState::TRef result(CreateEmptyStateHistory(pairKey.GetTarget(), pairKey.GetSource()));
                merger.Merge(*result, since, until);
                return result;
            });
        }

        const TTopologyStorage& TopologyStorage;

        const TSwitchConfig SwitchConfig;
        const TLineConfig LineConfig;
        const TDatacenterConfig DatacenterConfig;
    };

    THistoryRequester::THistoryRequester(const TTopologyStorage& topologyStorage)
        : Impl(MakeHolder<TImpl>(topologyStorage))
    {
    }

    THistoryRequester::~THistoryRequester() {
    }

    NThreading::TFuture<TSwitchPairIndex::TRef> THistoryRequester::CollectSwitchHistoryState(
            const TInstant& requestedTs, const TProbeAggregatorKey& key, const TLinePairKey& pairKey) const {
        return Impl->CollectSwitchHistoryState(requestedTs, key, pairKey);
    }

    NThreading::TFuture<TLinePairIndex::TRef> THistoryRequester::CollectLineHistoryState(
            const TInstant& requestedTs, const TProbeAggregatorKey& key, const TDatacenterPairKey& pairKey) const {
        return Impl->CollectLineHistoryState(requestedTs, key, pairKey);
    }

    NThreading::TFuture<TDatacenterPairIndex::TRef> THistoryRequester::CollectDatacenterHistoryState(
            const TInstant& requestedTs, const TProbeAggregatorKey& key) const {
        return Impl->CollectDatacenterHistoryState(requestedTs, key);
    }

    NThreading::TFuture<TSwitchPairStateHistory::TRef> THistoryRequester::CollectSwitchHistorySeries(
            const TInstant& since, const TInstant& until, const TProbeAggregatorKey& key, const TSwitchPairKey& pairKey) const {
        return Impl->CollectSwitchHistorySeries(since, until, key, pairKey);
    }

    NThreading::TFuture<TLinePairStateHistory::TRef> THistoryRequester::CollectLineHistorySeries(
            const TInstant& since, const TInstant& until, const TProbeAggregatorKey& key, const TLinePairKey& pairKey) const {
        return Impl->CollectLineHistorySeries(since, until, key, pairKey);
    }

    NThreading::TFuture<TDatacenterPairStateHistory::TRef> THistoryRequester::CollectDatacenterHistorySeries(
            const TInstant& since, const TInstant& until, const TProbeAggregatorKey& key, const TDatacenterPairKey& pairKey) const {
        return Impl->CollectDatacenterHistorySeries(since, until, key, pairKey);
    }
}
