#pragma once

#include <infra/netmon/topology/topology_storage.h>
#include <infra/netmon/library/boxes.h>

namespace NNetmon {
    class THostsChangeSet: public TNonCopyable {
    public:
        using TBox = TAtomicLockedBox<THostsChangeSet>;
        using TRef = TBox::TValueRef;

        template <typename... Args>
        static inline TRef Make(Args&&... args) {
            return new THostsChangeSet(std::forward<Args>(args)...);
        }

        inline void Add(ui64 hostId) {
            if (HostsToRemove.contains(hostId)) {
                HostsToRemove.erase(hostId);
            } else if (!HostsToAdd.contains(hostId)) {
                HostsToAdd.emplace(hostId);
            }
        }

        inline void Remove(ui64 hostId) {
            if (HostsToAdd.contains(hostId)) {
                HostsToAdd.erase(hostId);
            } else if (!HostsToRemove.contains(hostId)) {
                HostsToRemove.emplace(hostId);
            }
        }

        inline const TTopologyStorage::THostIdSet& GetHostsToAdd() const {
            return HostsToAdd;
        }

        inline const TTopologyStorage::THostIdSet& GetHostsToRemove() const {
            return HostsToRemove;
        }

        inline bool Empty() const {
            return HostsToAdd.empty() && HostsToRemove.empty();
        }

    private:
        TTopologyStorage::THostIdSet HostsToAdd;
        TTopologyStorage::THostIdSet HostsToRemove;
    };

    class THostsState {
    public:
        using TBox = TAtomicLockedBox<THostsState>;
        using TRef = TBox::TValueRef;
        using TConstValueRef = TBox::TConstValueRef;

        template <typename... Args>
        static inline TRef Make(Args&&... args) {
            return new THostsState(std::forward<Args>(args)...);
        }

        inline bool Add(ui64 hostId) {
            if (Hosts.find(hostId).IsEnd()) {
                Hosts.emplace(hostId);
                return true;
            }
            return false;
        }

        inline bool Remove(ui64 hostId) {
            auto it(Hosts.find(hostId));
            if (!it.IsEnd()) {
                Hosts.erase(it);
                return true;
            }
            return false;
        }

        inline void Apply(const THostsChangeSet::TRef changeSet) {
            for (auto hostId : changeSet->GetHostsToAdd()) {
                Add(hostId);
            }
            for (auto hostId : changeSet->GetHostsToRemove()) {
                Remove(hostId);
            }
        }

        inline std::size_t Size() const {
            return Hosts.size();
        }

        inline const TTopologyStorage::THostIdSet& GetHosts() const {
            return Hosts;
        }

    private:
        TTopologyStorage::THostIdSet Hosts;
    };

    class TCountedHostsState {
    public:
        using TBox = TAtomicLockedBox<TCountedHostsState>;
        using TRef = TBox::TValueRef;
        using TConstValueRef = TBox::TConstValueRef;

        template <typename... Args>
        static inline TRef Make(Args&&... args) {
            return new TCountedHostsState(std::forward<Args>(args)...);
        }

        inline bool Add(ui64 hostId) {
            auto it(Hosts.find(hostId));
            if (!it.IsEnd()) {
                ++it->second;
                return false;
            }
            Hosts[hostId] = 1;
            return true;
        }

        inline bool Remove(ui64 hostId) {
            if ((--Hosts[hostId]) == 0) {
                Hosts.erase(hostId);
                return true;
            }
            return false;
        }

        inline void Apply(const THostsChangeSet::TRef changeSet) {
            for (auto hostId : changeSet->GetHostsToAdd()) {
                Add(hostId);
            }
            for (auto hostId : changeSet->GetHostsToRemove()) {
                Remove(hostId);
            }
        }

        inline std::size_t Size() const {
            return Hosts.size();
        }

        inline TTopologyStorage::THostIdSet GetHosts() const {
            TTopologyStorage::THostIdSet result(Size());
            for (auto it(Hosts.begin()); it != Hosts.end(); ++it) {
                result.emplace(it->first);
            }
            return result;
        }

    private:
        THashMap<ui64, std::size_t> Hosts;
    };

    template <class T>
    class TBaseHostStorage: public TNonCopyable {
    public:
        using TState = T;

        inline void Add(const TTopology::THostRef host) {
            auto hostId(host->GetReducedId());
            if (State.Own()->Add(hostId)) {
                ChangeSet.Own()->Add(hostId);
            }
        }

        inline void Remove(const TTopology::THostRef host) {
            auto hostId(host->GetReducedId());
            if (State.Own()->Remove(hostId)) {
                ChangeSet.Own()->Remove(hostId);
            }
        }

        inline TTopologyStorage::THostIdSet GetHosts() const {
            return State.Own()->GetHosts();
        }

        inline THostsChangeSet::TRef ResetChangeSet() const {
            auto changeSet(THostsChangeSet::Make());
            ChangeSet.Swap(changeSet);
            return changeSet;
        }

    private:
        typename TState::TBox State;
        mutable THostsChangeSet::TBox ChangeSet;
    };

    using THostStorage = TBaseHostStorage<THostsState>;
    using TCountedHostStorage = TBaseHostStorage<TCountedHostsState>;

    class IHostsMaintainer {
    public:
        virtual ~IHostsMaintainer() = default;

        virtual TTopologyStorage::THostSetBox::TConstValueRef GetHosts() const = 0;
    };
}
