#include <infra/netmon/host_storage.h>
#include <infra/netmon/topology/common_ut.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NNetmon;

using THostIdSet = TTopologyStorage::THostIdSet;

class THostStorageTest: public TTestBase {
    UNIT_TEST_SUITE(THostStorageTest);
    UNIT_TEST(TestHostsStorage)
    UNIT_TEST(TestCountedHostsStorage)
    UNIT_TEST_SUITE_END();

public:
    THostStorageTest()
        : TopologyStorage(TGlobalTopology::GetTopologyStorage())
    {
    }

private:
    inline void TestHostsStorage() {
        THostStorage storage;
        THostsChangeSet::TRef changeSet;

        changeSet = storage.ResetChangeSet();
        UNIT_ASSERT(storage.GetHosts().empty());
        UNIT_ASSERT(changeSet->Empty());

        const auto hostA(TopologyStorage.FindHost("man1-0015.search.yandex.net"));
        const auto hostB(TopologyStorage.FindHost("sas1-1234.search.yandex.net"));
        UNIT_ASSERT(hostA);
        UNIT_ASSERT(hostB);

        auto hostIdA(hostA->GetReducedId());
        auto hostIdB(hostB->GetReducedId());

        storage.Add(hostA);
        storage.Add(hostB);
        changeSet = storage.ResetChangeSet();
        UNIT_ASSERT_EQUAL(storage.GetHosts(), THostIdSet({hostIdA, hostIdB}));
        UNIT_ASSERT_EQUAL(changeSet->GetHostsToAdd(), THostIdSet({hostIdA, hostIdB}));
        UNIT_ASSERT(changeSet->GetHostsToRemove().empty());

        storage.Add(hostA);
        changeSet = storage.ResetChangeSet();
        UNIT_ASSERT_EQUAL(storage.GetHosts(), THostIdSet({hostIdA, hostIdB}));
        UNIT_ASSERT(changeSet->GetHostsToAdd().empty());
        UNIT_ASSERT(changeSet->GetHostsToRemove().empty());

        storage.Remove(hostA);
        changeSet = storage.ResetChangeSet();
        UNIT_ASSERT_EQUAL(storage.GetHosts(), THostIdSet({hostIdB}));
        UNIT_ASSERT(changeSet->GetHostsToAdd().empty());
        UNIT_ASSERT_EQUAL(changeSet->GetHostsToRemove(), THostIdSet({hostIdA}));

        storage.Remove(hostB);
        storage.Add(hostA);
        changeSet = storage.ResetChangeSet();
        UNIT_ASSERT_EQUAL(storage.GetHosts(), THostIdSet({hostIdA}));
        UNIT_ASSERT_EQUAL(changeSet->GetHostsToAdd(), THostIdSet({hostIdA}));
        UNIT_ASSERT_EQUAL(changeSet->GetHostsToRemove(), THostIdSet({hostIdB}));

        storage.Remove(hostA);
        changeSet = storage.ResetChangeSet();
        UNIT_ASSERT(storage.GetHosts().empty());
        UNIT_ASSERT(changeSet->GetHostsToAdd().empty());
        UNIT_ASSERT_EQUAL(changeSet->GetHostsToRemove(), THostIdSet({hostIdA}));

        changeSet = storage.ResetChangeSet();
        UNIT_ASSERT(storage.GetHosts().empty());
        UNIT_ASSERT(changeSet->Empty());
    }

    inline void TestCountedHostsStorage() {
        TCountedHostStorage storage;
        THostsChangeSet::TRef changeSet;

        changeSet = storage.ResetChangeSet();
        UNIT_ASSERT(storage.GetHosts().empty());
        UNIT_ASSERT(changeSet->Empty());

        const auto hostA(TopologyStorage.FindHost("man1-0015.search.yandex.net"));
        const auto hostB(TopologyStorage.FindHost("sas1-1234.search.yandex.net"));
        UNIT_ASSERT(hostA);
        UNIT_ASSERT(hostB);

        auto hostIdA(hostA->GetReducedId());
        auto hostIdB(hostB->GetReducedId());

        storage.Add(hostA);
        storage.Add(hostB);
        changeSet = storage.ResetChangeSet();
        UNIT_ASSERT_EQUAL(storage.GetHosts(), THostIdSet({hostIdA, hostIdB}));
        UNIT_ASSERT_EQUAL(changeSet->GetHostsToAdd(), THostIdSet({hostIdA, hostIdB}));
        UNIT_ASSERT(changeSet->GetHostsToRemove().empty());

        storage.Add(hostA);
        changeSet = storage.ResetChangeSet();
        UNIT_ASSERT_EQUAL(storage.GetHosts(), THostIdSet({hostIdA, hostIdB}));
        UNIT_ASSERT(changeSet->GetHostsToAdd().empty());
        UNIT_ASSERT(changeSet->GetHostsToRemove().empty());

        storage.Remove(hostA);
        changeSet = storage.ResetChangeSet();
        UNIT_ASSERT_EQUAL(storage.GetHosts(), THostIdSet({hostIdA, hostIdB}));
        UNIT_ASSERT(changeSet->GetHostsToAdd().empty());
        UNIT_ASSERT(changeSet->GetHostsToRemove().empty());

        storage.Remove(hostA);
        changeSet = storage.ResetChangeSet();
        UNIT_ASSERT_EQUAL(storage.GetHosts(), THostIdSet({hostIdB}));
        UNIT_ASSERT(changeSet->GetHostsToAdd().empty());
        UNIT_ASSERT_EQUAL(changeSet->GetHostsToRemove(), THostIdSet({hostIdA}));

        storage.Remove(hostB);
        storage.Add(hostA);
        changeSet = storage.ResetChangeSet();
        UNIT_ASSERT_EQUAL(storage.GetHosts(), THostIdSet({hostIdA}));
        UNIT_ASSERT_EQUAL(changeSet->GetHostsToAdd(), THostIdSet({hostIdA}));
        UNIT_ASSERT_EQUAL(changeSet->GetHostsToRemove(), THostIdSet({hostIdB}));

        storage.Remove(hostA);
        changeSet = storage.ResetChangeSet();
        UNIT_ASSERT(storage.GetHosts().empty());
        UNIT_ASSERT(changeSet->GetHostsToAdd().empty());
        UNIT_ASSERT_EQUAL(changeSet->GetHostsToRemove(), THostIdSet({hostIdA}));

        changeSet = storage.ResetChangeSet();
        UNIT_ASSERT(storage.GetHosts().empty());
        UNIT_ASSERT(changeSet->Empty());
    }

    TTopologyStorage& TopologyStorage;
};

UNIT_TEST_SUITE_REGISTRATION(THostStorageTest);
